package com.bxm.localnews.convert.impl;

import com.bxm.localnews.convert.Converter;
import com.bxm.localnews.sync.primary.dao.NewsKindMapper;
import com.bxm.localnews.sync.vo.local.News;
import com.bxm.localnews.sync.vo.local.NewsImg;
import com.bxm.localnews.sync.vo.local.NewsKind;
import com.bxm.localnews.sync.vo.local.NewsReply;
import com.bxm.localnews.sync.vo.spider.NewsComment;
import com.bxm.newidea.component.tools.RandomUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

abstract class AbstractConverter<T, K> implements Converter<T, K> {

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private NewsKindMapper newsKindMapper;

    private ObjectMapper objectMapper = new ObjectMapper();

    Integer getKindId(String kindName) {
        Optional<NewsKind> kind = this.getKinds().stream().filter(newsKind -> newsKind.getName().equals(kindName.trim())).findFirst();
        return kind.isPresent() ? kind.get().getId() : 0;
    }

    boolean hasKindId(int kindId) {
        return this.getKinds().stream().anyMatch(kind -> kind.getId() == kindId);
    }


    public NewsKind getKind(int kindId) {
        return this.getKinds().stream().filter(kind -> kindId == kind.getId()).findFirst().orElse(new NewsKind());
    }

    @Cacheable( value = "getKinds", sync = true)
    public List<NewsKind> getKinds() {
        return this.newsKindMapper.selectAllKinds();
    }

    String toJsonString(Object obj) {
        try {
            return this.objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            this.logger.error(obj + "解析出错", e);
            return "";
        }
    }

    /**
     * 生成新闻初始静态数据
     *
     * @param news
     */
    protected void generateNewsStatistic(News news) {
        news.setClicks(RandomUtils.nextInt(2000, 6000));
        news.setViews(RandomUtils.nextInt(2000, 6000));
        news.setCollects(0);
        news.setComments(0);
        news.setRecommends(100);
        news.setActiveViews(RandomUtils.nextInt(2000, 6000));
        news.setShares(0);
        news.setNewClicks(RandomUtils.nextInt(2000, 6000));
    }

    /**
     * 处理内容中的video标签，将其提取出来
     *
     * @param content
     * @param postImgList
     * @return
     */
    protected String extractVideo(String content, List<NewsImg> postImgList) {
        //提取HTTP地址
        Document document = Jsoup.parse(content);
        document.outputSettings().prettyPrint(true).indentAmount(1);
        Elements elements = document.select("video");

        if (null != elements) {
            for (Element element : elements) {
                NewsImg img = new NewsImg();
                img.setType("VIDEO");

                String src = element.attr("src");
                if (org.apache.commons.lang.StringUtils.isBlank(src)) {
                    continue;
                }
                img.setVideoUrl(src);
                String poster = element.attr("poster");
                if (!StringUtils.isBlank(poster)) {
                    img.setImgUrl(poster);
                }
                postImgList.add(img);
            }

        }
        return document.body().html().replace("\n", "");
    }

    /**
     * 将爬虫评论数据模型转换成新闻评论数据模型
     * @param commentList
     * @return
     */
    protected List<NewsReply> convertNewsCommentToNewReply(List<NewsComment> commentList) {
        List<NewsReply> resultList = new ArrayList<>();
        if (CollectionUtils.isEmpty(commentList)) {
            return resultList;
        }
        List<NewsComment> oneLevelList = commentList.stream().filter(e -> Objects.isNull(e.getParentId())).collect(Collectors.toList());
        NewsReplyConverter replyConverter = new NewsReplyConverter();
        oneLevelList.forEach(e -> replyConverter.judgeNewsReplyLevel(resultList, e));
        return resultList;
    }

}
