package com.bxm.localnews.convert.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.constant.RedisKey;
import com.bxm.localnews.convert.Converter;
import com.bxm.localnews.sync.third.dao.UserMapper;
import com.bxm.localnews.sync.vo.business.PostImg;
import com.bxm.localnews.sync.vo.business.User;
import com.bxm.localnews.sync.vo.local.ForumPost;
import com.bxm.localnews.sync.vo.local.ForumPostTopic;
import com.bxm.localnews.sync.vo.local.NewsReply;
import com.bxm.localnews.sync.vo.spider.SpiderForumPost;
import com.bxm.localnews.sync.vo.spider.SpiderForumPostComment;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.HtmlContentUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Lists;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component
public class ForumPostConverter extends BaseService implements Converter<SpiderForumPost, ForumPost> {

    private final RedisStringAdapter redisStringAdapter;

    private final UserMapper userMapper;

    public ForumPostConverter(RedisStringAdapter redisStringAdapter, UserMapper userMapper) {
        this.redisStringAdapter = redisStringAdapter;
        this.userMapper = userMapper;
    }

    /**
     * 投放地区：根据内容投放地区来定
     * 定时发布时间：内容实际发布时间
     * 展示发布时间：内容实际发布时间
     * 帖子状态：地方号状态为【审核通过】，大V号状态为【审核通过】，普通马甲号状态为【待审核】
     */
    @Override
    public ForumPost convert(SpiderForumPost data) {
        ForumPost forumPost = new ForumPost();
        forumPost.setId(getPostId());
        forumPost.setTitle(data.getTitle());
        forumPost.setContent(data.getContent());
        forumPost.setTextField(HtmlContentUtils.delHtmlTags(data.getContent()));
        forumPost.setImgList(parseImageList(data.getContent()));
        forumPost.setCreateTime(new Date());
        forumPost.setUserId(data.getUserId());
        User user = userMapper.selectUserById(forumPost.getUserId());
        if (null != user) {
            forumPost.setUserImg(user.getHeadImg());
            forumPost.setUserName(user.getNickname());
        }
        forumPost.setAreaCode(data.getAreaCode());
        forumPost.setPublishTime(data.getDeployTime());
        forumPost.setDisplayTime(data.getDeployTime());
        forumPost.setDeliveryType((byte) (StringUtils.isBlank(data.getAreaCode()) ? 0 : 1));
        //帖子状态：4 地方号状态为【审核通过】，5 大V号状态为【审核通过】，3 普通马甲号状态为【待审核】
        forumPost.setStatus((byte) (data.getUserType() == 4 || data.getUserType() == 5 ? 1 : 2));

        // 评论
        if (!CollectionUtils.isEmpty(data.getComments())) {
            Map<Long, NewsReply> replyIdMap = new HashMap<>(); //key为爬虫帖子id,value为新闻库id
            forumPost.setReplyList(data.getComments().stream()
                    .map(comment -> this.convertReply(comment, forumPost.getId(), replyIdMap))
                    .collect(Collectors.toList()));
        }
        // 话题
        if (null!=data.getTopicId()){
            ForumPostTopic forumPostTopic = new ForumPostTopic();
            forumPostTopic.setId(nextSequence());
            forumPostTopic.setPostId(forumPost.getId());
            forumPostTopic.setTopicId(data.getTopicId());
            forumPost.setForumPostTopic(forumPostTopic);
        }

        return forumPost;
    }

    /**
     * 提取正文中的图片或者视频
     *
     * @param content 正文内容
     * @return 图片、视频json
     */
    private String parseImageList(String content) {
        if (StringUtils.isBlank(content)) {
            return "[]";
        }
        Document doc = Jsoup.parse(content);
        List<PostImg> postImg = Lists.newArrayList();

        // 视频
        Elements elements = doc.select("video");
        for (Element element : elements) {
            Elements sources = element.select("source");
            if (null != sources && sources.size() == 1) {
                String videoUrl = sources.get(0).attr("src");
                if (StringUtils.isNotBlank(videoUrl)) {
                    PostImg video = new PostImg();
                    video.setVideoUrl(videoUrl);
                    video.setType("VIDEO");
                    postImg.add(video);
                }
            }
        }

        // 图片
        doc = Jsoup.parse(content);
        Elements ImgElements = doc.select("img");
        if (!CollectionUtils.isEmpty(ImgElements)) {
            for (Element element : ImgElements) {
                String imgUrl = element.attr("src");
                if (StringUtils.isNotBlank(imgUrl)) {
                    PostImg img = PostImg.buildImg(imgUrl);
                    postImg.add(img);
                }
            }
        }
        return JSONObject.toJSONString(postImg);
    }

    private NewsReply convertReply(SpiderForumPostComment spiderForumPostComment, Long forumPostId, Map<Long, NewsReply> replyIdMap) {
        NewsReply newsReply = new NewsReply();
        newsReply.setId(nextSequence());
        newsReply.setUserId(spiderForumPostComment.getUserId());
        newsReply.setHeadImg(spiderForumPostComment.getHeadImg());
        newsReply.setUserNickname(spiderForumPostComment.getName());
        newsReply.setNewsId(forumPostId);
        newsReply.setAddTime(spiderForumPostComment.getDeployTime());
        if (spiderForumPostComment.getParentId() != null && replyIdMap.get(spiderForumPostComment.getParentId()) != null) {
            // 回复评论的评论
            newsReply.setParentId(replyIdMap.get(spiderForumPostComment.getParentId()).getId());
            newsReply.setRootId(replyIdMap.get(spiderForumPostComment.getParentId()).getId());
            newsReply.setLevel((byte) 1);
            newsReply.setParentId(replyIdMap.get(spiderForumPostComment.getParentId()).getId());
            newsReply.setParentUserId(replyIdMap.get(spiderForumPostComment.getParentId()).getUserId());
            newsReply.setParentHeadImg(replyIdMap.get(spiderForumPostComment.getParentId()).getHeadImg());
            newsReply.setParentUserNickname(replyIdMap.get(spiderForumPostComment.getParentId()).getUserNickname());
        } else {
            newsReply.setParentId(0L);
            newsReply.setRootId(0L);
            newsReply.setLevel((byte) 0);
        }
        newsReply.setLikeCount(0);
        newsReply.setReplyContent(spiderForumPostComment.getContent());
        newsReply.setType((byte) 3);
        newsReply.setStatus((byte) 1);
        newsReply.setInteractiveCount(0);


        replyIdMap.put(spiderForumPostComment.getId(), newsReply);
        return newsReply;
    }

    private Long getPostId() {
        return redisStringAdapter.incrementWithDefault(RedisKey.SEQ_POST_ID, 4000000000000000L, 1);
    }
}
