package com.bxm.localnews.service;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.sync.primary.dao.*;
import com.bxm.localnews.sync.third.dao.AreaDivisionMapper;
import com.bxm.localnews.sync.vo.local.*;
import com.bxm.newidea.component.service.BaseService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

@Component
public class NewsCleanService extends BaseService {

    private static final String primarySessionTemplate = "primarySessionTemplate";

    private final LocalnewsAgingMapper localnewsAgingMapper;

    private final LocalnewsPersistentMapper localnewsPersistentMapper;

    private final NewsPersistentMapper newsPersistentMapper;

    private final NewsAgingMapper newsAgingMapper;

    private final AreaDivisionMapper areaDivisionMapper;

    private final NewsKindMapper newsKindMapper;

    private final NewsReplyMapper newsReplyMapper;

    @Autowired
    public NewsCleanService(LocalnewsAgingMapper localnewsAgingMapper, LocalnewsPersistentMapper localnewsPersistentMapper, NewsPersistentMapper newsPersistentMapper, NewsAgingMapper newsAgingMapper, AreaDivisionMapper areaDivisionMapper, NewsKindMapper newsKindMapper, NewsReplyMapper newsReplyMapper) {
        this.localnewsAgingMapper = localnewsAgingMapper;
        this.localnewsPersistentMapper = localnewsPersistentMapper;
        this.newsPersistentMapper = newsPersistentMapper;
        this.newsAgingMapper = newsAgingMapper;
        this.areaDivisionMapper = areaDivisionMapper;
        this.newsKindMapper = newsKindMapper;
        this.newsReplyMapper = newsReplyMapper;
    }

    @Transactional(rollbackFor = Exception.class)
    public void clean() {
        Date today = new Date();
        Calendar calendar = Calendar.getInstance();

        //清理10天之前的短期推荐池中的数据
        calendar.setTime(today);
        calendar.add(Calendar.HOUR, -(24 * 10));
        Date agingTime = calendar.getTime();

        //清理90天之前的长期推荐池中的数据
        calendar.setTime(today);
        calendar.add(Calendar.HOUR, -(24 * 90));
        Date persistenceTime = calendar.getTime();

        //清理地区新闻
        cleanLocationNews(agingTime, persistenceTime);

        //清理不合理的评论
        cleanInvalidNews();

        //清理全国新闻
        cleanNationalNews(agingTime, persistenceTime);
    }

    /**
     * 清理一些有评论但是不存在的新闻/帖子的数据
     * 主要原因是在新增新闻或者帖子时，是事先得到id，并可以对新增的新闻和帖子进行录入评论，但最终没有保存新闻/帖子
     */
    private void cleanInvalidNews() {
        List<NewsReply> newsReplyList = newsReplyMapper.selectWithoutPostId();
        MybatisBatchBuilder.create(NewsReplyMapper.class, newsReplyList).sessionTemplateName(primarySessionTemplate).run((mapper, newsReply) -> {
            return mapper.deleteNewsReplyByNewsIdAndType(newsReply.getNewsId(), Integer.valueOf(newsReply.getType()));
        });
        MybatisBatchBuilder.create(UserReplyMapper.class, newsReplyList).sessionTemplateName(primarySessionTemplate).run((mapper, newsReply) -> {
            return mapper.deleteUserReplyByNewsIdAndType(newsReply.getNewsId(), Integer.valueOf(newsReply.getType()), newsReply.getUserId());
        });

    }

    /**
     * 清理地区新闻
     * 短期推荐池：只保留10天
     * 长期推荐池：只保留90天
     */
    private void cleanLocationNews(Date agingTime, Date persistenceTime) {
        //清理地区表
        AreaDivision areaDivision = new AreaDivision();
        areaDivision.setLevel((byte) 1);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivision);
        for (AreaDivision a : areaDivisionList) {
            String province = a.getCode();
            List<Long> localnewsAgingIds = localnewsAgingMapper.selectByLimitTime(Long.valueOf(province), agingTime);

            List<LocalnewsAging> localNewsAgingList = new ArrayList<>();
            List<NewsRecommended> newsRecommendedAgingList = new ArrayList<>();
            List<NewsTag> newsTagAgingList = new ArrayList<>();

            fillTheObject(localnewsAgingIds, Long.valueOf(province), localNewsAgingList, newsRecommendedAgingList, newsTagAgingList);

            MybatisBatchBuilder.create(LocalnewsAgingMapper.class, localNewsAgingList).sessionTemplateName(primarySessionTemplate).run(LocalnewsAgingMapper::deleteByModel);
            MybatisBatchBuilder.create(LocalnewsAgingRecommendedMapper.class, newsRecommendedAgingList).sessionTemplateName(primarySessionTemplate).run(LocalnewsAgingRecommendedMapper::deleteByModel);
            MybatisBatchBuilder.create(LocalnewsAgingTagMapper.class, newsTagAgingList).sessionTemplateName(primarySessionTemplate).run(LocalnewsAgingTagMapper::deleteByModel);

            List<Long> localnewsIds = localnewsPersistentMapper.selectByLimitTime(Long.valueOf(province), persistenceTime);

            List<LocalnewsAging> localNewsPerList = new ArrayList<>();
            List<NewsRecommended> newsRecommendedperList = new ArrayList<>();
            List<NewsTag> newsTagPerList = new ArrayList<>();

            fillTheObject(localnewsIds, Long.valueOf(province), localNewsPerList, newsRecommendedperList, newsTagPerList);

            MybatisBatchBuilder.create(LocalnewsPersistentMapper.class, localNewsPerList).sessionTemplateName(primarySessionTemplate).run(LocalnewsPersistentMapper::deleteByModel);
            MybatisBatchBuilder.create(LocalnewsPersistentRecommendedMapper.class, newsRecommendedperList).sessionTemplateName(primarySessionTemplate).run(LocalnewsPersistentRecommendedMapper::deleteByModel);
            MybatisBatchBuilder.create(LocalnewsPersistentTagMapper.class, newsTagPerList).sessionTemplateName(primarySessionTemplate).run(LocalnewsPersistentTagMapper::deleteByModel);
        }
    }

    /**
     * 清理全国新闻推荐池表
     * 短期推荐池:每种种类只保存5000条，一开始是想着只保留10天以内，奈何新闻录入的太少了，导致一个类目只有几百几十条，没有起到推荐的作用还是会去长期推荐池导致慢日志
     * 长期推荐池：只保留90天
     *
     * @param persistenceTime 长期推荐池时效
     * @param agingTime       短期推荐池时效
     */
    private void cleanNationalNews(Date agingTime, Date persistenceTime) {
        //清理全国表
        List<NewsKind> kindIdList = newsKindMapper.selectAllKinds();
        for (NewsKind newsKind : kindIdList) {
            List<Long> newsAgingIdList = newsAgingMapper.selectByKindId(newsKind.getId(), 5000);
            MybatisBatchBuilder.create(NewsAgingMapper.class, newsAgingIdList).sessionTemplateName(primarySessionTemplate).run(NewsAgingMapper::deleteById);
            MybatisBatchBuilder.create(NewsAgingRecommendedMapper.class, newsAgingIdList).sessionTemplateName(primarySessionTemplate).run(NewsAgingRecommendedMapper::deleteByNewsId);
            MybatisBatchBuilder.create(NewsAgingTagMapper.class, newsAgingIdList).sessionTemplateName(primarySessionTemplate).run(NewsAgingTagMapper::deleteByNewsId);
        }

        List<Long> newsPerIdList = newsPersistentMapper.selectByLimitTime(persistenceTime);
        MybatisBatchBuilder.create(NewsPersistentMapper.class, newsPerIdList).sessionTemplateName(primarySessionTemplate).run(NewsPersistentMapper::deleteById);
        MybatisBatchBuilder.create(NewsPersistentRecommendedMapper.class, newsPerIdList).sessionTemplateName(primarySessionTemplate).run(NewsPersistentRecommendedMapper::deleteByNewsId);
        MybatisBatchBuilder.create(NewsPersistentTagMapper.class, newsPerIdList).sessionTemplateName(primarySessionTemplate).run(NewsPersistentTagMapper::deleteByNewsId);
    }

    /**
     * 拼装本地新闻、标签、已推荐记录
     * @param ids
     * @param province
     * @param localnewsAgingList
     * @param newsRecommendedList
     * @param newsTagList
     */
    private void fillTheObject(List<Long> ids,
                               Long province,
                               List<LocalnewsAging> localnewsAgingList,
                               List<NewsRecommended> newsRecommendedList,
                               List<NewsTag> newsTagList) {

        ids.forEach(id -> {
            LocalnewsAging localnewsAging = new LocalnewsAging();
            localnewsAging.setProvince(province);
            localnewsAging.setId(id);
            localnewsAgingList.add(localnewsAging);

            NewsRecommended newsRecommended = new NewsRecommended();
            newsRecommended.setProvince(province);
            newsRecommended.setNewsId(id);
            newsRecommendedList.add(newsRecommended);

            NewsTag newsTag = new NewsTag();
            newsTag.setProvince(province);
            newsTag.setNewsId(id);
            newsTagList.add(newsTag);
        });
    }

}
