package com.bxm.localnews.service;

import com.bxm.localnews.processer.ProcesserChain;
import com.bxm.localnews.processer.ProcesserContext;
import com.bxm.localnews.sync.primary.dao.NewsKindMapper;
import com.bxm.localnews.sync.primary.dao.NewsPoolMapper;
import com.bxm.localnews.sync.vo.local.NewsKind;
import com.bxm.localnews.sync.vo.local.NewsPool;
import com.bxm.localnews.thread.NewsPoolSyncThread;
import com.bxm.newidea.component.service.BaseService;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CountDownLatch;

/**
 * 新闻缓冲池补充
 * 新闻缓冲池作为新闻推荐的最后一道防线，保证推荐的请求一定有数据
 */
@Component
public class NewsPoolSyncService extends BaseService {
    private static CountDownLatch countDownLatch = null;

    @Resource
    private NewsPoolMapper newsPoolMapper;

    @Resource
    private ProcesserChain processerChain;

    @Resource
    private AsyncTaskExecutor taskExecutor;

    @Resource
    private NewsKindMapper newsKindMapper;

    public void sync(Integer offset) {
        //初始时间
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_YEAR, -7);
        Date lastModifyTime = calendar.getTime();
        this.execute(offset, lastModifyTime);
    }

    private void execute(Integer offset, Date limitDate) {
        List<NewsPool> list = this.newsPoolMapper.listNewsPool(Long.valueOf(offset), limitDate);
        if (list == null || list.isEmpty()) {
            return;
        }
        logger.info("缓冲池数据填充开始，频道：{},数量：{}", offset, list.size());
        for (NewsPool news : list) {
            ProcesserContext<NewsPool> context = new ProcesserContext<>();
            //从推荐池拿的数据一定是上线状态的
            news.setStatus((byte) 1);
            context.setData(news);
            this.processerChain.process(context);
        }
        logger.info("缓冲池数据填充完毕，频道：{},数量：{}", offset, list.size());
    }

    @Async
    public void syncNewsPool() {
        //起多个线程进行缓冲池的填充
        if (countDownLatch == null || countDownLatch.getCount() == 0) {
            List<NewsKind> kindList = this.newsKindMapper.selectAllKinds();
            countDownLatch = new CountDownLatch(kindList.size());

            //先清除缓冲池中的表数据
            for (NewsKind newsKind : kindList) {
                newsPoolMapper.deleteTotal(Long.valueOf(newsKind.getId()));
                this.taskExecutor.execute(new NewsPoolSyncThread().build(newsKind.getId(), countDownLatch));
            }
        }
    }

}
