package com.bxm.localnews.service;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.processer.ProcesserChain;
import com.bxm.localnews.processer.ProcesserContext;
import com.bxm.localnews.sync.vo.local.NewsRecommended;
import com.bxm.localnews.sync.vo.local.NewsStatistics;
import com.bxm.newidea.component.service.BaseService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.lang.ref.SoftReference;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 新闻静态数据定时更新
 */
@Service
public class NewsStatisticService extends BaseService {

    private static SoftReference<ConcurrentMap<String, NewsStatistics>> cache;

    @Autowired
    private ProcesserChain processerChain;

    @Async
    public void updateStatistic(List<NewsStatistics> newsStatisticsList) {
        for (NewsStatistics newsStatistics : newsStatisticsList) {
            ConcurrentMap<String, NewsStatistics> dataMap;

            if (cache == null) {
                dataMap = getCache();
            }

            dataMap = cache.get();
            if (dataMap == null) {
                dataMap = getCache();
            }

            String newsId = String.valueOf(newsStatistics.getNewsId());

            if (dataMap.containsKey(newsId)) {
                NewsStatistics statistics = dataMap.get(newsId);
                dataMap.put(newsId, this.plus(newsStatistics, statistics));
            } else {
                dataMap.put(newsId, newsStatistics);
            }

        }
    }

    /**
     * 将内存中的静态数据和当前得到的数据累加
     * @param n1 缓存中的数据map
     * @param n2 当前得到的map
     * @return
     */
    private NewsStatistics plus(NewsStatistics n1, NewsStatistics n2) {
        n1.setRecommends((n1.getRecommends() == null ? 0 : n1.getRecommends()) + (n2.getRecommends() == null ? 0 : n2.getRecommends()));
        n1.setComments((n1.getComments() == null ? 0 : n1.getComments()) + (n2.getComments() == null ? 0 : n2.getComments()));
        n1.setActiveViews((n1.getActiveViews() == null ? 0 : n1.getActiveViews()) + (n2.getActiveViews() == null ? 0 : n2.getActiveViews()));
        n1.setClicks((n1.getClicks() == null ? 0 : n1.getClicks()) + (n2.getClicks() == null ? 0 : n2.getClicks()));
        n1.setCollects((n1.getCollects() == null ? 0 : n1.getCollects()) + (n2.getCollects() == null ? 0 : n2.getCollects()));
        n1.setNewClicks((n1.getNewClicks() == null ? 0 : n1.getNewClicks()) + (n2.getNewClicks() == null ? 0 : n2.getNewClicks()));
        n1.setShares((n1.getShares() == null ? 0 : n1.getShares()) + (n2.getShares() == null ? 0 : n2.getShares()));
        n1.setViews((n1.getViews() == null ? 0 : n1.getViews()) + (n2.getViews() == null ? 0 : n2.getViews()));
        return n1;
    }

    /**
     * 得到缓存中的map
     * @return
     */
    private ConcurrentMap<String, NewsStatistics> getCache() {
        ConcurrentMap<String, NewsStatistics> dataMap = new ConcurrentHashMap<>();
        cache = new SoftReference<>(dataMap);
        return dataMap;
    }

    /**
     * 每当一段时间的数据累加之后对，根据数据对新闻进行更新，并清除缓存
     */
    public void sync() {
        ConcurrentMap<String, NewsStatistics> dataMap;

        if (cache == null) {
            return;
        }

        dataMap = cache.get();
        if (dataMap == null||dataMap.isEmpty()) {
            return;
        }

        ConcurrentMap<String, NewsStatistics> statisticMap = new ConcurrentHashMap<>(dataMap);

        dataMap.clear();

        logger.info("获取缓冲静态数据：{}",JSON.toJSONString(statisticMap));
        statisticMap.forEach((k, v) -> {
            ProcesserContext<NewsStatistics> context = new ProcesserContext<>();
            context.setData(v);
            this.processerChain.process(context);
        });

    }

    /**
     * 增加新闻的已推荐记录
     * @param newsRecommendedList
     */
    @Async
    public void saveRecommended(List<NewsRecommended> newsRecommendedList) {
        for (NewsRecommended newsRecommended : newsRecommendedList) {
            ProcesserContext<NewsRecommended> context = new ProcesserContext<>();
            context.setData(newsRecommended);
            this.processerChain.process(context);
        }
    }
}
