package com.bxm.localnews.service;

import com.bxm.localnews.processer.ProcesserChain;
import com.bxm.localnews.processer.ProcesserContext;
import com.bxm.localnews.sync.primary.dao.NewsMapper;
import com.bxm.localnews.sync.second.dao.NewsCommentMapper;
import com.bxm.localnews.sync.second.dao.SpiderNewsContentMapper;
import com.bxm.localnews.sync.second.dao.SpiderNewsMapper;
import com.bxm.localnews.sync.vo.local.News;
import com.bxm.localnews.sync.vo.local.NewsStatus;
import com.bxm.localnews.sync.vo.spider.SpiderNews;
import com.bxm.localnews.thread.NewsSyncThread;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CountDownLatch;

import static com.bxm.localnews.constant.RedisKey.SPIDER_NEWS_SYNC;

/**
 * 新闻同步
 */
@Component
public class NewsSyncService extends BaseService {

    private static CountDownLatch countDownLatch = null;

    private final SpiderNewsMapper spiderNewsMapper;

    private final SpiderNewsContentMapper spiderNewsContentMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final ProcesserChain processerChain;

    private final NewsCommentMapper newsCommentMapper;

    private final NewsMapper newsMapper;

    private final AsyncTaskExecutor taskExecutor;

    @Autowired
    public NewsSyncService(SpiderNewsMapper spiderNewsMapper,
                           SpiderNewsContentMapper spiderNewsContentMapper,
                           RedisStringAdapter redisStringAdapter,
                           ProcesserChain processerChain,
                           NewsCommentMapper newsCommentMapper,
                           NewsMapper newsMapper,
                           AsyncTaskExecutor taskExecutor) {
        this.spiderNewsMapper = spiderNewsMapper;
        this.spiderNewsContentMapper = spiderNewsContentMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.processerChain = processerChain;
        this.newsCommentMapper = newsCommentMapper;
        this.newsMapper = newsMapper;
        this.taskExecutor = taskExecutor;
    }

    public void sync(Integer offset) {
        String tableName = "news_" + offset;
        String contentTable = "news_content_" + offset;
        String commentTable = "news_comment_" + offset;

        //初始时间
        KeyGenerator syncKey = SPIDER_NEWS_SYNC.copy().appendKey(offset);
        Date lastModifyTime = this.redisStringAdapter.get(syncKey, Date.class);

        if (null == lastModifyTime) {
            lastModifyTime = DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR, -14);
        }
        this.execute(tableName, contentTable, commentTable, lastModifyTime, syncKey, 0L);
    }

    /**
     * 持续将爬虫库中的新闻加入到新闻推荐库
     *
     * @param newsTableName
     * @param contentTableName
     * @param commentTable
     * @param limitDate
     * @param syncKey
     * @param id
     */
    private void execute(String newsTableName, String contentTableName, String commentTable, Date limitDate, KeyGenerator syncKey, long id) {
        List<SpiderNews> list = this.spiderNewsMapper.listForSync(id, newsTableName, limitDate);
        if (list == null || list.isEmpty()) {
            return;
        }

        for (SpiderNews news : list) {
            news.setContent(this.spiderNewsContentMapper.findById(contentTableName, news.getId()));
            news.setCommentList(newsCommentMapper.selectNewsCommentsBySourceId(commentTable, news.getId()));
            ProcesserContext<SpiderNews> context = new ProcesserContext<>();
            context.setData(news);
            this.processerChain.process(context);
        }

        SpiderNews lastNews = list.get(list.size() - 1);
        this.redisStringAdapter.set(syncKey, lastNews.getModifyTime());

        this.execute(newsTableName, contentTableName, commentTable, limitDate, syncKey, lastNews.getId());
    }

    /**
     * 扫描5分钟未发布的新闻进行发布
     */
    public void newsPublishSync() {
        //扫描前5分钟未发布成功的新闻
        List<News> newsList = newsMapper.listNews(0, (byte) -1);
        for (News news : newsList) {
            NewsStatus newsStatus = new NewsStatus();
            newsStatus.setId(news.getId());
            newsStatus.setStatus((byte) 1);
            ProcesserContext<NewsStatus> context = new ProcesserContext<>();
            context.setData(newsStatus);
            this.processerChain.process(context);
        }
    }

    @Async
    public void syncNews() {
        if (countDownLatch == null || countDownLatch.getCount() == 0) {
            /**
             * 分表总数
             */
            int tableTotal = 10;
            countDownLatch = new CountDownLatch(tableTotal);
            for (int i = 0; i < tableTotal; i++) {
                //需要等countDownLatch被置为0时，才向下走
                this.taskExecutor.execute(new NewsSyncThread().build(i, countDownLatch));
            }
            try {
                countDownLatch.await();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }
}
