package com.bxm.localnews.service.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.constant.RedisKey;
import com.bxm.localnews.processer.ProcesserChain;
import com.bxm.localnews.processer.ProcesserContext;
import com.bxm.localnews.service.CommonTagService;
import com.bxm.localnews.service.ForumPostService;
import com.bxm.localnews.service.NewsManageService;
import com.bxm.localnews.service.UserStatisticService;
import com.bxm.localnews.sync.enums.NewsTagTypeEnum;
import com.bxm.localnews.sync.primary.dao.ForumPostMapper;
import com.bxm.localnews.sync.second.dao.SpiderForumPostCommentMapper;
import com.bxm.localnews.sync.second.dao.SpiderForumPostMapper;
import com.bxm.localnews.sync.vo.Keyword;
import com.bxm.localnews.sync.vo.business.ForumPostLike;
import com.bxm.localnews.sync.vo.business.User;
import com.bxm.localnews.sync.vo.local.ForumPost;
import com.bxm.localnews.sync.vo.local.PostTag;
import com.bxm.localnews.sync.vo.spider.SpiderForumPost;
import com.bxm.localnews.sync.vo.spider.SpiderWechatNews;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;

@Log4j2
@Service
public class ForumPostServiceImpl extends BaseService implements ForumPostService {

    private final ForumPostMapper forumPostMapper;

    private final SpiderForumPostMapper spiderForumPostMapper;

    private final SpiderForumPostCommentMapper spiderForumPostCommentMapper;

    private final NewsManageService newsManageService;

    private final CommonTagService commonTagService;

    private final UserStatisticService userStatisticService;

    private final RedisStringAdapter redisStringAdapter;

    private final ProcesserChain processerChain;

    @Autowired
    public ForumPostServiceImpl(ForumPostMapper forumPostMapper, SpiderForumPostMapper spiderForumPostMapper, SpiderForumPostCommentMapper spiderForumPostCommentMapper, NewsManageService newsManageService, CommonTagService commonTagService, UserStatisticService userStatisticService, RedisStringAdapter redisStringAdapter, ProcesserChain processerChain) {
        this.forumPostMapper = forumPostMapper;
        this.spiderForumPostMapper = spiderForumPostMapper;
        this.spiderForumPostCommentMapper = spiderForumPostCommentMapper;
        this.newsManageService = newsManageService;
        this.commonTagService = commonTagService;
        this.userStatisticService = userStatisticService;
        this.redisStringAdapter = redisStringAdapter;
        this.processerChain = processerChain;
    }

    @Override
    public void supplementPostTag() {
        List<ForumPost> postList = forumPostMapper.supplementPostTag();
        for (ForumPost forumPost : postList) {

            List<Keyword> keywords = newsManageService.tfidf(forumPost.getTitle(), forumPost.getContent());
            if (log.isDebugEnabled()) {
                StringBuffer sb = new StringBuffer();
                keywords.forEach(word -> sb.append(word).append(","));
                log.debug("tfidfResult -> title:[{}],content:[{}],result:[{}]", forumPost.getTitle(), forumPost.getContent(), sb.toString());
            }
            if (!CollectionUtils.isEmpty(keywords)) {
                for (Keyword keyword : keywords) {
                    PostTag postTag = new PostTag();
                    postTag.setId(nextId());
                    postTag.setName(keyword.getName());
                    postTag.setTagType(NewsTagTypeEnum.CHANNEL_TAG.getCode());
                    postTag.setWeight(keyword.getTfidfvalue());
                    postTag.setPostId(forumPost.getId());
                    postTag.setTagId(commonTagService.saveAndGetTag(keyword.getName()));
                    forumPostMapper.savePostTag(postTag);
                }
            }
        }
    }

    @Override
    public void scanningForumPost() {
        Map<Long, User> userMap = new HashMap<>();
        List<ForumPost> forumPostList = forumPostMapper.selectForumPostWithoutUser();
        for (ForumPost forumPost : forumPostList) {
            Long userId = forumPost.getUserId();
            User user = userStatisticService.getUserInfo(userMap, userId);
            if (null != user) {
                String nickname = user.getNickname();
                String headImg = user.getHeadImg();
                forumPostMapper.updateUserInfo(forumPost.getId(), userId, headImg, nickname);
            }
        }

        List<ForumPostLike> list = forumPostMapper.selectNotExitsUserInfo();
        List<ForumPostLike> paramList = Lists.newArrayList();
        list.forEach(e -> {
            ForumPostLike postLike = new ForumPostLike();
            postLike.setUserId(e.getUserId());
            postLike.setId(e.getId());
            User user = userStatisticService.getUserInfo(userMap, e.getUserId());
            if (Objects.nonNull(user)) {
                if (!(StringUtils.isEmpty(user.getNickname()) && StringUtils.isEmpty(user.getHeadImg()))) {
                    postLike.setUserNickname(user.getNickname());
                    postLike.setHeadImg(user.getHeadImg());
                    paramList.add(postLike);
                }
            }
        });
        if (!CollectionUtils.isEmpty(paramList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, paramList).sessionTemplateName("primarySessionTemplate").run(ForumPostMapper::updateUserInfoByUserId);
        }


    }

    private void execute(Date limitDate) {
        // 按创建时间查询出1000条帖子和帖子的评论
        List<SpiderForumPost> spiderForumPosts = spiderForumPostMapper.listByLimitDate(limitDate);
        if (CollectionUtils.isEmpty(spiderForumPosts)) {
            return;
        }
        for (SpiderForumPost spiderForumPost : spiderForumPosts) {
            spiderForumPost.setComments(spiderForumPostCommentMapper.ListByPostId(spiderForumPost.getId()));
            ProcesserContext<SpiderForumPost> context = new ProcesserContext<>();
            spiderForumPost.setStatus((byte)1);
            context.setData(spiderForumPost);
            this.processerChain.process(context);
        }
        limitDate = spiderForumPosts.get(spiderForumPosts.size()-1).getCreateTime();

        this.redisStringAdapter.set(RedisKey.SPIDER_FORUM_SYNC, limitDate);
        this.execute(limitDate);
    }

    @Override
    public void syncPost() {
        Date lastSyncTime = this.redisStringAdapter.get(RedisKey.SPIDER_FORUM_SYNC, Date.class);
        if (null == lastSyncTime) {
            Calendar calendar = Calendar.getInstance();
            //初始从5天前开始拉取
            calendar.add(Calendar.DAY_OF_YEAR, -5);
            lastSyncTime = calendar.getTime();
        }
        this.execute(lastSyncTime);
    }
}
