package com.bxm.localnews.simhash.impl;

import com.bxm.localnews.simhash.RepeatCheckService;
import com.bxm.localnews.simhash.utils.Simhash;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import static com.bxm.localnews.constant.RedisKey.SPIDER_SIMHASH_CACHE;

@Service
public class RepeatCheckServiceImpl implements RepeatCheckService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RepeatCheckServiceImpl.class);
    private final RedisSetAdapter redisSetAdapter;
    private Set<Long> cacheFingerprintSet;
    private Simhash simhash = new Simhash();

    private int repeatTotal;

    @Autowired
    public RepeatCheckServiceImpl(RedisSetAdapter redisSetAdapter) {
        this.redisSetAdapter = redisSetAdapter;
    }

    @Override
    public boolean repeatNews(Long newsId, String title, String content) {
        if (null == cacheFingerprintSet) {
            reload();
        }

        if (StringUtils.isNotBlank(title)) {
            content = title + content;
        }

        long targetFingerprint = simhash.simhash64(content);
        if (0 == targetFingerprint) {
            LOGGER.info("{}的simhash值为0,不进行重复判断", newsId);
            return true;
        }

        for (Long cacheFingerprint : cacheFingerprintSet) {
            if (simhash.hammingDistance(cacheFingerprint, targetFingerprint) < 4) {
                repeatTotal++;
                LOGGER.info("{}和{}重复.重复总数：{}", newsId, cacheFingerprint, repeatTotal);
                return false;
            }
        }

        cacheFingerprintSet.add(targetFingerprint);

        String currentDate = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date());

        KeyGenerator simhashKey = SPIDER_SIMHASH_CACHE.copy().appendKey(currentDate);

        redisSetAdapter.add(simhashKey, targetFingerprint);
        redisSetAdapter.expire(simhashKey, DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR, 15));
        return true;
    }

    @Override
    public void reload() {
        KeyGenerator[] subKeys = new KeyGenerator[14];
        Date current = new Date();

        String currentDate = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(current);
        KeyGenerator mainKey = SPIDER_SIMHASH_CACHE.copy().appendKey(currentDate);

        for (int i = 1; i <= 14; i++) {
            String keyPart = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(DateUtils.addField(current, Calendar.DAY_OF_YEAR, -i));
            subKeys[i - 1] = SPIDER_SIMHASH_CACHE.copy().appendKey(keyPart);
        }

        cacheFingerprintSet = new CopyOnWriteArraySet<>();
        cacheFingerprintSet.addAll(redisSetAdapter.union(mainKey, Long.class, subKeys));
        repeatTotal = 0;
    }
}
