package com.bxm.localnews.thirdparty.controller;

import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.thirdparty.dto.ClassificationDTO;
import com.bxm.localnews.thirdparty.dto.HomeWindowDTO;
import com.bxm.localnews.thirdparty.param.*;
import com.bxm.localnews.thirdparty.service.AdvertService;
import com.bxm.localnews.thirdparty.service.PopService;
import com.bxm.localnews.thirdparty.vo.AdvertVO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.ResponseJson;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Objects;

/**
 * Created by hsq 17:37 2018/5/23
 */
@Api(tags = "3-01 广告接口", description = "广告接口相关操作")
@RestController
@RequestMapping("api/public/advert")
@Slf4j
public class AdvertController {

    private AdvertService advertService;

    private PopService popService;

    private RedisSetAdapter redisSetAdapter;

    @Autowired
    public AdvertController(AdvertService advertService, PopService popService, RedisSetAdapter redisSetAdapter) {
        this.advertService = advertService;
        this.popService = popService;
        this.redisSetAdapter = redisSetAdapter;
    }

    @ApiOperation(value = "3-01-1 根据类型获取广告")
    @GetMapping("/type")
    public Json<List<AdvertVO>> getAdvertsByType(AdvertisementApiParam apiParam) {
        AdvertisementParam param = new AdvertisementParam();
        BeanUtils.copyProperties(apiParam, param);

        return ResultUtil.genSuccessResult(this.advertService.queryAdByType(param));
    }

    @ApiOperation(value = "3-01-2 获取首页弹窗")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true),
            @ApiImplicitParam(name = "areaName", value = "地区名", required = false),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true)
    })
    @GetMapping("/homeWindow")
    public Json<HomeWindowDTO> getHomeWindow(HomeWindowParam homeWindowParam, HttpServletRequest request) {
        //如果是3.1.2登录则让用户重新登录一次
        KeyGenerator k = RedisConfig.IS_NEED_RE_LOGIN.copy();
        if ("3.1.2".equals(homeWindowParam.getCurVer())) {
            log.info("3.1.2的用户进行判断是否已登录");
            if (Objects.equals(Boolean.FALSE, redisSetAdapter.exists(k, homeWindowParam.getUserId()))) {
                log.info("3.1.2的用户进行判断是否已登录，如果没有登录的痕迹，则让用户重新登陆");
                return ResultUtil.genFailedResult(RespCode.UNAUTHORIZED, "登录失效");
            }
        } else {
            if (Objects.equals(Boolean.TRUE, redisSetAdapter.exists(k, homeWindowParam.getUserId()))) {
                redisSetAdapter.remove(k, homeWindowParam.getUserId());
            }
        }
        homeWindowParam.setIp(WebUtils.getIpAddr(request));
        return ResultUtil.genSuccessResult(this.popService.getHomeWindow(homeWindowParam));
    }

    @ApiOperation(value = "3-01-3 VIP开卡权益页-全部分类", notes = "VIP开卡权益页-全部分类")
    @GetMapping("/classification")
    public ResponseJson<List<ClassificationDTO>> listVipClassification(AdvertParam param) {
        return ResponseJson.ok(advertService.vipPageClassificationList(param));
    }

    @ApiOperation(value = "3-01-4 根据类型和投放id集合获取广告")
    @GetMapping("/typeIds")
    public Json<List<AdvertVO>> getAdvertsByTypeIds(AdvertisementApiParam apiParam, AppParam app, DeviceParam device) {
        AdvertisementParam param = new AdvertisementParam();
        BeanUtils.copyProperties(apiParam, param);
        param.setApp(app);
        param.setDevice(device);

        return ResultUtil.genSuccessResult(this.advertService.queryAdByTypeIds(param));
    }
}
