package com.bxm.localnews.thirdparty.advert.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.OKHttpService;
import com.bxm.localnews.thirdparty.advert.AdvertCallbackService;
import com.bxm.localnews.thirdparty.config.ThirdPartyProperties;
import com.bxm.localnews.thirdparty.param.AdvertCallbackCommonParam;
import com.bxm.localnews.thirdparty.param.AiqiyiAdvertCallbackParam;
import com.bxm.localnews.thirdparty.param.BaiduAdvertCallbackParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;

import static com.bxm.localnews.common.constant.RedisConfig.AD_CALLBACK;


@Service
@Slf4j
public class AdvertCallbackServiceImpl implements AdvertCallbackService {

    private final OKHttpService okHttpService;

    private final RedisStringAdapter redisStringAdapter;

    private final ThirdPartyProperties thirdPartyProperties;

    private enum AdType {
        /**
         * 百度广告平台
         */
        BAIDU,

        /**
         * 爱奇艺广告平台
         */
        AIQIYI
    }

    /**
     * 等待用户激活的过期时间
     */
    private final static long EXPIRE_TIME = 5 * 60 * 60;

    @Autowired
    public AdvertCallbackServiceImpl(OKHttpService okHttpService,
                                     RedisStringAdapter redisStringAdapter,
                                     ThirdPartyProperties thirdPartyProperties) {
        this.okHttpService = okHttpService;
        this.redisStringAdapter = redisStringAdapter;
        this.thirdPartyProperties = thirdPartyProperties;
    }

    @Override
    public void baiduCallback(BaiduAdvertCallbackParam param) {
        log.debug("百度广告请求点击：{}", param);

        String androidId = param.getAndroid_id();
        redisStringAdapter.set(AD_CALLBACK.copy().appendKey(androidId), convert(param), EXPIRE_TIME);
    }

    private AdvertCallbackCommonParam convert(BaiduAdvertCallbackParam param) {
        return AdvertCallbackCommonParam.builder()
                .callback(param.getCallback_url())
                .imei(param.getImei_md5())
                .timestamp(param.getTs())
                .type(AdType.BAIDU.name())
                .build();
    }

    @Override
    public void aiqiyiCallback(AiqiyiAdvertCallbackParam param) {
        log.debug("爱奇艺广告回调参数：{}", param);

        String androidId = param.getAndroidid();
        redisStringAdapter.set(AD_CALLBACK.copy().appendKey(androidId), convert(param), EXPIRE_TIME);
    }

    private AdvertCallbackCommonParam convert(AiqiyiAdvertCallbackParam param) {
        return AdvertCallbackCommonParam.builder()
                .callback(param.getCallback_url())
                .imei(param.getImei())
                .timestamp(NumberUtils.parseToLong(param.getTimestamp()))
                .type(AdType.AIQIYI.name())
                .build();
    }

    @Override
    public boolean triggerCallback(String androidId) {
        //从缓存中查找，是否存在
        KeyGenerator key = AD_CALLBACK.copy().appendKey(androidId);
        AdvertCallbackCommonParam param = redisStringAdapter.get(key, AdvertCallbackCommonParam.class);

        //不存在对应的广告回调，直接返回
        if (param == null) {
            //androidId进行md5加密
            String md5AndrondId = MD5Util.md5(androidId).toLowerCase();
            key = AD_CALLBACK.copy().appendKey(md5AndrondId);
            param = redisStringAdapter.get(key, AdvertCallbackCommonParam.class);

            log.debug("尝试使用MD5获取：{},param:{}", md5AndrondId, param);

            if (param == null) {
                log.debug("请求的安卓ID不存在：{}", androidId);
                return false;
            }
        }

        //如果广告客户端传递了请求时间，要求请求时间不可以在注册时间之后
        if (param.getTimestamp() > 0) {
            Date regDate = new Date();
            Date requestDate = new Date(param.getTimestamp());

            if (log.isDebugEnabled()) {
                log.debug("request timestamp:{},date:{}", param.getTimestamp(), DateUtils.formatDateTime(requestDate));
            }

            if (DateUtils.after(requestDate, regDate)) {
                log.error("{} 设备注册时，广告回调信息已过期。广告请求时间：{}", androidId, requestDate);
                redisStringAdapter.remove(key);
                return false;
            }
        }


        String callbackUrl;
        if (AdType.BAIDU.name().equals(param.getType())) {
            callbackUrl = buildBaiduCallbackUrl(param.getCallback());
            String response = okHttpService.get(callbackUrl, Maps.newHashMap());

            log.debug("response:{}", response);

            JSONObject json = JSON.parseObject(response);
            if (json.getIntValue("error_code") != 0) {
                log.error("baidu advert callback error,response:{},request:{}", response, param);
                return false;
            }
        } else if (AdType.AIQIYI.name().equals(param.getType())) {
            callbackUrl = buildAiqiyiCallback(param.getCallback());
            okHttpService.get(callbackUrl, Maps.newHashMap());
        }

        redisStringAdapter.remove(key);

        log.info("用户激活成功,请求参数：{}", param);
        return true;
    }

    /**
     * 构建百度回调地址
     * 1.替换占位符
     * 2.添加advert key
     * 3.构建签名，进行MD5加密
     * @param callback 请求携带的回调地址
     * @return 组装完成的回调地址
     */
    private String buildBaiduCallbackUrl(String callback) {
        String callBack = StringUtils.replace(callback, "{{ATYPE}}", "activate");
        callBack = StringUtils.replace(callBack, "{{AVALUE}}", "0");

        callBack += "&akey=" + thirdPartyProperties.getBaiduAndroidKey();

        String sign = MD5Util.encode(callBack);

        callBack += "&sign=" + sign;

        return callBack;
    }

    /**
     * 构建爱奇艺回调地址
     * 1.增加激活类型
     * 2.进行MD5计算签名
     * 3.请求路径追加签名
     * @param url 原始回调地址
     * @return 处理后的回调地址
     */
    private String buildAiqiyiCallback(String url) {
        //0：表示激活，1：注册，2：付费
        url += "&event_type=0";

        //计算签名值
        String signature = MD5Util.encode(url + thirdPartyProperties.getAiqiyiAndroidKey());

        url += "&signature=" + signature;

        log.debug("爱奇艺回调地址：{}", url);

        return url;
    }

}
