package com.bxm.localnews.thirdparty.config;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.thirdparty.dto.ActiveWechatMP;
import com.bxm.localnews.thirdparty.dto.WechatMP;
import com.bxm.localnews.thirdparty.dto.WechatMPDomainName;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpConfigStorage;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 *
 * @author Gonzo
 * @date 2019-12-18 15:59
 */
@Configuration
@Slf4j
public class WechatMpConfiguration {

    private WechatMPConfig wechatMpConfig;

    private CustomApacheHttpClientBuilder customApacheHttpClientBuilder;

    private RedisStringAdapter redisStringAdapter;

    private WechatAccountConfig wechatAccountConfig;

    private static final String ACTIVE_APP_ID = "thirdparty.config.wechat.mp.active.active-app-id";
    private static final String ACTIVE_APP_DOMAIN_ID = "thirdparty.config.wechat.mp.active.active-domain-name-id";

    private WechatMpServiceInvocationHandler wechatMpServiceInvocationHandler = null;

    public WechatMpConfiguration(WechatMPConfig wechatMpConfig, CustomApacheHttpClientBuilder customApacheHttpClientBuilder,
                                 RedisStringAdapter redisStringAdapter, WechatAccountConfig wechatAccountConfig) {
        this.wechatMpConfig = wechatMpConfig;
        this.customApacheHttpClientBuilder = customApacheHttpClientBuilder;
        this.redisStringAdapter = redisStringAdapter;
        this.wechatAccountConfig = wechatAccountConfig;
    }

    @Bean
    public WxMpService wxMpService() {
        // 创建对象
        WxMpService wxMpService = createWxMpServiceWithMultiApp();

        // 生成代理对象返回，方便动态替换公众号
        wechatMpServiceInvocationHandler = new WechatMpServiceInvocationHandler(wxMpService, wechatMpConfig);
        return (WxMpService) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class<?>[]{WxMpService.class},
                wechatMpServiceInvocationHandler);
    }

    /**
     * 一个对象，多个微信公众号配置
     * 使用哪个公众号在 {@link WechatMpServiceInvocationHandler#invoke} 中指定
     * @return
     */
    private WxMpService createWxMpServiceWithMultiApp() {
        WxMpService wxMpService = new WxMpServiceImpl();

        // 添加多套公众号信息
        Map<String, WxMpConfigStorage> configStorages = JSON.parseArray(wechatMpConfig.getWechatMps(), WechatMP.class)
                .stream()
                .collect(Collectors.toMap(WechatMP::getAppId, this::buildWxMpConfigStorage));

        log.info("初始化的微信配置信息: {}", JSON.toJSONString(configStorages));
        wxMpService.setMultiConfigStorages(configStorages);
        return wxMpService;
    }

    /**
     * 根据公众号信息构建缓存对象
     * @param wechatMP
     * @return
     */
    private WxMpConfigStorage buildWxMpConfigStorage(WechatMP wechatMP) {

        WxMpInMemoryConfigStorage wxConfigProvider = new WxMpInRedisConfigStorage(redisStringAdapter, customApacheHttpClientBuilder);
        wxConfigProvider.setAppId(wechatMP.getAppId());
        wxConfigProvider.setSecret(wechatMP.getAppSecret());
        wxConfigProvider.setAesKey(wechatMP.getAesKey());
        wxConfigProvider.setToken(wechatMP.getToken());
        return wxConfigProvider;
    }


    // --下面是一开始的方案，直接生成多个WxMpService 但是组件提供了更好的，同时配置多个公众号信息，所以暂时先弃用下面的方式----------
    /**
     * 获取可用的微信公众号配置
     * @return
     */
    @Deprecated
    private ActiveWechatMP getActiveWechatMp() {
        if (Objects.isNull(wechatMpConfig.getActive())) {
            throw new RuntimeException("没有配置启用的公众号信息！！！");
        }

        // 获取启用的公众号信息
        Optional<WechatMP> wechatMpOptional = JSON.parseArray(wechatMpConfig.getWechatMps(), WechatMP.class).stream()
                .filter(p -> Objects.equals(p.getAppId(), wechatMpConfig.getActive().getActiveAppId()))
                .findFirst();

        if (!wechatMpOptional.isPresent()) {
            log.error("appId: {} 没有对应的公众号配置", wechatMpConfig.getActive().getActiveAppId());
            throw new RuntimeException("公众号配置的appId没有可用的信息！！！");
        }

        WechatMP wechatMp = wechatMpOptional.get();
        // 获取启用的域名信息
        Optional<WechatMPDomainName> wechatMpDomainNameOptional = wechatMp.getDomainNames().stream()
                .filter(p -> Objects.equals(p.getId(), wechatMpConfig.getActive().getActiveDomainNameId()))
                .findFirst();

        if (!wechatMpDomainNameOptional.isPresent()) {
            log.error("域名id: {} 没有对应的域名信息", wechatMpConfig.getActive().getActiveDomainNameId());
            throw new RuntimeException("公众号配置的域名id没有可用的信息！！！");
        }

        return ActiveWechatMP.builder()
                .appId(wechatMp.getAppId())
                .appSecret(wechatMp.getAppSecret())
                .domainName(wechatMpDomainNameOptional.get().getDomainName())
                .build();
    }

    @Deprecated
    private WxMpService createWxMpService() {
        // 创建WxMpService实例并设置appid和sectret
        WxMpService wxMpService = new WxMpServiceImpl();
        // 这里的设置方式是跟着这个sdk的文档写的
        wxMpService.setWxMpConfigStorage(wxConfigProvider());
        return wxMpService;
    }

    @Deprecated
    private WxMpConfigStorage wxConfigProvider() {
        WxMpInMemoryConfigStorage wxConfigProvider = new WxMpInRedisConfigStorage(redisStringAdapter, customApacheHttpClientBuilder);
        ActiveWechatMP activeWechatMp = getActiveWechatMp();
        wxConfigProvider.setAppId(activeWechatMp.getAppId());
        wxConfigProvider.setSecret(activeWechatMp.getAppSecret());
        return wxConfigProvider;
    }

    /**
     *  监听apollo配置变更
     * @param configChangeEvent
     */
    @Deprecated
    // @ApolloConfigChangeListener(value = {"application", "LN.common", "activity", "payment", "clientConfig", "thirdparty"})
    public void changeHandler(ConfigChangeEvent configChangeEvent) {

        try {
            boolean changed = false;
            if (configChangeEvent.changedKeys().contains(ACTIVE_APP_ID)) {
                wechatMpConfig.getActive().setActiveAppId(configChangeEvent.getChange(ACTIVE_APP_ID).getNewValue());
                changed = true;
            }

            if (configChangeEvent.changedKeys().contains(ACTIVE_APP_DOMAIN_ID)) {
                wechatMpConfig.getActive().setActiveDomainNameId(Long.parseLong(configChangeEvent.getChange(ACTIVE_APP_DOMAIN_ID).getOldValue()));
                changed = true;
            }

            if (changed) {
                log.info("更换了微信公众号配置，重新生成并替换公众号组件bean");
                wechatMpServiceInvocationHandler.setWxMpService(createWxMpService());
            }
        } catch (Exception e) {
            log.error("动态替换微信公众号组件Bean失败：", e);
        }
    }
}