package com.bxm.localnews.thirdparty.filter.impl;

import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.thirdparty.advert.annotation.AdverFilter;
import com.bxm.localnews.thirdparty.constant.AdvertTypeEnum;
import com.bxm.localnews.thirdparty.filter.AbstractAdvertFilterHandler;
import com.bxm.localnews.thirdparty.filter.AdvertFilterService;
import com.bxm.localnews.thirdparty.param.AdvertisementParam;
import com.bxm.localnews.thirdparty.vo.AdvertVO;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Author: pf.w
 * @Date: 2020/5/12 20:02
 */
@Service
public class AdvertFilterServiceImpl implements AdvertFilterService, ApplicationContextAware {

    private Map<AdvertTypeEnum, List<AbstractAdvertFilterHandler>> advertisements = Maps.newHashMap();

    /**
     * 默认filter
     */
    private List<AbstractAdvertFilterHandler> defaultFilters = Lists.newArrayList();

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Override
    public void filter(List<AdvertVO> advertVOS, AdvertisementParam param) {
        if (CollectionUtils.isEmpty(advertVOS)) {
            return;
        }

        if (null != param.getUserId()) {
            UserInfoDTO userInfo = userIntegrationService.getUserFromRedisDB(param.getUserId());
            param.setUserInfo(userInfo);
        }

        if (null != defaultFilters) {
            for (AbstractAdvertFilterHandler defaultFilter : defaultFilters) {
                if (!defaultFilter.filter(advertVOS, param)) {
                    return;
                }
            }
        }
        List<AbstractAdvertFilterHandler> filters = advertisements.get(AdvertTypeEnum.getByPosition(param.getType()));
        if (null != filters) {
            for (AbstractAdvertFilterHandler filter : filters) {
                if (!filter.filter(advertVOS, param)) {
                    return;
                }
            }
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Collection<AbstractAdvertFilterHandler> filterMap = applicationContext.getBeansOfType
                (AbstractAdvertFilterHandler.class).values();
        filterMap.forEach(p -> {
            AdverFilter adverFilter = AnnotationUtils.getAnnotation(AopUtils.getTargetClass(p), AdverFilter.class);
            if (null != adverFilter) {
                addFilter(adverFilter.value(), p);
            }
        });
    }

    /**
     * 根据position 进行初始化map
     *
     * @param positions
     * @param filter
     */
    private void addFilter(AdvertTypeEnum[] positions, AbstractAdvertFilterHandler filter) {

        if (null == positions || positions.length == 0) {
            defaultFilters.add(filter);
        }

        for (AdvertTypeEnum position : positions) {
            List<AbstractAdvertFilterHandler> filters = advertisements.get(position);

            if (CollectionUtils.isEmpty(filters)) {
                filters = Lists.newArrayList();
                advertisements.put(position, filters);
            }

            filters.add(filter);
        }

    }
}
