package com.bxm.localnews.thirdparty.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.activity.common.config.AdvertProperties;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.AreaWhiteBlackService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.AreaWhiteBlackKeyEnum;
import com.bxm.localnews.common.dto.AppVersionDTO;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.thirdparty.constant.AdvertTypeEnum;
import com.bxm.localnews.thirdparty.domain.AdvertAreaMapper;
import com.bxm.localnews.thirdparty.domain.AdvertMapper;
import com.bxm.localnews.thirdparty.domain.AdvertPositionMapper;
import com.bxm.localnews.thirdparty.dto.AdvertDTO;
import com.bxm.localnews.thirdparty.dto.ClassificationDTO;
import com.bxm.localnews.thirdparty.dto.HomeWindowDTO;
import com.bxm.localnews.thirdparty.filter.AdvertFilterService;
import com.bxm.localnews.thirdparty.param.AdvertParam;
import com.bxm.localnews.thirdparty.param.AdvertisementParam;
import com.bxm.localnews.thirdparty.param.HomeWindowParam;
import com.bxm.localnews.thirdparty.service.AdvertService;
import com.bxm.localnews.thirdparty.vo.Advert;
import com.bxm.localnews.thirdparty.vo.AdvertArea;
import com.bxm.localnews.thirdparty.vo.AdvertRecord;
import com.bxm.localnews.thirdparty.vo.AdvertVO;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Created by hsq 10:33 2018/2/8
 */
@Service
public class AdvertServiceImpl implements AdvertService {
    private static final Logger LOG = LoggerFactory.getLogger(AdvertServiceImpl.class);

    private AdvertMapper advertMapper;

    private AdvertAreaMapper advertAreaMapper;

    private RedisStringAdapter redisStringAdapter;

    private LocationFacadeService locationFacadeService;

    private AdvertPositionMapper advertPositionMapper;

    private UserIntegrationService userIntegrationService;

    private AppVersionSupplyService appVersionSupplyService;

    @Autowired
    private AdvertFilterService advertFilterService;

    @Autowired
    private AdvertProperties adverProperties;

    /**
     * 地区黑白名单服务
     */
    private AreaWhiteBlackService areaWhiteBlackService;

    public AdvertServiceImpl(AdvertMapper advertMapper,
                             AdvertAreaMapper advertAreaMapper,
                             RedisStringAdapter redisStringAdapter,
                             LocationFacadeService locationFacadeService,
                             AdvertPositionMapper advertPositionMapper,
                             UserIntegrationService userIntegrationService,
                             AppVersionSupplyService appVersionSupplyService,
                             AreaWhiteBlackService areaWhiteBlackService) {
        this.advertMapper = advertMapper;
        this.advertAreaMapper = advertAreaMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.locationFacadeService = locationFacadeService;
        this.advertPositionMapper = advertPositionMapper;
        this.userIntegrationService = userIntegrationService;
        this.appVersionSupplyService = appVersionSupplyService;
        this.areaWhiteBlackService = areaWhiteBlackService;
    }

    private static String DEFAULT_AREACODE = "0";

    @Override
    public List<AdvertVO> queryAdByType(Byte type, String areaCode, Long userId, BasicParam basicParam) {
        AdvertisementParam param = new AdvertisementParam();
        if (null != basicParam) {
            param.merge(basicParam);
        }
        param.setUserId(userId);
        param.setType(type);
        param.setAreaCode(areaCode);
        return queryAdByType(param);
    }

    @Override
    public List<AdvertVO> queryAdByType(AdvertisementParam param) {

        List<AdvertDTO> advertList = loadCacheInfo(param);
        // 转换
        List<AdvertVO> advertVOS = advertList.stream().map(this::convert).collect(Collectors.toList());

        advertFilterService.filter(advertVOS, param);

        //记录广告查看信息
        if (param.getUserId() != null && CollectionUtils.isNotEmpty(advertList)) {
            recordAdvert(param.getUserId(), advertList);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("根据地区{}获取得到的广告列表信息为:{}", isBlank(param.getAreaCode()) ? "0" : param.getAreaCode(), JSON
                    .toJSONString(advertVOS));
        }

        return advertVOS;
    }

    private List<AdvertDTO> loadCacheInfo(AdvertisementParam param) {
        Byte type = param.getType();
        String areaCode = param.getAreaCode();
        Long userId = param.getUserId();
        // 某些栏目对应的具体业务id
        Long positionBizId = param.getPositionBizId();

        if (LOG.isDebugEnabled()) {
            LOG.debug("请求广告接口,参数,type:[{}],positionBizId:[{}]areaCode:[{}],userId:[{}],basicParam:[{}]", type,
                    positionBizId, areaCode, userId, param);
        }

        //如果地区编码为空，则设置默认areaCode为0
        if (isBlank(areaCode)) {
            areaCode = DEFAULT_AREACODE;
        }
        // 先从缓存中取 baseKey + type
        String key = org.apache.commons.lang3.StringUtils.join(AreaWhiteBlackKeyEnum.ADVERT.getKey(), type);
        List<AdvertDTO> advertList = areaWhiteBlackService.getCacheByAreaCode(
                key, areaCode, new TypeReference<List<AdvertDTO>>() {
                });
        if (CollectionUtils.isEmpty(advertList)) {
            // 根据类型获取全部广告信息 这里，不适用bizId查询 因为缓存不好控制，查询出来之后再做剔除
            advertList = advertPositionMapper.getAdvertPositionListByType((int) type, null, 1);

            //有areaCode的情况下进行黑白名单过滤
            if (!DEFAULT_AREACODE.equals(areaCode)) {
                //areaType为1的则为区域开放的白名单列表
                List<Long> whiteAreaList = getAdvertArea(areaCode, 1);
                //areaType为2的则为全局开放的限制区域--即黑名单
                List<Long> blackAreaList = getAdvertArea(areaCode, 2);

                //黑白名单过滤
                advertList = areaWhiteBlackService.filterCacheByAreaCode(advertList, whiteAreaList, blackAreaList);
            }
            areaWhiteBlackService.setCacheByAreaCode(key, areaCode, advertList);
        }

        advertList = advertList.stream()
                .filter(this::removeByPutTime)
                .filter(p -> {
                    // 如果携带了bizId 这里需要过滤出只包含bizId的广告
                    if (Objects.nonNull(param.getPositionBizId())) {
                        return Objects.equals(p.getPositionBizId(), param.getPositionBizId());
                    } else {
                        // 如果当前bizId未传，广告对象中bizId有值，也需要过滤掉
                        return p.getPositionBizId() == null || Objects.equals(p.getPositionBizId(), 0L);
                    }
                })
                .collect(Collectors.toList());
        return advertList;
    }

    @Override
    public List<AdvertVO> queryAdByTypeIds(AdvertisementParam param) {
        // android如果栏目的id为空也会请求接口，所以这里处理下 为空就不返回了
        if (isBlank(param.getIds())) {
            return Collections.emptyList();
        }

        List<AdvertDTO> advertList = loadCacheInfoByIds(param);
        // 转换
        List<AdvertVO> advertVOS = advertList.stream().map(this::convert).collect(Collectors.toList());

        advertFilterService.filter(advertVOS, param);

        //记录广告查看信息
        //v3.4.0 暂时不需要这个逻辑
        if (param.getUserId() != null && CollectionUtils.isNotEmpty(advertList)) {
            //recordAdvert(param.getUserId(), advertList);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("根据地区{}获取得到的工具列表信息为:{}",
                    isBlank(param.getAreaCode()) ? "0" : param.getAreaCode(),
                    JSON.toJSONString(advertVOS));
        }

        return advertVOS;
    }


    private List<AdvertDTO> loadCacheInfoByIds(AdvertisementParam param) {
        Byte type = param.getType();
        String areaCode = param.getAreaCode();
        Long userId = param.getUserId();
        List<Long> ids = null;
        String key = AreaWhiteBlackKeyEnum.ADVERT_IDS.getKey() + type;
        if (StringUtils.isNotEmpty(param.getIds())) {
            ids = Arrays.asList(param.getIds().split(",")).stream().map(id -> Long.parseLong(id)).collect(Collectors.toList());
            key += MD5Util.encode(param.getIds());
        }
        LOG.info("请求广告接口,参数,type:[{}],areaCode:[{}],userId:[{}],basicParam:[{}],ids[{}]", type, areaCode, userId, param, ids);

        //如果地区编码为空，则设置默认areaCode为0
        if (isBlank(areaCode)) {
            areaCode = DEFAULT_AREACODE;
        }

        //先从缓存中取
        List<AdvertDTO> advertList = areaWhiteBlackService.getCacheByAreaCode(
                key, areaCode, new TypeReference<List<AdvertDTO>>() {
                });
        if (CollectionUtils.isEmpty(advertList)) {
            //根据类型获取全部广告信息
            advertList = advertPositionMapper.getAdvertPositionListByIds((int) type, 1, ids);

            //有areaCode的情况下进行黑白名单过滤
            if (!DEFAULT_AREACODE.equals(areaCode)) {
                //areaType为1的则为区域开放的白名单列表
                List<Long> whiteAreaList = getAdvertArea(areaCode, 1);
                //areaType为2的则为全局开放的限制区域--即黑名单
                List<Long> blackAreaList = getAdvertArea(areaCode, 2);

                //黑白名单过滤
                advertList = areaWhiteBlackService.filterCacheByAreaCode(advertList, whiteAreaList, blackAreaList);
            }
            areaWhiteBlackService.setCacheByAreaCode(AreaWhiteBlackKeyEnum.ADVERT_IDS.getKey() + type, areaCode, advertList);
        }
        advertList = advertList.stream().filter(this::removeByPutTime).collect(Collectors.toList());
        return advertList;
    }

    private AdvertVO convert(AdvertDTO advertDTO) {
        AdvertVO advertVO = new AdvertVO();
        BeanUtils.copyProperties(advertDTO, advertVO);

        List<Integer> preconditions = Lists.newArrayList();
        if (StringUtils.isNotBlank(advertDTO.getPreconditions())) {
            preconditions = Arrays.stream(advertDTO.getPreconditions().split(","))
                    .map(Integer::parseInt)
                    .collect(Collectors.toList());
        }
        advertVO.setPreconditions(preconditions);
        JSONObject extData = null;
        try {
            extData = JSONObject.parseObject(advertDTO.getExtraData());
        } catch (Exception e) {
            LOG.error("广告类型扩展字段: {},转换错误：", advertDTO.getExtraData(), e);
        }
        advertVO.setExtData(extData);
        return advertVO;
    }

    /**
     * 如果不传基础参数则不做验证
     * 如果版本小于等于2.3.1且平台等于安卓且是（首页广告或者banner轮播广告）
     *
     * @param basicParam
     * @param type
     * @return
     */
    private boolean checkIsFlashbackVersion(BasicParam basicParam, int type) {
        return null != basicParam
                && appVersionSupplyService.isHighVersion(basicParam.getCurVer(), "3.0.0") != 1
                && 1 == basicParam.getPlatform()
                && (AdvertTypeEnum.INDEX_POP_WINDOW.getType() == type || AdvertTypeEnum.TURN_AROUND_ADVERT.getType() == type);
    }

    /**
     * 为修复安卓2.3.1升级闪退的一个bug，
     * 只有设置为静默升级才不会闪退，
     * 在首页弹窗和banner轮播中放置固定的版本升级素材，
     * 目的在于让用户主动去升级
     *
     * @param type
     * @param basicParam
     * @return
     */
    private List<AdvertDTO> getTempAdvert(Byte type, BasicParam basicParam) {
        AppVersionDTO appVersionDTO = appVersionSupplyService.getAppVersion(basicParam, null);
        if (null != appVersionDTO && StringUtils.isNoneEmpty(appVersionDTO.getDownloadLink())) {
            AdvertDTO advert = new AdvertDTO();
            if (AdvertTypeEnum.INDEX_POP_WINDOW.getType() == type) {
                advert.setImgUrl("https://m.wstong.com/localnews_prod/png/20190726/1V966V81DXI7O1J60XAP9C34PEOT212NHXJYJF4.png?x-oss-process=style/scompress");
            } else if (AdvertTypeEnum.TURN_AROUND_ADVERT.getType() == type) {
                advert.setImgUrl("https://m.wstong.com/localnews_prod/jpg/20190726/1V966V81DXI7O1J60XAP9C34PEOT2032IXJYJG4.jpg?x-oss-process=style/scompress");
            }
            advert.setAddress(appVersionDTO.getDownloadLink());
            advert.setGlobalFlag((byte) 1);
            advert.setTitle("升级");
            advert.setId((long) 10086);
            advert.setStartTime(new Date(System.currentTimeMillis() - 1000));
            advert.setEndTime(new Date(System.currentTimeMillis() + 1000));
            advert.setShowType((byte) 0);

            return Lists.newArrayList(advert);
        }
        return new ArrayList<>();
    }

    /**
     * 判断时间是否处于投放开始时间和结束时间之间
     *
     * @param advertDTO
     * @return
     */
    private Boolean removeByPutTime(AdvertDTO advertDTO) {
        Date now = new Date();
        return !(advertDTO.getStartTime() != null && DateUtils.after(advertDTO.getStartTime(), now)
                || (advertDTO.getEndTime() != null && DateUtils.before(advertDTO.getEndTime(), now)));
    }

    /**
     * 记录广告查看时间
     *
     * @param userId
     * @param advertDTOList
     */
    private void recordAdvert(Long userId, List<AdvertDTO> advertDTOList) {
        List<AdvertRecord> advertRecords = Lists.newArrayList();
        Date now = new Date();
        advertDTOList.forEach(advertDTO -> {
            AdvertRecord advertRecord = new AdvertRecord();
            advertRecord.setAdvertId(advertDTO.getId());
            advertRecord.setUserId(userId);
            advertRecord.setViewTime(now);
            advertRecords.add(advertRecord);
        });
        if (CollectionUtils.isEmpty(advertRecords)) {
            return;
        }

        advertMapper.recordAdvert(advertRecords);
    }

    @Override
    public Advert selectByPrimaryKey(Long id) {
        return advertMapper.selectByPrimaryKey(id);
    }

    @Override
    public List<AdvertDTO> getListAds(int size) {
        return advertMapper.getListAds(size);
    }

    /**
     * 通过地区编码获取广告id列表
     *
     * @param areaCode
     * @return
     */
    private List<Long> getAdvertArea(String areaCode, Integer areaType) {
        areaCode = locationFacadeService.completeAreaCode(areaCode);
        return advertAreaMapper.getAllAdvertAreaByAreaCodeAndType(areaCode, areaType)
                .stream()
                .map(AdvertArea::getAdvertId)
                .collect(Collectors.toList());
    }

    @Override
    public HomeWindowDTO getHomeWindow(HomeWindowParam homeWindowParam, String ip) {
        return new HomeWindowDTO();
    }

    @Override
    public List<ClassificationDTO> vipPageClassificationList(AdvertParam param) {

        List<ClassificationDTO> classificationDTOList = JSONObject.parseArray(adverProperties.getClassification(), ClassificationDTO.class);

        AdvertisementParam advertisementParam = new AdvertisementParam();
        BeanUtils.copyProperties(param, advertisementParam);


        return classificationDTOList.stream().parallel().map(e -> {
            advertisementParam.setType(e.getId());
            ClassificationDTO classificationDTO = new ClassificationDTO();
            BeanUtils.copyProperties(e, classificationDTO);
            List<AdvertVO> advertDtoList = queryAdByType(advertisementParam);
            classificationDTO.setAdvertVOList(advertDtoList);
            return classificationDTO;
        }).collect(Collectors.toList());
    }

    @Override
    public AdvertVO getByMateriaId(Long id) {
        return advertMapper.getByMateriaId(id);
    }
}
