package com.bxm.localnews.thirdparty.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.thirdparty.config.ThirdPartyProperties;
import com.bxm.localnews.thirdparty.domain.MachineRecordMapper;
import com.bxm.localnews.thirdparty.service.MachineService;
import com.bxm.localnews.thirdparty.dto.MachineBaseResponse;
import com.bxm.localnews.thirdparty.param.MachineCallbackParam;
import com.bxm.localnews.thirdparty.vo.MachineRecordBean;
import com.bxm.localnews.thirdparty.dto.MachineResponse;
import com.bxm.localnews.thirdparty.vo.MachineState;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.RandomUtils;
import com.google.common.collect.Maps;
import com.qq.weixin.mp.aes.AesException;
import com.qq.weixin.mp.aes.SHA1;
import com.qq.weixin.mp.aes.WXBizMsgCrypt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Map;

@Service
public class MachineServiceImpl extends BaseService implements MachineService {

    private String openPaperUrl = "http://open.yilepie.com/app/index.php?c=api&a=opaper";

    private String verifyUrl = "http://open.yilepie.com/app/index.php?c=api&a=verify";

    private ThirdPartyProperties thirdPartyProperties;

    private HttpClientService httpClientService;

    private MachineRecordMapper machineRecordMapper;

    @Autowired
    public MachineServiceImpl(ThirdPartyProperties thirdPartyProperties, HttpClientService httpClientService,
                              MachineRecordMapper machineRecordMapper) {
        this.thirdPartyProperties = thirdPartyProperties;
        this.httpClientService = httpClientService;
        this.machineRecordMapper = machineRecordMapper;
    }

    @Override
    public Json<MachineState> sendPaper(String machineId, String devcId) {
        Json<MachineState> result = Json.build(new MachineState(0));
        //判断设备号是否已经进行过领取
        if (machineRecordMapper.hasRecord(devcId) > 0) {
            return result.setErrorMsg("您已经领取过纸巾了").setResult(new MachineState(2));
        }
        //判断设备是否在线
        MachineResponse onlineResponse = getMachineOnlineStatus(machineId);
        if (onlineResponse != null && "1".equals(onlineResponse.getStatus())) {
            //执行发纸操作
            boolean openPaperResult = buildOpenpaperRequest(machineId, devcId);
            if (!openPaperResult) {
                result.setErrorMsg("设备响应状态异常").setResult(new MachineState(3));
            }
        } else {
            result.setErrorMsg("设备故障").setResult(new MachineState(3));
        }
        return result;
    }

    @Override
    public MachineBaseResponse sendPaperSuccessCallback(String paramData, String timestamp, String nonce, String signature) {
        try {
            String responseJson = getMsgCrypt().decryptMsg(signature, timestamp, nonce, paramData);
            MachineCallbackParam param = JSONObject.parseObject(responseJson, MachineCallbackParam.class);
            if (null == param) {
                logger.error("非法请求,请求参数：paramData[{}],timestamp:[{}],nonce:[{}],signatrue:[{}]", paramData,
                        timestamp, nonce, signature);
            } else if ("PostLogin".equalsIgnoreCase(param.getTradeType())) {
                logger.info("PostLogin:[{}]设备已上线", param.getImei());
            } else if ("PostTask".equalsIgnoreCase(param.getTradeType())) {
                //记录出纸结果
                machineRecordMapper.updateRecord(param.getTaskId());
                logger.info("PostTask:出纸记录ID：{},出纸设备ID：{}，出纸时间：{}", param.getTaskId(), param.getImei(), param.getOpTime());
            } else if ("PostClose".equalsIgnoreCase(param.getTradeType())) {
                logger.info("PostClose:[{}]设备已下线", param.getImei());
            } else if ("PostError".equalsIgnoreCase(param.getTradeType())) {
                logger.info("PostError:[{}]设备异常", param.getImei());
            }
        } catch (AesException e) {
            logger.error(e.getMessage(), e);
        }
        return MachineBaseResponse.build();
    }

    private WXBizMsgCrypt getMsgCrypt() throws AesException {
        return new WXBizMsgCrypt(thirdPartyProperties.getMachineToken(),
                thirdPartyProperties.getMachineEncodingAESKey(),
                thirdPartyProperties.getMachineAppId());
    }

    @Override
    public MachineResponse getMachineOnlineStatus(String machineId) {
        String jsonBody = JSON.toJSONString(new OnlineRequest(machineId));
        //当前时间戳
        String currentMillis = System.currentTimeMillis() + "";
        //随机数
        String nonce = RandomUtils.getRandomStr(6, true);
        try {
            String afterEncrypt = getMsgCrypt().encryptMsg(jsonBody, currentMillis, nonce);
            String signature = SHA1.getSHA1(thirdPartyProperties.getMachineToken(), currentMillis, nonce, afterEncrypt);

            return post(verifyUrl, currentMillis, nonce, afterEncrypt, signature);
        } catch (AesException e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 发起远程请求
     */
    private MachineResponse post(String url, String currentMillis, String nonce, String afterEncrypt, String signature) {
        Map<String, String> postParamMap = Maps.newHashMap();
        postParamMap.put("ParamData", afterEncrypt);
        postParamMap.put("AppId", thirdPartyProperties.getMachineAppId());
        postParamMap.put("TimeStamp", currentMillis);
        postParamMap.put("Nonce", nonce);
        postParamMap.put("Signature", signature);

        String response = httpClientService.doPost(url, postParamMap);
        logger.info("post url:" + url);
        logger.info("post response:" + response);

        return JSONObject.parseObject(response, MachineResponse.class);
    }

    /**
     * 发起出纸指令
     * @param machineId 机器ID
     * @return true表示发送成功
     */
    private boolean buildOpenpaperRequest(String machineId, String devcId) {
        String taskId = nextSequence().toString();
        RequestBody body = new RequestBody(machineId, taskId);

        //主体内容
        String jsonBody = JSON.toJSONString(body);
        //当前时间戳
        String currentMillis = System.currentTimeMillis() + "";
        //随机数
        String nonce = RandomUtils.getRandomStr(6, true);

        try {
            String afterEncrypt = getMsgCrypt().encryptMsg(jsonBody, currentMillis, nonce);
            String signature = SHA1.getSHA1(thirdPartyProperties.getMachineToken(), currentMillis, nonce, afterEncrypt);
            MachineResponse response = post(openPaperUrl, currentMillis, nonce, afterEncrypt, signature);

            if (response != null && response.getResultCode() == 0) {
                //记录数据库状态
                MachineRecordBean entity = new MachineRecordBean();
                entity.setId(Long.valueOf(taskId));
                entity.setAddTime(new Date());
                entity.setMachineId(machineId);
                entity.setDevcId(devcId);
                entity.setStatus(0);
                machineRecordMapper.insert(entity);
                return true;
            }

            return false;
        } catch (AesException e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }

    public static class RequestBody {

        @JSONField(name = "IMEI")
        private String imei;

        @JSONField(name = "TaskId")
        private String taskId;

        @JSONField(name = "Number")
        private int number = 1;

        RequestBody(String imei, String taskId) {
            this.imei = imei;
            this.taskId = taskId;
        }

        public String getImei() {
            return imei;
        }

        public void setImei(String imei) {
            this.imei = imei;
        }

        public String getTaskId() {
            return taskId;
        }

        public void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        public int getNumber() {
            return number;
        }

        public void setNumber(int number) {
            this.number = number;
        }

    }

    public class OnlineRequest {

        @JSONField(name = "IMEI")
        private String imei;

        OnlineRequest(String imei) {
            this.imei = imei;
        }

        public String getImei() {
            return imei;
        }

        public void setImei(String imei) {
            this.imei = imei;
        }

    }

}
