package com.bxm.localnews.thirdparty.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.thirdparty.config.ThirdpartyJumpInfoProperties;
import com.bxm.localnews.thirdparty.dto.OrderTabDTO;
import com.bxm.localnews.thirdparty.param.JumpInfoConvertParam;
import com.bxm.localnews.thirdparty.param.OrderTabsParam;
import com.bxm.localnews.thirdparty.service.jump.JumpInfoConvert;
import com.bxm.localnews.thirdparty.service.jump.JumpInfoConvertService;
import com.bxm.localnews.thirdparty.vo.JumpInfo;
import com.bxm.localnews.thirdparty.vo.OrderTab;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Slf4j
@Service
public class ThirdpartyJumpInfoServiceImpl implements ThirdpartyJumpInfoService {

    @Autowired
    private ThirdpartyJumpInfoProperties thirdpartyJumpInfoProperties;

    @Autowired
    private JumpInfoConvertService jumpInfoConvertService;

    private LoadingCache<String, List<OrderTab>> cache;

    @Override
    public List<OrderTabDTO> orderTabs(OrderTabsParam param) {
        List<OrderTab> orderTabs = convertTabs();
        return orderTabs.stream()
                // 可用
                .filter(p -> p.getStatus().intValue() == 1)
                .map(p -> {
                    // 转换
                    Optional<JumpInfo> jpInfoOptional = jumpInfoConvertService.convert(JumpInfoConvertParam.builder()
                            .basicParam(param)
                            .jumpType(p.getJumpType())
                            .userId(param.getUserId())
                            .build());

                    if (jpInfoOptional.isPresent() && isNotBlank(jpInfoOptional.get().getProtocol())) {
                        OrderTabDTO tabDTO = new OrderTabDTO();
                        tabDTO.setJumpUrl(jpInfoOptional.get().getProtocol());
                        tabDTO.setTitle(p.getTitle());
                        tabDTO.setDefaultChose(p.getDefaultChose());
                        tabDTO.setId(p.getId());
                        return tabDTO;
                    }
                    return null;
                })
                // 过滤null的
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private List<OrderTab> convertTabs() {
        if (Objects.isNull(cache)) {
            cache = CacheBuilder.newBuilder()
                    // 只有一个
                    .maximumSize(1)
                    // 1h失效
                    .expireAfterWrite(1, TimeUnit.HOURS)
                    .build(new CacheLoader<String, List<OrderTab>>() {

                        @Override
                        public List<OrderTab> load(String key) throws Exception {
                            return thirdpartyJumpInfoProperties.getOrderTabs()
                                    .stream()
                                    // 转换对象
                                    .map(p -> JSON.parseObject(p, OrderTab.class))
                                    .collect(Collectors.toList());
                        }
                    });
        }

        return cache.getUnchecked("key");
    }

}
