package com.bxm.localnews.thirdparty.service.pop.popinstance.impl;

import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.thirdparty.config.PopupProperties;
import com.bxm.localnews.thirdparty.dto.HomeWindowDTO;
import com.bxm.localnews.thirdparty.dto.NormalNoviceUserGuideDTO;
import com.bxm.localnews.thirdparty.param.HomeWindowParam;
import com.bxm.localnews.thirdparty.service.pop.PopTypeEnum;
import com.bxm.localnews.thirdparty.service.pop.popinstance.PopContext;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import static java.util.Objects.isNull;

/**
 * 非vip用户的引导弹窗
 *
 * @author gonzo
 * @date 2020-08-21 13:54
 **/
@Slf4j
@Component
@AllArgsConstructor
public class NormalUserGuidePop extends AbstractGuidePopup {

    private final UserIntegrationService userIntegrationService;

    private final PopupProperties popupProperties;

    @Override
    boolean filter(PopContext context) {
        // 调用公共的校验
        if (!super.filter(context)) {
            return false;
        }

        HomeWindowParam param = context.getHomeWindowParam();
        LocationDetailDTO locationDetailDTO = context.getParam(LOCATION_DETAIL_DTO);

        // 查询用户信息
        UserInfoDTO userInfo = userIntegrationService.getUserFromRedisDB(param.getUserId());
        if (isNull(userInfo)) {
            log.warn("用户: {}不存在，跳过非vip用户的引导弹窗", param.getUserId());
            return false;
        }

        // 判断用户是新版本注册用户
        if (!check350JudgeMarker(userInfo.getJudgeMarker())) {
            log.info("用户: {} 非新版本用户 增加弹出缓存，永不弹出", param.getUserId());
            super.closeCache(context);
            return false;
        }

        if (check360JudgeMarker(userInfo.getJudgeMarker())) {
            // 新版本普通引导
            return version360(context, locationDetailDTO, userInfo, param);
        } else {
            // 老版本继续走老版本引导
            return version350(context, locationDetailDTO, userInfo, param);
        }
    }


    /**
     * 新版本弹出判断
     * @param context
     * @param locationDetailDTO
     * @param userInfo
     * @param param
     * @return
     */
    private boolean version360(PopContext context, LocationDetailDTO locationDetailDTO, UserInfoDTO userInfo, HomeWindowParam param) {
        // 新版本新用户 不管新是否是VIP 都弹出普通引导弹窗
        open(context, locationDetailDTO, userInfo, param);
        return true;
    }

    /**
     * 老版本的弹出判断
     * @param context
     * @param locationDetailDTO
     * @param userInfo
     * @param param
     * @return
     */
    private boolean version350(PopContext context, LocationDetailDTO locationDetailDTO, UserInfoDTO userInfo, HomeWindowParam param) {
        // 是否是站外邀请激活而来的VIP
        if (isInviteVip(userInfo)) {
            log.info("用户: {} 是激活类型的VIP 增加普通引导弹窗的弹出缓存，永不弹出", param.getUserId());
            super.closeCache(context);
            return false;
        }

        // 新版本 新用户 弹出一次
        open(context, locationDetailDTO, userInfo, param);
        return true;
    }

    /**
     * 设置弹窗数据
     * @param context
     * @param locationDetailDTO
     * @param userInfo
     * @param param
     */
    private void open(PopContext context, LocationDetailDTO locationDetailDTO, UserInfoDTO userInfo, HomeWindowParam param) {
        NormalNoviceUserGuideDTO normalNoviceUserGuide = new NormalNoviceUserGuideDTO();
        normalNoviceUserGuide.setUserId(param.getUserId());
        normalNoviceUserGuide.setImgUrl(userInfo.getHeadImg());
        normalNoviceUserGuide.setNickName(userInfo.getNickname());
        normalNoviceUserGuide.setTitle(String.format(popupProperties.getNormalNoviceUserGuideTitle(),
                locationDetailDTO.getName()));
        normalNoviceUserGuide.setContent(String.format(popupProperties.getNormalNoviceUserGuideContent(),
                locationDetailDTO.getName()));

        HomeWindowDTO windowDTO = new HomeWindowDTO();
        windowDTO.setType(PopTypeEnum.NORMAL_GUIDE_POP.getType());
        windowDTO.setNormalNoviceUserGuide(normalNoviceUserGuide);

        context.setHomeWindowDTO(windowDTO);
    }


    @Override
    protected String afterIncludeVersion() {
        return VER_3_5_0;
    }

    @Override
    protected String beforeExcludeVersion() {
        return VER_3_9_0;
    }
}
