package com.bxm.localnews.thirdparty.service.wx.push.impl;

import com.bxm.localnews.mq.common.param.WechatMpPushMessage;
import com.bxm.localnews.thirdparty.config.WechatMPConfig;
import com.bxm.localnews.thirdparty.constant.WechatMpTypeEnum;
import com.bxm.localnews.thirdparty.service.wx.push.WxMpMessageService;
import com.bxm.localnews.thirdparty.service.wx.push.templateStrategy.WxTemplateContextProcess;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.WxMpTemplateMsgService;
import me.chanjar.weixin.mp.bean.template.WxMpTemplateData;
import me.chanjar.weixin.mp.bean.template.WxMpTemplateMessage;
import me.chanjar.weixin.mp.util.WxMpConfigStorageHolder;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

@Service
@Slf4j
@AllArgsConstructor(onConstructor = @__(@Autowired))
public class WxMpMessageServiceImpl implements WxMpMessageService {

    private final WxMpService wxMpService;

    private final WechatMPConfig wechatMpConfig;

    private final WxTemplateContextProcess wxTemplateContextProcess;

    @Override
    public Message singlePush(WechatMpPushMessage message) {
        if (null == message || StringUtils.isBlank(message.getOid())
                || MapUtils.isEmpty(message.getValueMap())) {
            log.warn("使用公众号推送，参数不完整:{}", message);
            return Message.build(false, "未开启公众号推送功能");
        }
        String templateId = wxTemplateContextProcess.getTemplateId(message.getWxMpTemplate());
        if (null == templateId) {
            log.warn("使用公众号推送，模板不存在：{}", message);
            return Message.build(false, "使用公众号推送，模板不存在");
        }
        message.setTemplateId(templateId);
        List<WxMpTemplateData> wxMaTemplateDataList = generateTemplate(message);
        return sendTemplate(message, wxMaTemplateDataList, 3);
    }

    /**
     * 发送模板消息
     *
     * @param message    推送消息
     * @param list       消息填充参数，具体组装过程发生在消息模块
     * @param retryTimes 重试次数
     */
    private Message sendTemplate(WechatMpPushMessage message, List<WxMpTemplateData> list, int retryTimes) {
        if (retryTimes == 0) {
            log.error("重试后发送模板消息仍然失败");
            return Message.build(false);
        }
        try {
            String templateId = message.getTemplateId();
            Map<String, String> appIdByType = wechatMpConfig.getAppIdByType();
            log.debug("给用户[{}]发送[{}]模板消息", message.getUserId(), templateId);
            WxMpTemplateMsgService service = wxMpService.getTemplateMsgService();
            WxMpConfigStorageHolder.set(appIdByType.get(WechatMpTypeEnum.PUSHMSG.name()));
            String msgId = service.sendTemplateMsg(WxMpTemplateMessage.builder()
                    .templateId(templateId)
                    .data(list)
                    .url(message.getUrl())
                    .toUser(message.getOid())
                    .build());

            log.info("給用戶[{}]发送公众号消息成功，消息ID[{}]", message.getUserId(), msgId);
            return Message.build(true);
        } catch (WxErrorException e) {
            log.warn("给用户发送模板消息失败", e);
            //重试机制
            return sendTemplate(message, list, --retryTimes);
        } finally {
            WxMpConfigStorageHolder.set("default");
        }
    }

    /**
     * 拼装模板消息内容
     *
     * @param message 原始推送参数
     * @return 转换后的模板参数
     */
    private List<WxMpTemplateData> generateTemplate(WechatMpPushMessage message) {
        List<WxMpTemplateData> templateDataList = Lists.newArrayList();

        message.getValueMap().forEach((postion, content) -> {
            templateDataList.add(new WxMpTemplateData(postion, content));
        });

        return templateDataList;
    }
}
