package com.bxm.localnews.thirdparty.timer;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.task.AbstractCronTask;
import com.bxm.newidea.component.tools.DateUtils;
import com.xxl.job.core.biz.model.ReturnT;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import static com.bxm.localnews.common.rediskey.ShortUrlRedisKey.SHORT_URL;
import static com.bxm.localnews.common.rediskey.ShortUrlRedisKey.SHORT_URL_HIT_KEY;

/**
 * 每天将前一天被访问过的短链失效时间延长
 * 每天凌晨0:05执行
 * 其实会有一点小问题，如果被访问的短链在前一天刚好失效前被访问，但定时任务是第二天凌晨执行的，就会出现前一天失效的短链没办法续命
 * 但这个情况比较极端，略过吧
 *
 * @author gonzo
 * @date 2020-12-24 15:09
 **/
@Slf4j
@Component
public class HitedShortKeyRenewalTask extends AbstractCronTask<String> {

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    private static final long EXPIRE_SEC = 3600 * 24 * 90;

    @Override
    protected ReturnT<String> service(String s) {
        log.info("定时任务开始执行: 每天将前一天被访问过的短链失效时间延长");

        Date yesterday = DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -1);
        // 要考虑key很多的情况 但短时间内应该问题不大
        KeyGenerator key = SHORT_URL_HIT_KEY.copy().appendKey(DateUtils.formatDate(yesterday));
        Set<String> hotKeys = redisSetAdapter.getAllMembers(key, String.class);

        hotKeys.forEach(p -> {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("缓存key: {} 昨日被访问过，延迟失效", p);
                }

                // 添加失效时间
                redisStringAdapter.expire(SHORT_URL.copy().appendKey(p), EXPIRE_SEC);
            } catch (Exception e) {
                log.error("缓存key: {} 延迟时间时间失败", p, e);
            }
        });

        // 移除昨日的记录
        redisSetAdapter.remove(key);

        log.info("定时任务结束执行: 每天将前一天被访问过的短链失效时间延长");
        return ReturnT.SUCCESS;
    }

    @Override
    public String taskName() {
        return "HitedShortKeyRenewalTask";
    }

    @Override
    public String cron() {
        return "0 5 0 * * ?";
    }

    @Override
    public String description() {
        return "每天将前一天被访问过的短链失效时间延长";
    }
}
