package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import com.bxm.localnews.user.domain.CashFlowMapper;
import com.bxm.localnews.user.dto.CashFlowDTO;
import com.bxm.localnews.user.dto.UserDayCashDTO;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.WithdrawEnum;
import com.bxm.localnews.user.param.CashFlowParam;
import com.bxm.localnews.user.service.CashFlowService;
import com.bxm.localnews.user.service.UserWithdrawService;
import com.bxm.localnews.user.vo.CashFlow;
import com.bxm.localnews.user.vo.WithdrawFlow;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;

/**
 * @author zhaoyadong 2019/3/5 22:13
 * @desc
 */
@Service
public class CashFlowServiceImpl extends BaseService implements CashFlowService {
    private static final Logger LOG = LoggerFactory.getLogger(CashFlowServiceImpl.class);

    @Resource
    private CashFlowMapper cashFlowMapper;

    @Resource
    private UserWithdrawService userWithdrawService;

    @Override
    public List<UserDayCashDTO> getRecentSevenDayCashFlowByUser(Long userId) {
        return cashFlowMapper.querySevenDayCashFlows(userId);
    }

    @Override
    public List<CashFlowDTO> getUserAllCashFlow(CashFlowParam cashFlowParam) {
        if (StringUtils.isNotEmpty(cashFlowParam.getDate())) {
            Date date = DateUtils.parse(cashFlowParam.getDate());
            Date endDate = DateUtils.addField(date, Calendar.DATE, 1);
            cashFlowParam.setEndDate(DateUtils.formatAtWill(endDate, DateUtils.DATE_FORMAT));
            cashFlowParam.setType("income");
        } else {
            //查询所有的明细需要微信提现失败流水
            cashFlowParam.setType(null);
        }
        List<CashFlow> cashFlowList = cashFlowMapper.queryCashFlows(cashFlowParam);
        return generateCashFlowDTO(cashFlowList);
    }

    /**
     * 转换数据
     *
     * @param cashFlows
     * @return
     */
    private List<CashFlowDTO> generateCashFlowDTO(List<CashFlow> cashFlows) {
        if (CollectionUtils.isEmpty(cashFlows)) {
            return new ArrayList<>();
        }

        return cashFlows.stream().map(this::convertCashFlow).collect(Collectors.toList());
    }

    /**
     * 转换对象
     *
     * @param cashFlow
     * @return
     */
    private CashFlowDTO convertCashFlow(CashFlow cashFlow) {
        CashFlowDTO cashFlowDTO = new CashFlowDTO();
        cashFlow.setAmount(cashFlow.getAmount().setScale(2, BigDecimal.ROUND_HALF_UP));
        String amount;
        if (cashFlow.getAmount().compareTo(BigDecimal.ZERO) > 0) {
            amount = "+" + cashFlow.getAmount();
            cashFlowDTO.setPlus(Boolean.TRUE);
        } else {
            amount = cashFlow.getAmount().toString();
            cashFlowDTO.setPlus(Boolean.FALSE);
        }
        cashFlowDTO.setAmount(amount);
        cashFlowDTO.setCreateTime(cashFlow.getCreateTime());
        cashFlowDTO.setTypeDesc(CashFlowTypeEnum.valueOf(cashFlow.getType()).getDesc());
        if (CashFlowTypeEnum.WECHAT_WITHDRAWAL.getName().equals(cashFlow.getType()) && cashFlow.getRelationId() != null) {
            WithdrawFlow withdrawFlow = userWithdrawService.getWithdrawFlowById(cashFlow.getRelationId());
            if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc("提现到微信--已成功");
            }

            if (WithdrawEnum.AUDIT.getState().equals(withdrawFlow.getState()) ||
                    WithdrawEnum.DELAY_AUDIT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc("提现到微信--审核中");
            }

            if (WithdrawEnum.FAIL_PAYMENT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc("提现到微信–失败已退回");
            }
        }
        return cashFlowDTO;
    }

    @Override
    public int addCashFlow(CashFlow cashFlow) {
        logger.debug("新增流水：{}", cashFlow);
        return cashFlowMapper.addCashFlow(cashFlow);
    }

    @Override
    public BigDecimal getMySelfBounty(Long userId) {
        BigDecimal myBounty = this.cashFlowMapper.getMySelfBounty(userId);
        return myBounty == null ? BigDecimal.ZERO : myBounty;
    }

    @Override
    public BigDecimal getMyRedPacket(Long userId) {
        BigDecimal myRedPacket = this.cashFlowMapper.getMyRedPacket(userId);
        return myRedPacket == null ? BigDecimal.ZERO : myRedPacket;
    }

    @Override
    public boolean enableWithdraw(Long userId) {
        CashFlowParam cashFlowParam = new CashFlowParam();
        cashFlowParam.setUserId(userId);

        List<CashFlow> cashFlowList = cashFlowMapper.queryCashFlows(cashFlowParam);
        if (CollectionUtils.isEmpty(cashFlowList)) {
            LOG.info("未领取过新用户赏金，用户id：" + userId);
            return false;
        }
        return true;
    }
}
