package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.InviteRecordMapper;
import com.bxm.localnews.user.dto.BountyDTO;
import com.bxm.localnews.user.dto.InviteRecordDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserPayPromoteDTO;
import com.bxm.localnews.user.enums.AwardTypeEnum;
import com.bxm.localnews.user.enums.InviteRecordStateEnum;
import com.bxm.localnews.user.enums.InviteRecordStatusEnum;
import com.bxm.localnews.user.integration.QuartzManageIntegrationService;
import com.bxm.localnews.user.param.InviteExpireTaskParam;
import com.bxm.localnews.user.param.InviteRecordParam;
import com.bxm.localnews.user.service.CashFlowService;
import com.bxm.localnews.user.service.InviteRecordService;
import com.bxm.localnews.user.service.UserService;
import com.bxm.localnews.user.vo.InviteRecord;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;

import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;

/**
 * @author zhaoyadong 2019/3/6 9:25
 * @desc
 */
@Service
public class InviteRecordServiceImpl extends BaseService implements InviteRecordService {

    @Resource
    private InviteRecordMapper inviteRecordMapper;

    @Resource
    private QuartzManageIntegrationService quartzManageIntegrationService;

    @Resource
    private UserService userService;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private CashFlowService cashFlowService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public InviteRecord getInviteRecordByInvitedUser(Long inviteUserId, Long invitedUserId) {

        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(inviteUserId);
        inviteRecordParam.setInvitedUserId(invitedUserId);
        inviteRecordParam.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        inviteRecordParam.setAwardType(AwardTypeEnum.CASH.name());
        return inviteRecordMapper.getInviteRecordByUser(inviteRecordParam);
    }

    @Override
    public InviteRecord getInviteRecordById(Long inviteUserId, Long id) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setId(id);
        inviteRecordParam.setUserId(inviteUserId);
        return inviteRecordMapper.getInviteRecordByUser(inviteRecordParam);
    }

    @Override
    public Boolean updateInviteRecord(InviteRecord inviteRecord) {
        logger.info("更新邀请关系[{}]", JSON.toJSONString(inviteRecord));
        this.inviteRecordMapper.updateInviteRecord(inviteRecord);
        return Boolean.TRUE;
    }

    @Override
    public BountyDTO getUserBountyInfo(Long userId) {
        List<InviteRecord> inviteRecords = getUserAllInviteRecordByCash(userId);

        return generateBounty(inviteRecords);
    }

    @Override
    public UserPayPromoteDTO getUserPayPromote(Long userId) {
        List<InviteRecord> inviteRecords = getUserAllInviteRecordByCash(userId);
        return generatePayPromote(inviteRecords, userId);
    }

    @Override
    public Long addInviteRecord(Long inviteUserId, Long invitedUserId, String type) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(inviteUserId);
        inviteRecordParam.setInvitedUserId(invitedUserId);
        inviteRecordParam.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        List<InviteRecord> inviteRecordList = inviteRecordMapper.queryUserInviteRecord(inviteRecordParam);

        if (CollectionUtils.isEmpty(inviteRecordList)) {
            Long id = nextSequence();
            Date now = new Date();
            Date effectiveTime = DateUtils.addDays(now, 1);
            UserInfoDTO userInfoDTO = userService.getUserFromRedisDB(invitedUserId);
            //分布式锁保证不会重复插入
            if (distributedLock.lock(invitedUserId.toString(), id.toString(), 1, TimeUnit.SECONDS)) {
                InviteRecord inviteRecord = new InviteRecord();
                inviteRecord.setId(id);
                inviteRecord.setUserId(inviteUserId);
                inviteRecord.setInvitedUserId(invitedUserId);
                inviteRecord.setType(type);
                inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.name());
                inviteRecord.setEffectTime(effectiveTime);
                inviteRecord.setInvitedUserName(userInfoDTO.getNickname());
                inviteRecord.setInvitedUserImg(userInfoDTO.getHeadImg());
                inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
                inviteRecord.setCreateTime(now);
                inviteRecord.setModifyTime(now);

                inviteRecordMapper.addInviteRecord(inviteRecord);

                InviteExpireTaskParam param = new InviteExpireTaskParam();
                param.setJobId(inviteRecord.getUserId() + "_" + inviteRecord.getInvitedUserId());
                param.setInviteUserId(inviteRecord.getUserId());
                param.setInvitedUserId(inviteRecord.getInvitedUserId());
                param.setStartTime(effectiveTime);
                quartzManageIntegrationService.createInviteTask(param);

                return id;
            }
        }

        return 0L;
    }

    @Override
    public Long updateInviteCash(Long inviteUserId, Long invitedUserId, BigDecimal cash) {
        if (null == inviteUserId || null == invitedUserId) {
            return 0L;
        }

        InviteRecordParam param = new InviteRecordParam();
        param.setUserId(inviteUserId);
        param.setInvitedUserId(invitedUserId);
        param.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        param.setAwardType(AwardTypeEnum.CASH.name());
        List<InviteRecord> records = inviteRecordMapper.queryUserInviteRecord(param);

        if (!CollectionUtils.isEmpty(records)) {
            inviteRecordMapper.updateInviteCash(inviteUserId, invitedUserId, cash);
            return records.get(0).getId();
        }

        return 0L;
    }

    @Override
    public Boolean checkIsAssociation(Long userId) {
        return inviteRecordMapper.checkIsAssociation(userId) > 0;
    }

    /**
     * 获取用户所有现金奖励类型的数据
     *
     * @return
     */
    private List<InviteRecord> getUserAllInviteRecordByCash(Long userId) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(userId);
        inviteRecordParam.setAwardType(AwardTypeEnum.CASH.toString());
        inviteRecordParam.setAward(BigDecimal.ZERO);
        return inviteRecordMapper.queryUserInviteRecord(inviteRecordParam);
    }

    /**
     * 生成赏金信息
     *
     * @param inviteRecords
     * @return
     */
    private BountyDTO generateBounty(List<InviteRecord> inviteRecords) {
        BountyDTO bountyDTO = new BountyDTO();
        //筛选有效和无效的赏金信息
        if (CollectionUtils.isEmpty(inviteRecords)) {
            return bountyDTO;
        }

        //有效赏金
        List<InviteRecordDTO> validList = filterValid(inviteRecords);
        if (!CollectionUtils.isEmpty(validList)) {
            List<InviteRecordDTO> availableCashList = bountyDTO.getAvailableCashList();
            BigDecimal availableCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : validList) {
                if (InviteRecordStateEnum.ACCEPT_INVITE.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.getDesc());
                    availableCashList.add(inviteRecord);
                } else if (InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(getInviteStateDesc(inviteRecord.getInviteState(), inviteRecord.getInvitedUserId()));
                    inviteRecord.setEffectTime(null);
                    availableCashList.add(inviteRecord);
                }
                availableCash = availableCash.add(inviteRecord.getAward());
            }
            bountyDTO.setAvailableCash(availableCash);
            bountyDTO.setAvailableCashList(availableCashList);
        }

        //失效赏金
        List<InviteRecordDTO> invalidList = filterInValid(inviteRecords);
        if (!CollectionUtils.isEmpty(invalidList)) {
            List<InviteRecordDTO> discardCashList = bountyDTO.getDiscardCashList();
            BigDecimal discardCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : invalidList) {
                if (InviteRecordStateEnum.OVERDUE_INVALID.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.getDesc());
                } else if (InviteRecordStateEnum.NOT_NEW_USER.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.NOT_NEW_USER.getDesc());
                }
                inviteRecord.setEffectTime(null);
                discardCashList.add(inviteRecord);
                discardCash = discardCash.add(inviteRecord.getAward());
            }
            bountyDTO.setDiscardCash(discardCash);
            bountyDTO.setDiscardCashList(discardCashList);
        }

        return bountyDTO;
    }

    /**
     * 筛选在路上的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecordDTO> filterValid(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                                !InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
    }

    /**
     * 筛选已废弃的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecordDTO> filterInValid(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.INVALID.getStatus().equals(inviteRecord.getStatus()))
                .collect(Collectors.toList());
    }

    /**
     * 筛选已获得的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecordDTO> filterObtain(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                                InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
    }

    /**
     * 转换对象
     *
     * @param inviteRecord
     * @return
     */
    private InviteRecordDTO convertInviteRecord(InviteRecord inviteRecord) {
        InviteRecordDTO inviteRecordDTO = new InviteRecordDTO();
        BeanUtils.copyProperties(inviteRecord, inviteRecordDTO);
        return inviteRecordDTO;
    }

    /**
     * 生成推广赚钱信息
     *
     * @param inviteRecords
     * @return
     */
    private UserPayPromoteDTO generatePayPromote(List<InviteRecord> inviteRecords, Long userId) {
        BountyDTO bountyDTO = generateBounty(inviteRecords);
        UserPayPromoteDTO userPayPromoteDTO = new UserPayPromoteDTO();
        userPayPromoteDTO.setAvailableCash(bountyDTO.getAvailableCash());
        userPayPromoteDTO.setAvailableCashList(bountyDTO.getAvailableCashList());
        userPayPromoteDTO.setDiscardCash(bountyDTO.getDiscardCash());
        userPayPromoteDTO.setDiscardCashList(bountyDTO.getDiscardCashList());

        //获取新用户红包
        BigDecimal myRedPacket = cashFlowService.getMyRedPacket(userId);

        //获取已获得赏金
        List<InviteRecordDTO> obtainList = filterObtain(inviteRecords);
        if (!CollectionUtils.isEmpty(obtainList)) {
            List<InviteRecordDTO> havingCashList = userPayPromoteDTO.getHavingCashList();
            BigDecimal havingCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : obtainList) {
                inviteRecord.setInviteState(getInviteStateDesc(inviteRecord.getInviteState(), userId));
                havingCashList.add(inviteRecord);
                havingCash = havingCash.add(inviteRecord.getAward());
            }

            userPayPromoteDTO.setHavingCash(havingCash);
            userPayPromoteDTO.setHavingCashList(havingCashList);
        }

        BigDecimal havingCash = userPayPromoteDTO.getHavingCash();
        if (null != myRedPacket && myRedPacket.compareTo(BigDecimal.ZERO) > 0) {
            UserInfoDTO userInfo = userService.getUserFromRedisDB(userId);
            InviteRecordDTO inviteRecord = new InviteRecordDTO();
            inviteRecord.setInvitedUserName(userInfo.getNickname());
            inviteRecord.setInvitedUserImg(userInfo.getHeadImg());
            inviteRecord.setAward(myRedPacket);
            inviteRecord.setInviteState("新用户红包");
            userPayPromoteDTO.getHavingCashList().add(inviteRecord);
            userPayPromoteDTO.setHavingCash(havingCash.add(inviteRecord.getAward()));
        }

        //累计赚钱
        userPayPromoteDTO.setTotalGainCash(userPayPromoteDTO.getAvailableCash().add(userPayPromoteDTO.getHavingCash()));
        return userPayPromoteDTO;
    }

    //获取邀请状态描述
    private String getInviteStateDesc(String name, Long userId) {
        if (InviteRecordStateEnum.LOGIN_APP.getName().equals(name)) {
            Integer inviteState = redisHashMapAdapter.get(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), Integer.class);
            if (null != inviteState && 1 == inviteState) {
                return "需第二天登录";
            } else {
                return InviteRecordStateEnum.LOGIN_APP.getDesc();
            }
        } else  {
            return InviteRecordStateEnum.valueOf(name).getDesc();
        }
    }

}
