package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;
import java.util.Date;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.integration.*;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.param.RiskParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

/**
 * Created by mars on 2018/2/23.
 * 登录
 */
@Service("loginService")
@RefreshScope
public class LoginServiceImpl extends BaseService implements LoginService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Resource
    private SmsIntegrationService smsIntegrationService;

    @Resource
    private UserAuthMapper userAuthMapper;

    @Resource
    private UserMapper userMapper;

    @Resource
    private InviteRelationIntegrationService inviteRelationIntegrationService;

    @Resource
    private UserService userService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private VipIntegrationService vipIntegrationService;

    @Resource
    private UserLoginHistoryService userLoginHistoryService;

    @Resource
    private UserNewsIntegrationService userNewsIntegrationService;

    @Resource
    private UserSyncService userSyncService;

    @Resource
    private InviteRecordService inviteRecordService;

    @Resource
    private UserAccountService userAccountService;

    @Resource
    private QuartzManageIntegrationService quartzManageIntegrationService;

    @Resource
    private CashFlowService cashFlowService;

    @Resource
    private RiskRecordService riskRecordService;

    @Resource
    private UserEquipmentService userEquipmentService;

    @Resource
    private ChannelService channelService;

    @Override
    public Json<LoginMeta> doLogin(LoginInfo loginInfo, BasicParam basicParam) {
        boolean flag = false;

        //ios提测时用的测试账号
        if (this.appVersionIntegrationService.inWhiteList(loginInfo.getLoginName())) {
            flag = true;
            loginInfo.setCode("123456");
        }

        //判断参数是否正确
        if (!this.checkParam(loginInfo)) {
            this.logger.info("doLogin failed:" + loginInfo);
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }

        UserAuth userAuth = this.userAuthMapper.selectByUserAuth(loginInfo.getType(), loginInfo.getLoginName());

        //第三方登录,如果未绑定手机号码，提示进行绑定
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()) {
            if (null == userAuth) {
                if (getPublishState(basicParam)) {
                    //TODO 处于审核状态时用户信息未返回
                    LoginMeta meta = new LoginMeta();
                    meta.setCode(RespCode.BIND_PHONE);
                    return ResultUtil.genSuccessResult(meta);
                } else {
                    return registerAndBindingPhone(loginInfo, basicParam.getPlatform(), null, basicParam.getChnl());
                }

            }
        }

        //找回密码（弃用）
        if (checkRetrievePwd(loginInfo)) {
            Message message = retrievePwd(loginInfo, userAuth, flag);
            if (!message.isSuccess()) {
                return ResultUtil.genFailedResult(message.getMessages().get(0));
            }
        }

        //密码登录
        if (checkVerifyPwd(loginInfo)) {
            Message message = verifyPwd(loginInfo.getPassword(), userAuth);
            if (!message.isSuccess()) {
                return ResultUtil.genFailedResult(message.getLashMessaage());
            }
        }

        //更新用户信息
        LoginMeta meta = updateUserAfterLogin(userAuth, loginInfo, basicParam);

        if (null != meta && meta.getUser() != null) {
            bizLogIntegrationService.login(meta.getUser().getId(), basicParam.getPlatform());
        }

        return ResultUtil.genSuccessResult(meta);
    }

    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionIntegrationService.getPublishState(basicParam);
    }

    /**
     * 判断是否是密码登录
     *
     * @param loginInfo
     * @return
     */
    private boolean checkVerifyPwd(LoginInfo loginInfo) {
        return AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isNotBlank(loginInfo.getPassword())
                && !loginInfo.getResetPwd();
    }

    /**
     * 判断是否是找回密码
     *
     * @param loginInfo
     * @return
     */
    private boolean checkRetrievePwd(LoginInfo loginInfo) {
        return AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isNotBlank(loginInfo.getPassword())
                && null != loginInfo.getResetPwd()
                && loginInfo.getResetPwd();
    }

    /**
     * 找回密码进行的操作
     *
     * @param loginInfo
     * @param userAuth
     * @return
     */
    private Message retrievePwd(LoginInfo loginInfo, UserAuth userAuth, boolean flag) {
        if (null == userAuth) {
            return Message.build(false, "手机号码未注册，请前往注册");
        }

        if (!flag && !this.smsIntegrationService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_RETRIEVE_PWD,
                loginInfo.getLoginName(), loginInfo.getCode())) {
            return Message.build(false, "请输入正确的验证码");
        }

        String encodePwd = MD5Util.hgmd5(loginInfo.getPassword());
        userAuth.setCredential(encodePwd);
        this.userAuthMapper.updateByPrimaryKeySelective(userAuth);
        return Message.build(true);
    }

    /**
     * 验证账号密码登录
     *
     * @param password
     * @param userAuth
     * @return
     */
    private Message verifyPwd(String password, UserAuth userAuth) {
        String encodePwd = MD5Util.hgmd5(password);
        if (null == userAuth) {
            //若用户账号不存在
            return Message.build(false, "手机号码未注册，请前往注册");
        }
        if (!userAuth.getCredential().equals(encodePwd)) {
            //密码不一致则直接返回
            return Message.build(false, "账号或密码错误");
        }
        return Message.build(true);
    }

    /**
     * 在登录之后需要对用户信息进行更新
     *
     * @param userAuth
     * @param loginInfo
     * @return
     */
    private LoginMeta updateUserAfterLogin(UserAuth userAuth, LoginInfo loginInfo, BasicParam basicParam) {
        boolean isFirstLogin;
        User user = this.userMapper.selectByPrimaryKey(userAuth.getUserId());

        isFirstLogin = StringUtils.isBlank(user.getEquipment());

        //如果一开始是手机号登录，那么用户是没有头像和昵称的，接下来如果用户用微信登录，就更新一次同步微信头像，昵称，现在已无用
        //this.updateNicknameAndHeadImg(loginInfo, user);

        //如果是在站外H5微信授权登录，更新一次同步微信头像，昵称，并且在auth中插入openId和unionId
        this.updateWechat(loginInfo, user, basicParam.getPlatform());

        //保存用户登录信息
        this.userLoginHistoryService.save(user, loginInfo);

        //更新用户位置，跟随静默用户的位置
        this.callAsyncUpdateLocation(loginInfo, user);

        //得到登录用户的包装类
        UserInfoDTO dto = this.getUpdateUser(loginInfo, user, basicParam.getChnl());

        //返回给客户端包装类
        LoginMeta meta = new LoginMeta(dto, isFirstLogin);

        //首次APP登录，设置用户为有效用户
        //第一次登陆绑定VIP邀请关系
        if (isFirstLogin) {
            setNewUserCache(userAuth.getUserId());

            if (loginInfo.getType() == AppConst.LOGIN_TYPE.WEIXIN) {
                logger.info("用户第一次微信授权登陆-----绑定VIP邀请关系");
                //处理vip邀请信息
                dealVipActivation(user.getId(), loginInfo.getLoginName(), basicParam.getPlatform());
                userNewsIntegrationService.initUserNewsImage(user.getId(), null);
            }
        }

        //缓存redis
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userAuth.getUserId().toString(), JSON.toJSONString(meta.getUser()));
        return meta;
    }

    /**
     * 得到更新后的用户包装类
     *
     * @param loginInfo
     * @param user
     * @return
     */
    private UserInfoDTO getUpdateUser(LoginInfo loginInfo, User user, String chnl) {
        if (StringUtils.isNotBlank(loginInfo.getEquipment())) {
            user.setEquipment(loginInfo.getEquipment());
        }
        //更新用户token
        user.setToken(this.userService.getToken());
        user.setExpiretime(this.userService.getExpireTime());
        user.setRefreshtoken(user.getRefreshtoken());
        user.setLastLoginTime(new Date());
        user.setLastLoginIp(loginInfo.getLastLoginIp());
        if (user.getChannelId() == 0) {
            user.setChannelId(channelService.getChannelByCode(chnl).getId());
        }
        logger.debug("用户[{}]对应的状态：[{}]", user.getId(), user.getState());
        logger.debug("用户状态是否处于未激活状态：[{}]", AppConst.USER_STATE.NOT_ACTIVE == user.getState());

        if (AppConst.USER_STATE.NOT_ACTIVE == user.getState()) {
            UserAccount userAccount = userAccountService.getUserAccountDetail(user.getId());
//            if (userAccount.getTotalCash().compareTo(BigDecimal.ZERO) > 0) {
//                user.setReceiveRedPacket((byte) 0);
//            }
            user.setState(AppConst.USER_STATE.NORMAL);
            //若存在同设备号用户已领取推广赏金，则失效当前赏金
            Boolean equipmentExist = this.userEquipmentService.checkExistEquipment(loginInfo.getEquipment(), "DRAW");
            BigDecimal cash = cashFlowService.getMySelfBounty(userAccount.getUserId());

            Long relationId;
            if (equipmentExist) {
                relationId = updateInviteStatus(user, false);
                updateUserCashState(userAccount, relationId, false, cash);
            } else {
                relationId = updateInviteStatus(user, true);
                updateUserCashState(userAccount, relationId, true, cash);
                if (cash.compareTo(BigDecimal.ZERO) > 0) {
                    UserEquipment userEquipment = new UserEquipment(nextId(), loginInfo.getEquipment(), "DRAW");
                    userEquipmentService.addUserEquipment(userEquipment);
                }
            }

            quartzManageIntegrationService.removeInviteTask(user.getInviteUserId() + "_" + user.getId());
            bizLogIntegrationService.newUser(user.getId(), chnl, user.getToken(),
                    userAccount.getTotalCash(), StringUtils.isNotEmpty(user.getRegisterChannel()) ? user.getRegisterChannel() : "0"
                    , null == user.getRelationId() ? null : user.getRelationId().toString(),
                    null == user.getRegisterClient() ? null : Integer.valueOf(user.getRegisterClient()));
        }

        this.userMapper.updateByPrimaryKeySelective(user);
        return this.userService.convertUserToDTO(user);
    }

    private Long updateInviteStatus(User user, boolean isEffective) {
        if (null == user.getInviteUserId()) {
            return null;
        }

        InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), user.getId());
        if (null == inviteRecord) {
            return null;
        }

        updateInviteUserState(inviteRecord, isEffective);
        if (isEffective) {
            //增加邀请成功日志
            bizLogIntegrationService.inviteSuccessed(inviteRecord.getUserId(), user.getId(),
                    null == user.getRegisterClient() ? null : Integer.valueOf(user.getRegisterClient()));
        }

        return inviteRecord.getId();
    }

    /**
     * 新用户首次登陆赏金状态更改
     *
     * @param userAccount
     * @param relationId
     * @param isEffective
     */
    private void updateUserCashState(UserAccount userAccount, Long relationId, boolean isEffective, BigDecimal cash) {
        logger.debug("新用户[{}]首次登陆临时赏金[{}]转换为可提现赏金", userAccount.getUserId(), cash);
        if (cash.compareTo(BigDecimal.ZERO) <= 0) {
            return;
        }

        if (isEffective) {
            AccountCashParam param = new AccountCashParam(userAccount.getUserId(), CashEnum.TEMP_CASH.name(), false,
                    cash, relationId, null, CashFlowConvertTypeEnum.TEMP_TO_DRAWABLE.name());
            userAccountService.transferCash(param);
        } else {
            AccountCashParam accountCashParam = new AccountCashParam();
            accountCashParam.setUserId(userAccount.getUserId());
            accountCashParam.setCashType(CashEnum.TEMP_CASH.name());
            accountCashParam.setAddTotal(false);
            accountCashParam.setCashFlowType(CashFlowTypeEnum.INVALID_PACKET.name());
            accountCashParam.setCash(cash.negate());
            accountCashParam.setRelationId(relationId);
            userAccountService.addCash(accountCashParam);
        }
    }

    /**
     * 更新邀请人的赏金状态
     *
     * @param inviteRecord
     * @param isEffective
     */
    private void updateInviteUserState(InviteRecord inviteRecord, boolean isEffective) {
        //邀请关系被解除
        if (inviteRecord == null) {
            return;
        }
        if (!InviteRecordStateEnum.ACCEPT_INVITE.name().equals(inviteRecord.getInviteState())) {
            logger.info("当前邀请人[{}]与受邀人[{}]的状态处于[{}]------不予处理", inviteRecord.getUserId(),
                    inviteRecord.getInvitedUserId(), inviteRecord.getInviteState());
        }

        if (isEffective) {
            AccountCashParam param = new AccountCashParam(
                    inviteRecord.getUserId(),
                    CashEnum.STABILIZE_CASH.name(),
                    false,
                    inviteRecord.getAward(),
                    inviteRecord.getId(), CashFlowTypeEnum.PROMOTE_BOUNTY.name(),
                    CashFlowConvertTypeEnum.TEMP_TO_STABILIZE.name());
            userAccountService.transferCash(param);
            //更新邀请状态为已登陆
            inviteRecord.setInviteState(InviteRecordStateEnum.LOGIN_APP.getName());
            inviteRecord.setEffectTime(null);
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecordService.updateInviteRecord(inviteRecord);
        } else {
            AccountCashParam accountCashParam = new AccountCashParam();
            accountCashParam.setUserId(inviteRecord.getUserId());
            accountCashParam.setCashType(CashEnum.TEMP_CASH.name());
            accountCashParam.setAddTotal(false);
            accountCashParam.setCashFlowType(CashFlowTypeEnum.INVALID_BOUNTY.name());
            accountCashParam.setCash(inviteRecord.getAward().negate());
            accountCashParam.setRelationId(inviteRecord.getId());
            userAccountService.addCash(accountCashParam);
            //更新邀请状态为无效
            inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.name());
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecord.setStatus((byte) 0);
            inviteRecordService.updateInviteRecord(inviteRecord);
        }
    }

    /**
     * 如果是微信登录，更新一次同步微信头像，昵称
     *
     * @param loginInfo
     * @param user
     */
    private void updateNicknameAndHeadImg(LoginInfo loginInfo, User user) {
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {

            if (isTempNickName(loginInfo, user) || isTempHeadImg(loginInfo, user)) {
                User updateUser = new User();
                updateUser.setId(user.getId());
                if (isTempNickName(loginInfo, user)) {
                    updateUser.setNickname(loginInfo.getNickname());
                }

                if (isTempHeadImg(loginInfo, user)) {
                    updateUser.setHeadImg(loginInfo.getHeadImg());
                }

                this.userSyncService.sync(updateUser);
            }


        }
    }

    private boolean isTempNickName(LoginInfo loginInfo, User user) {
        return user.getIsTempNickName() == 1 && null != loginInfo.getNickname();
    }

    private boolean isTempHeadImg(LoginInfo loginInfo, User user) {
        return user.getIsTempHeadImg() == 1 && null != loginInfo.getHeadImg();
    }

    /**
     * 如果是在站外H5微信授权登录,则更新openId和unionId和昵称头像
     * 2.0.0版本 没有h5授权登陆 openId是APP端传入保证用户能够正常提现
     *
     * @param loginInfo
     * @param user
     */
    private void updateWechat(LoginInfo loginInfo, User user, int platform) {
        logger.debug("用户登陆传入的参数：[{}]", JSON.toJSONString(loginInfo));
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            loginInfo.setUnionId(loginInfo.getLoginName());
        }
        if (null != loginInfo.getOpenId() && null != loginInfo.getUnionId()) {
            UserAuth userAuth = userAuthMapper.selectByUserId(AppConst.LOGIN_TYPE.WEIXIN, user.getId());
            if (userAuth == null) {
                logger.debug("当前用户创建unionId鉴权信息:用户id：[{}], unionId:[{}]", user.getId(), loginInfo.getUnionId());
                Message isBindedMsg = checkBindExist(AppConst.LOGIN_TYPE.WEIXIN, loginInfo.getUnionId());
                if (isBindedMsg.isSuccess()) {
                    user.setWeixin(loginInfo.getUnionId());
                    this.userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN, user.getId(),
                            loginInfo.getUnionId(), null);
//                    if (null != loginInfo.getNickname()) {
//                        user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
//                    }
//                    if (null != loginInfo.getHeadImg()) {
//                        user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
//                    }
//                    //同步微信
//                    userSyncService.sync(user);
                }
            }
//            if (platform == AppConst.PLATFORM.WEB) {
//                userAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID,
//                        loginInfo.getOpenId());
//                if (userAuth == null) {
//                    this.userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID, user.getId(),
//                            loginInfo.getOpenId(), null);
//                }
//            } else {
//
//            }
            userAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.APP_OPENID, loginInfo.getOpenId());
            if (userAuth == null) {
                logger.debug("当前用户创建APP微信授权登陆openId鉴权信息:用户id：[{}], openId:[{}]", user.getId(),
                        loginInfo.getOpenId());
                this.userAuthService.addUserAuth(AppConst.LOGIN_TYPE.APP_OPENID, user.getId(),
                        loginInfo.getOpenId(), null);
            }
        }
    }

    private void callAsyncUpdateLocation(LoginInfo loginInfo, User user) {
        LoginService loginService = SpringContextHolder.getBean(LoginService.class);
        loginService.updateLocation(loginInfo, user);
    }

    @Override
    @Async
    public void updateLocation(LoginInfo loginInfo, User user) {
        if (null != loginInfo.getId()) {
            User originalUser = userMapper.selectByPrimaryKey(loginInfo.getId());
            if (null != originalUser && null != originalUser.getLocationCode() && null != originalUser.getLocationName()) {
                user.setLocationCode(originalUser.getLocationCode());
                user.setLocationName(originalUser.getLocationName());
            }
        }
    }

    private void copyUserInfo(LoginInfo loginInfo, User user) {
        // 从访客账号拷贝标签权重信息和频道列表
        if (loginInfo.getId() == null || loginInfo.getId() == 0L) {
            this.logger.warn(LogMarker.BIZ, "用户注册，但是没有提供静默注册ID[H5站外注册例外]，loginInfo:[{}],user:[{}]", loginInfo, user);
            userNewsIntegrationService.initUserNewsImage(user.getId(), null);
        } else {
            this.logger.debug(LogMarker.BIZ, "用户注册,从访客账号拷贝标签权重信息和频道列表，loginInfo:[{}],user:[{}]", loginInfo, user);
            userNewsIntegrationService.initUserNewsImage(user.getId(), loginInfo.getId());
        }
    }

    @Override
    public Json<LoginMeta> registerAndBindingPhone(LoginInfo loginInfo, int platform, Long userId, String chnl) {
        if (!this.bandingAndRegisterCheckParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }

        //auth层面的验证  根据登录类型和登录标识去auth表里查 看看有没有被别人绑定
        //1.手机号注册：根据type=1和手机号去auth表里查是否#手机号#有被绑定的记录
        //2.第三方注册：根据type=3和微信unionId去auth表里查是否#微信#有被绑定的记录
        Message messageByCheckAuth = checkBindExist(loginInfo.getType(), loginInfo.getLoginName());
        if (!messageByCheckAuth.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth.getLashMessaage());
        }

        //user层面的验证  根据手机号去user表里查是否有相同手机号的绑定，如果有第三方绑定的话则视为为已有用户绑定第三方
        User user = getUserByType(loginInfo);
        Message messageByCheckUser = checkRepeatBind(user, loginInfo);
        if (!messageByCheckUser.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckUser.getLashMessaage());
        }

        //用户绑定手机号时
        if (null != userId) {
            user = userMapper.selectByPrimaryKey(userId);
            if (null != user && 0 == user.getState()) {
                user = null;
            }
        }

        //手机号绑定需要验证码校验
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            if (StringUtils.isNotBlank(loginInfo.getCode()) &&
                    !this.smsIntegrationService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE,
                            loginInfo.getPhone(), loginInfo.getCode())) {
                return ResultUtil.genFailedResult("请输入正确的验证码");
            }
        }

        //如果用户存在则进行绑定，如果用户不存在则添加用户并绑定第三方
        LoginMeta meta = registerAndBindingPhone(user, loginInfo, platform, chnl);

        return ResultUtil.genSuccessResult(meta);
    }

    private User getUserByType(LoginInfo loginInfo) {
        User user = null;
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user = this.userMapper.findByWeixin(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user = this.userMapper.findByPhone(loginInfo.getPhone());
        }
        return user;
    }

    private Message checkRepeatBind(User user, LoginInfo loginInfo) {
        if (null != user) {
            if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType() && StringUtils.isNotEmpty(user.getPhone())) {
                return Message.build(false, "手机号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType() && StringUtils.isNotEmpty(user.getWeixin())) {
                return Message.build(false, "微信号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType() && StringUtils.isNotEmpty(user.getWeibo())) {
                return Message.build(false, "微博已被绑定");
            } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType() && StringUtils.isNotEmpty(user.getQq())) {
                return Message.build(false, "QQ号已被绑定");
            }
        }
        return Message.build(true);
    }

    private Message checkBindExist(Byte loginType, String loginName) {
        UserAuth auth = this.userAuthMapper.selectByUserAuth(loginType, loginName);
        if (null != auth) {
            if (AppConst.LOGIN_TYPE.PHONE == loginType) {
                return Message.build(false, "手机号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginType) {
                return Message.build(false, "微信号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginType) {
                return Message.build(false, "微博已被绑定");
            } else if (AppConst.LOGIN_TYPE.QQ == loginType) {
                return Message.build(false, "QQ号已被绑定");
            }
        }
        return Message.build(true);
    }

    /**
     * 注册并绑定第三方
     *
     * @param user
     * @param loginInfo
     * @param platform
     * @param chnl
     * @return
     */
    private LoginMeta registerAndBindingPhone(User user, LoginInfo loginInfo, int platform, String chnl) {
        //是否首次登录app
        boolean isFirstLogin = false;
        LoginMeta meta = new LoginMeta();
        UserInfoDTO dto;
        if (null != user) {
            if (StringUtils.isBlank(user.getEquipment())) {
                isFirstLogin = true;
            }
            //同步第三方信息(手机绑定就同步手机号，第三方绑定就同步第三方信息)
            this.syncThirdPartyInfo(user, loginInfo);

            //更新用户token
            this.updateTokenAfterBinding(user, loginInfo, platform);

            //添加用户权限信息
            this.addUserAuth(loginInfo, user.getId());

            //判断用户是否待唤醒(似乎没有卵用)
            //this.inviteRelationService.updateUserAwakenState(user.getId());

            //返回包装类
            dto = this.userService.convertUserToDTO(user);
        } else {
            //新用户第一次注册登陆
            isFirstLogin = true;
            //新增用户并绑定第三方信息
            DefaultUserWarper warper = this.addDefaultUser(loginInfo, platform, chnl);
            user = warper.user;
            dto = warper.dto;
            //设置为有效用户
//            pushRedPackageMsg(user.getId());
        }
        //保存用户最后登录信息
        userLoginHistoryService.save(user, loginInfo);

        //首次APP登录，设置用户为有效用户
        //第一次登陆绑定VIP邀请关系
        if (isFirstLogin) {
            setNewUserCache(user.getId());

            if (loginInfo.getType() == AppConst.LOGIN_TYPE.WEIXIN) {
                logger.info("用户第一次微信授权登陆-----绑定VIP邀请关系");
                //处理vip邀请信息
                dealVipActivation(user.getId(), loginInfo.getLoginName(), platform);
            }
        }

        //缓存redis
        logger.info("用户在注册登录后存入redis,实体:{}", JSON.toJSONString(dto));
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId().toString(), JSON.toJSONString(dto));

        //检测是否完成完善个人资料任务
        userService.checkCompleteInfo(dto);

        meta.setUser(dto);
        meta.setFirstLogin(isFirstLogin);

        return meta;
    }


    /**
     * 记录首次登录用户缓存
     */
    private void setNewUserCache(Long id) {
        KeyGenerator key = RedisConfig.NEW_USER_LOGIN.copy().appendKey(id);
        redisStringAdapter.set(key, true);
        redisStringAdapter.expire(key, DateUtils.getCurSeconds());
    }

    /**
     * 第三方绑定后更新用户token
     *
     * @param user
     */
    private void updateTokenAfterBinding(User user, LoginInfo loginInfo, int platform) {
        //更新用户token
//        user.setToken(this.userService.getToken());
//        user.setExpiretime(this.userService.getExpireTime());
        user.setLastLoginTime(new Date());

        this.userMapper.updateByPrimaryKeySelective(user);

        //判断当前手机号是否存在风险
        if (StringUtils.isNotEmpty(user.getPhone()) &&
                (user.getPhone().startsWith("170")) || user.getPhone().startsWith("171")) {
            this.userMapper.updateUserToRisk(user.getId());
        } else {
            if (checkRisk(user, loginInfo, platform)) {
                this.userMapper.updateUserToRisk(user.getId());
            }
        }
    }

    /**
     * 检测手机账号风险系数
     *
     * @param user
     * @param loginInfo
     * @param platform
     * @return
     */
    private Boolean checkRisk(User user, LoginInfo loginInfo, int platform) {
        String operateSource = StringUtils.EMPTY;
        if (platform == 1) {
            operateSource = "ANDROID";
        } else if (platform == 2) {
            operateSource = "IOS";
        }
        RiskParam riskParam = new RiskParam();
        riskParam.setDeviceToken(loginInfo.getDeviceToken());
        riskParam.setIp(loginInfo.getRegIp());
        riskParam.setOperateSource(operateSource);
        riskParam.setPhone(user.getPhone());
        riskParam.setEquiment(user.getEquipment());
        riskParam.setOperateSystem(loginInfo.getPhonemodel());
        return riskRecordService.checkRiskDevice(riskParam);
    }

    private DefaultUserWarper addDefaultUser(LoginInfo loginInfo, int platform, String chnl) {
        //新用户信息录入
        User user = this.userService.addUser(loginInfo, platform, chnl, AppConst.USER_STATE.NORMAL);
        UserInfoDTO dto = this.userService.convertUserToDTO(user);
        dto.setCreateTime(new Date());
        //从访客账号拷贝标签权重信息和频道列表
        this.copyUserInfo(loginInfo, user);
        //尝试绑定
        if (null != loginInfo.getInviteCode()) {
            this.inviteRelationIntegrationService.addInviteRelation(user.getId(), loginInfo.getInviteCode());
        }
        return new DefaultUserWarper(user, dto);
    }

    /**
     * 根据用户的微信uid去查找vip的邀请信息
     *
     * @param userId 用户ID
     * @param wechat 微信uid
     */
    private void dealVipActivation(Long userId, String wechat, Integer platform) {
        vipIntegrationService.saveVipViewRecord(userId, wechat, platform);
    }

    /**
     * 添加三方登录标识
     *
     * @param loginInfo
     * @param userId
     */
    private void addUserAuth(LoginInfo loginInfo, Long userId) {
        logger.info("添加用户权限认证");
        UserAuth auth = new UserAuth();
        auth.setUserId(userId);
        auth.setType(loginInfo.getType());

        //如果是手机号码绑定
        if (StringUtils.equals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getPhone());
            //将密码通过MD5加密
            if (StringUtils.isNotEmpty(loginInfo.getPassword())) {
                auth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
            }
            this.userAuthMapper.insertSelective(auth);
        }

        //第三方登录绑定
        if (StringUtils.notEquals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getLoginName());
            this.userAuthMapper.insertSelective(auth);
        }
    }

    private boolean checkParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }

        //手机号标识登录，则验证码和密码不能为空
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isEmpty(loginInfo.getPassword())
                && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

    /**
     * 校验注册数据并绑定
     *
     * @param loginInfo
     * @return
     */
    private boolean bandingAndRegisterCheckParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //验证必填地段
        if (StringUtils.isBlank(loginInfo.getLoginName())
                && StringUtils.isBlank(loginInfo.getPhone())
                && StringUtils.isBlank(loginInfo.getCode())
                && StringUtils.isBlank(loginInfo.getPassword())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

    /**
     * 同步第三方信息
     *
     * @param user
     * @param loginInfo
     */
    private void syncThirdPartyInfo(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            if (null != loginInfo.getNickname()) {
                user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
            }
            if (null != loginInfo.getHeadImg()) {
                user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
            }
        }
    }

    private class DefaultUserWarper {

        User user;

        UserInfoDTO dto;

        DefaultUserWarper(User user, UserInfoDTO dto) {
            this.user = user;
            this.dto = dto;
        }

    }

}
