package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;
import java.util.Date;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.user.config.UserProperties;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAccountMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.integration.MessageUserIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.integration.UserEventIntegrationService;
import com.bxm.localnews.user.integration.WithdrawIntegrationService;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.param.AccountGoldParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.vo.Message;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * @author zhaoyadong 2019/3/6 9:25
 * @desc
 */
@Service
public class UserAccountServiceImpl extends BaseService implements UserAccountService {

    @Resource
    private UserAccountMapper userAccountMapper;

    @Resource
    private CashFlowService cashFlowService;

    @Resource
    private UserWithdrawService userWithdrawService;

    @Resource
    private UserService userService;

    @Resource
    private InviteRecordService inviteRecordService;

    @Resource
    private UserEventIntegrationService userEventIntegrationService;

    @Resource
    private GoldFlowService goldFlowService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Resource
    private MessageUserIntegrationService messageUserIntegrationService;

    @Resource
    private UserProperties userProperties;

    @Resource
    private WithdrawIntegrationService withdrawIntegrationService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public BigDecimal getUserTotalCash(Long userId) {
        return getUserAccountDetail(userId).getTotalCash();
    }

    @Override
    public UserAccount getUserAccountDetail(Long userId) {
        logger.debug("从数据库获取用户账户信息：用户id:[{}]", userId);
        return getAndSetUserAccount(userId);
    }

    @Override
    public Message createUserAccount(Long userId) {
        logger.debug("创建用户账户信息：用户id:[{}]", userId);
        getAndSetUserAccount(userId);
        return Message.build();
    }

    @Override
    public UserAccountDTO getUserAccount(Long userId) {
        logger.debug("转换用户账户信息：从userAccount 转为userAccountDTO;用户id:[{}]", userId);
        UserAccountDTO userAccountDTO = convertUserAccount(getAndSetUserAccount(userId));

        if (!redisStringAdapter.hasKey(getUserWithdrawPop(userId))) {
            if (BigDecimal.ZERO.compareTo(userAccountDTO.getDrawablelCash().add(userAccountDTO.getNotArriveCash())) < 1) {
                redisStringAdapter.set(getUserWithdrawPop(userId), 1);
            }
            userAccountDTO.setEnablePopWithdraw(true);
        }
        return userAccountDTO;
    }

    @Override
    public UserAccountDTO getUserAccountAndEnableWithdraw(Long userId) {
        logger.debug("转换用户账户信息：从userAccount 转为userAccountDTO;用户id:[{}]", userId);
        UserAccountDTO userAccountDTO = convertUserAccount(getAndSetUserAccount(userId));

        //增加判断是否可以出现弹窗-是否在app外成功领取新用户赏金
        if (cashFlowService.enableWithdraw(userId)) {
            logger.info("用户在app外成功领取新用户赏金,可提现状态改为true,用户id为:{}", userId);
            userAccountDTO.setEnablePopWithdraw(true);
        }

        if (!redisStringAdapter.hasKey(getUserWithdrawPop(userId))) {
            //可提现金额和未到账金额两值相加大于等于0
            if (BigDecimal.ZERO.compareTo(userAccountDTO.getDrawablelCash().add(userAccountDTO.getNotArriveCash())) < 0) {
                redisStringAdapter.set(getUserWithdrawPop(userId), 1);
                userAccountDTO.setEnablePopWithdraw(true);
            }
        } else {
            userAccountDTO.setEnablePopWithdraw(false);

        }
        return userAccountDTO;
    }

    @Override
    public UserAccountDayCashDTO getUserAccountAndDetail(Long userId) {
        UserWithdrawDTO userWithdrawDTO = userWithdrawService.getUserRecentWithdrawInfo(userId);
        //判断用户最后一笔订单是否处于审核状态
        if (userWithdrawDTO == null || WithdrawEnum.SUCCESS_PAYMENT.getState().equals(userWithdrawDTO.getStatus())
                || WithdrawEnum.FAIL_PAYMENT.getState().equals(userWithdrawDTO.getStatus())) {
            userWithdrawDTO = null;
        }

        return new UserAccountDayCashDTO(getUserAccount(userId),
                cashFlowService.getRecentSevenDayCashFlowByUser(userId),
                userWithdrawDTO);
    }

    @Override
    public UserCashGoldDTO getUserCashGold(Long userId) {
        logger.debug("获取用户账户在个人中心的信息：用户id:[{}]", userId);
        UserAccount userAccount = getAndSetUserAccount(userId);
        BigDecimal cash = userAccount.getDrawablelCash().add(userAccount.getTempCash())
                .add(userAccount.getStabilizeCash());
        return new UserCashGoldDTO(cash, userAccount.getUsableGold(),
                BigDecimal.valueOf(userAccount.getUsableGold())
                        .divide(userProperties.getGoldRate(), 2, BigDecimal.ROUND_DOWN),
                userAccount.getTotalGold());
    }

    @Override
    public Boolean addCash(AccountCashParam param) {
        if (null == param || null == param.getUserId() || null == param.getCashType() ||
                null == param.getAddTotal() || null == param.getCash() || param.getCash().compareTo(BigDecimal.ZERO) == 0) {
            return false;
        }

        logger.debug("用户账户金额更新：{}", param);
        int isAdded = 0;

        if (CashEnum.TEMP_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserTempCash(param.getUserId(), param.getCash());
        } else if (CashEnum.STABILIZE_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserStabilizeCash(param.getUserId(), param.getCash());
        } else if (CashEnum.DRAWABLEL_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserDrawableCash(param.getUserId(), param.getCash());
        }

        if (isAdded > 0) {
            if (param.getAddTotal()) {
                userAccountMapper.addUserTotalCash(param.getUserId(), param.getCash());
                addUserEvent(param.getUserId(), param.getCash());
                // 不发送推送  用户红包不发
                if (!(CashFlowTypeEnum.RED_PACKET.name().equals(param.getCashFlowType())
                        || CashFlowTypeEnum.COMMAND_RED_PACKET.name().equals(param.getCashFlowType()))) {
                    pushMsgIntegrationService.pushChangeAccountMsg(param);
                }
            }

            CashFlow cashFlow = new CashFlow();
            cashFlow.setId(nextSequence());
            cashFlow.setUserId(param.getUserId());
            cashFlow.setAmount(param.getCash());
            cashFlow.setRelationId(param.getRelationId());
            cashFlow.setType(param.getCashFlowType());
            cashFlow.setConvertType(param.getConvertType());
            cashFlow.setCreateTime(new Date());
            cashFlowService.addCashFlow(cashFlow);

            return true;
        } else {
            return false;
        }

    }

    @Override
    public Boolean transferCash(AccountCashParam param) {
        logger.debug("转换的账户信息：[{}]", JSON.toJSONString(param));
        if (null == param || null == param.getUserId() || null == param.getCashType() || null == param.getCash()
                || null == param.getConvertType() || param.getCash().compareTo(BigDecimal.ZERO) <= 0) {
            return false;
        }

        logger.debug("用户账户金额转化：{}", param);
        int isTransferred = 0;

        if (CashFlowConvertTypeEnum.TEMP_TO_STABILIZE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.tempToStabilize(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.STABILIZE_TO_DRAWABLE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.stabilizeToDrawable(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.DRAWABLE_TO_WITHDRAWAL.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.drawableToWithdrawal(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.TEMP_TO_DRAWABLE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.tempToDrawable(param.getUserId(), param.getCash());
        }

        if (isTransferred > 0) {
            CashFlow cashFlow = new CashFlow();
            cashFlow.setId(nextSequence());
            cashFlow.setUserId(param.getUserId());
            cashFlow.setAmount(param.getCash());
            cashFlow.setRelationId(param.getRelationId());
            cashFlow.setType(param.getCashFlowType());
            cashFlow.setConvertType(param.getConvertType());
            cashFlow.setCreateTime(new Date());
            cashFlowService.addCashFlow(cashFlow);

            return true;
        } else {
            return false;
        }

    }

    @Override
    public Message updateUserWithdrawInfo(Long userId, BigDecimal amount, Byte state) {
        //提现成功
        if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(state)) {
            //更新用户提现赏金转为已提现赏金
            userAccountMapper.addUserWithdrawalCash(userId, amount);
            //更新邀请人的赏金为可提现赏金
            updateInviteUserCash(userId, ExtraInviteStateEnum.WITHDRAW);
        }

        //提现失败
        if (WithdrawEnum.FAIL_PAYMENT.getState().equals(state)) {
            AccountCashParam accountCashParam = new AccountCashParam(userId, CashEnum.DRAWABLEL_CASH.name(), Boolean.FALSE,
                    amount, null, CashFlowTypeEnum.FAIL_WITHDRAWAL.name(),
                    null);
            addCash(accountCashParam);
        }
        return Message.build();
    }

    @Override
    public BigDecimal getUserDrawableCash(Long userId) {
        return getUserAccount(userId).getDrawablelCash();
    }

    @Override
    public Boolean addGold(AccountGoldParam param) {
        logger.debug("用户账户更新金币的参数：[{}]", param);
        if (null == param || null == param.getUserId() || null == param.getGoldType()
                || null == param.getAddTotal() || null == param.getGold() || 0 == param.getGold()) {
            return false;
        }
        int isAdd = 0;
        logger.debug("用户账户金币更新：{}", param);
        Integer gold = param.getGold();
        if (GoldEnum.TEMP_GOLD.name().equals(param.getGoldType())) {
            isAdd = userAccountMapper.addUserTempGold(param.getUserId(), gold);
        } else if (GoldEnum.USABLE_GOLD.name().equals(param.getGoldType())) {
            UserAccount userAccount = userAccountMapper.getUserAccount(param.getUserId());
            isAdd = userAccountMapper.addUserUsableGold(param.getUserId(), gold);
            //当扣除金币数大于用户账户可用金币数时，只扣除可用金币数
            if (gold < 0 && userAccount.getUsableGold() + gold < 0) {
                gold = -userAccount.getUsableGold();
            }
        } else if (GoldEnum.CONSUME_GOLD.name().equals(param.getGoldType())) {
            isAdd = userAccountMapper.addUserConsumeGold(param.getUserId(), gold);
            gold = Math.negateExact(gold);
        }

        if (isAdd > 0) {
            if (param.getAddTotal()) {
                userAccountMapper.addUserTotalGold(param.getUserId(), gold);
            }

            GoldFlow goldFlow = new GoldFlow(nextSequence(), param.getUserId(), gold, param.getGoldFlowType(),
                    param.getRelationId(), param.getContent());
            if (StringUtils.isNotBlank(param.getReason())) {
                goldFlow.setReason(param.getReason());
            }
            goldFlowService.addGoldFlow(goldFlow);

            //文章被赞,或点赞文章,分享内容获得红花不推送
            boolean flag = StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.ARTICLE_LIKE_ADD.getName(), param.getGoldFlowType()) ||
                    StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.ARTICLE_LIKE_EXPEND.getName(), param.getGoldFlowType()) ||
                    StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.TASK_NEWS_SHARE.getName(), param.getGoldFlowType());
            if (!flag) {
                messageUserIntegrationService.addGoldMsg(goldFlow, getUserUsableGold(param.getUserId()));
            }
            return true;
        }

        return false;
    }

    @Override
    public Integer getUserUsableGold(Long userId) {
        logger.debug("获取用户[{}]的账户金币可用余额", userId);
        return getUserAccountDetail(userId).getUsableGold();
    }

    @Override
    public void userWithdrawPop(Long userId) {
        redisStringAdapter.set(getUserWithdrawPop(userId), 1);
    }

    @Override
    public Json<BigDecimal> userGoldWithdraw(Long userId, String openId, BigDecimal amount, String devcId, String ip) {
        if (userProperties.getMixWithdrawBalance().compareTo(amount) > 0 ||
                amount.compareTo(userProperties.getMaxWithdrawBalance()) > 0) {
            logger.info("当前用户[{}]提现金额[{}], 不满足提现范围", userId, amount);
            return ResultUtil.genFailedResult("请输入正确的提现金额");
        }

        Integer usableGold = getUserUsableGold(userId);
        Integer transferGold = amount.multiply(userProperties.getGoldRate()).intValue();
        if (usableGold < transferGold) {
            logger.info("当前用户[{}]可用积分[{}]，小于提现积分[{}], 不允许提现", userId, usableGold, transferGold);
            return ResultUtil.genFailedResult("余额不足");
        }

        BigDecimal withdrawAmount = userWithdrawService.getMiniAppUserWithdraw(userId, openId, WithdrawEnum.SUCCESS_PAYMENT.getState());
        if (userProperties.getTodayLimitBalance().compareTo(withdrawAmount) <= 0) {
            logger.info("用户[{}]今日提现金额已达上限", userId);
            return ResultUtil.genFailedResult("单日兑换已达限额");
        }

        AccountGoldParam param = new AccountGoldParam(userId, "CONSUME_GOLD", false,
                transferGold, null, GoldFlowTypeEnum.MINI_APP_WITHDRAW.name());
        if (addGold(param)) {
            Message message = withdrawIntegrationService.appletWithdraw(userId, openId, amount, devcId, ip);
            if (message.isSuccess()) {
                //提现成功，记录流水
                CashFlow cashFlow = new CashFlow();
                cashFlow.setId(nextSequence());
                cashFlow.setUserId(userId);
                cashFlow.setAmount(amount);
                cashFlow.setType(CashFlowTypeEnum.WECHAT_WITHDRAWAL.getName());
                cashFlow.setCreateTime(new Date());
                cashFlowService.addCashFlow(cashFlow);
                return ResultUtil.genSuccessResult(amount);
            } else {
                //提现失败---金币退回
                logger.info("用户[{}]提现失败-----金币退回", userId);
                param = new AccountGoldParam(userId, "CONSUME_GOLD", false, -transferGold,
                        null, GoldFlowTypeEnum.WITHDRAW_FAIL_RETURN.name(), null);
                addGold(param);
                return ResultUtil.genFailedResult(message.getLashMessaage());
            }
        } else {
            return ResultUtil.genFailedResult("积分扣除失败");
        }
    }

    @Override
    public UserCashInfoVo getUserCashInfo(Long userId) {
        UserCashInfoVo userCashInfoVo = new UserCashInfoVo();

        //获取用户账户金额
        userCashInfoVo.setUserId(userId);
        UserInfoDTO userInfoDTO = userService.getUserFromRedisDB(userId);
        if (null == userInfoDTO) {
            return userCashInfoVo;
        }
        userCashInfoVo.setNickname(userInfoDTO.getNickname());
        userCashInfoVo.setHeadImg(userInfoDTO.getHeadImg());
        UserAccount userAccount = userAccountMapper.getUserAccount(userId);
        if (null != userAccount) {
            userCashInfoVo.setCash(userAccount.getTotalCash());
        }

        User user = userService.selectByPrimaryKey(userId);
        if (null == user || user.getInviteUserId() == null) {
            return userCashInfoVo;
        }

        //获取邀请人账户金额
        userCashInfoVo.setInviteUserId(user.getInviteUserId());
        UserInfoDTO inviteUserInfoDTO = userService.getUserFromRedisDB(user.getInviteUserId());
        if (null == inviteUserInfoDTO) {
            return userCashInfoVo;
        }
        userCashInfoVo.setInviteUserNickname(inviteUserInfoDTO.getNickname());
        userCashInfoVo.setInviteUserHeadImg(inviteUserInfoDTO.getHeadImg());
        UserAccount inviteUserAccount = userAccountMapper.getUserAccount(user.getInviteUserId());
        if (null != inviteUserAccount) {
            userCashInfoVo.setInviteUserCash(inviteUserAccount.getTotalCash());
        }

        return userCashInfoVo;
    }

    /**
     * 构建一个用户账户信息
     *
     * @param userId
     * @return
     */
    private UserAccount generateUserAccount(Long userId) {
        UserAccount userAccount = new UserAccount(userId);
        userAccount.setId(nextSequence());
        return userAccount;
    }

    /**
     * 转换账户信息
     *
     * @param userAccount
     * @return
     */
    private UserAccountDTO convertUserAccount(UserAccount userAccount) {
        UserAccountDTO userAccountDTO = new UserAccountDTO();
        userAccountDTO.setId(userAccount.getId());
        userAccountDTO.setUserId(userAccount.getUserId());
        userAccountDTO.setDrawablelCash(userAccount.getDrawablelCash());
        //计算未到账金额---临时金额加上固化现金
        userAccountDTO.setNotArriveCash(userAccount.getTempCash().add(userAccount.getStabilizeCash()));
        userAccountDTO.setTotalCash(userAccount.getTotalCash());

        return userAccountDTO;
    }

    /**
     * 获取用户账户信息（不存在则创建）
     *
     * @param userId
     * @return
     */
    private UserAccount getAndSetUserAccount(Long userId) {
        UserAccount userAccount = this.userAccountMapper.getUserAccount(userId);
        if (null == userAccount) {
            logger.warn("用户[{}]账户信息不存在", userId);
            userAccount = generateUserAccount(userId);
            this.userAccountMapper.addUserAccount(userAccount);
        } else {
            BigDecimal usableGold = BigDecimal.valueOf(userAccount.getUsableGold());
            if (usableGold != null && usableGold.compareTo(BigDecimal.ZERO) < 0) {
                userAccount.setUsableGold(0);
            }
        }

        return userAccount;
    }

    /**
     * 添加用户账户变化事件
     */
    private void addUserEvent(Long userId, BigDecimal amount) {
        UserCashGoldDTO userCashGoldDTO = getUserCashGold(userId);
        PushPayloadInfo pushPayloadInfo = PushPayloadInfo.build(PushMessageEnum.USER_ACCOUNT_ENVET)
                .addExtend("userId", userId).addExtend("current", userCashGoldDTO.getCash())
                .addExtend("val", amount);
        userEventIntegrationService.addUserEvent(pushPayloadInfo);
    }

    @Override
    public void updateInviteUserCash(Long userId, ExtraInviteStateEnum currentStateEnum) {
        User user = userService.selectByPrimaryKey(userId);
        if (null == user.getInviteUserId()) {
            logger.debug("用户[{}]的邀请人为空", userId);
            return;
        }

        InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), userId);
        if (null == inviteRecord) {
            logger.debug("用户[{}]与邀请人[{}]的关系无效", userId, user.getInviteUserId());
            return;
        }

        if (!InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
            logger.debug("当前邀请关系不是处于登录app状态----不进行更新操作");
            return;
        }

        Boolean changeInviteUserCashState = false;

        //只有面对面邀请需要同时判断受邀人是否提现且第二天登录
        if (InviteTypeEnum.FACEINVITE.getName().equals(inviteRecord.getType())) {
            Integer inviteState = redisHashMapAdapter.get(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), Integer.class);
            if (null != inviteState) {
                if (ExtraInviteStateEnum.WITHDRAW.equals(currentStateEnum) && ExtraInviteStateEnum.LOGIN_TWICE.getStatus().equals(inviteState)
                        || ExtraInviteStateEnum.LOGIN_TWICE.equals(currentStateEnum) && ExtraInviteStateEnum.WITHDRAW.getStatus().equals(inviteState)) {
                    changeInviteUserCashState = true;
                    redisHashMapAdapter.remove(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString());
                }
            } else {
                redisHashMapAdapter.put(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), currentStateEnum.getStatus());
            }
        } else {
            changeInviteUserCashState = true;
        }

        if (changeInviteUserCashState) {
            //更新邀请关系
            logger.info("邀请人[{}]增加可提现赏金[{}]", user.getInviteUserId(), inviteRecord.getAward());
            inviteRecord.setInviteState(InviteRecordStateEnum.SUCCESS_WITHDRAW.getName());
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecordService.updateInviteRecord(inviteRecord);

            //转换邀请人金额
            AccountCashParam accountCashParam = new AccountCashParam(inviteRecord.getUserId(), CashEnum.DRAWABLEL_CASH.name(), Boolean.FALSE,
                    inviteRecord.getAward(), inviteRecord.getId(), CashFlowTypeEnum.PROMOTE_BOUNTY.name(), CashFlowConvertTypeEnum.STABILIZE_TO_DRAWABLE.name());
            transferCash(accountCashParam);
        }
    }

    private KeyGenerator getUserWithdrawPop(Long userId) {
        return RedisConfig.USER_WITHDRAW_POP.copy().appendKey(userId);
    }
}
