package com.bxm.localnews.user.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.activity.service.InviteRelationService;
import com.bxm.localnews.activity.service.VipFacadeService;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.common.constant.AppConst;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.feign.SmsSupplyFeignService;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.user.config.UserProperties;
import com.bxm.localnews.user.domain.RegSourceMapper;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.service.LoginService;
import com.bxm.localnews.user.service.UserAuthService;
import com.bxm.localnews.user.service.UserLoginHistoryService;
import com.bxm.localnews.user.service.UserService;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.LoginMeta;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserAuth;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;

import static com.bxm.localnews.common.constant.AppConst.LOGIN_TYPE.*;

/**
 * Created by mars on 2018/2/23.
 * 登录
 */
@Service("loginService")
@RefreshScope
public class LoginServiceImpl implements LoginService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Resource
    private SmsSupplyFeignService smsSupplyFeignService;

    @Resource
    private UserAuthMapper userAuthMapper;

    @Resource
    private UserMapper userMapper;

    @Resource
    private InviteRelationService inviteRelationService;

    @Resource
    private UserService userService;

    @Resource
    private UserTagService userTagService;

    @Resource
    private NewsRecommendedService newsRecommendedService;

    @Resource
    private UserKindService userKindService;

    @Resource
    private NewsReplyService newsReplyService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private RegSourceMapper regSourceMapper;

    @Resource
    private AppVersionSupplyService appVersionSupplyService;

    @Resource
    private UserProperties userProperties;

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private BizLogService bizLogService;

    @Resource
    private InitTagService initTagService;

    @Resource
    private NewsKindService newsKindService;

    @Resource
    private VipFacadeService vipFacadeService;

    @Resource
    private UserLoginHistoryService userLoginHistoryService;

    @Resource
    private VideoReplyFacadeService videoReplyFacadeService;

    @Override
    public Json<LoginMeta> doLogin(LoginInfo loginInfo, BasicParam basicParam ) {
        boolean flag = false;

        //ios提测时用的测试账号
        if (this.appVersionSupplyService.inWhiteList(loginInfo.getLoginName())) {
            flag = true;
            loginInfo.setCode("123456");
        }

        //判断参数是否正确
        if (!this.checkParam(loginInfo)) {
            this.logger.error("doLogin failed:" + loginInfo);
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }

        UserAuth userAuth = this.userAuthMapper.selectByUserAuth(loginInfo.getType(), loginInfo.getLoginName());

        //第三方登录,如果未绑定手机号码，提示进行绑定
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()) {
            if (null == userAuth) {
                if (getPublishState(basicParam)) {
                    LoginMeta meta = new LoginMeta();
                    meta.setCode(RespCode.BIND_PHONE);
                    return ResultUtil.genSuccessResult(meta);
                }else{
                  return registerAndBindingPhone(loginInfo,basicParam.getPlatform(),null,basicParam.getChnl());
                }
            }
        }

        //找回密码
        if (checkRetrievePwd(loginInfo)) {
            Message message = retrievePwd(loginInfo, userAuth, flag);
            if (!message.isSuccess()) {
                return ResultUtil.genFailedResult(message.getMessages().get(0));
            }
        }

        //密码登录
        if (checkVerifyPwd(loginInfo)) {
            Message message = verifyPwd(loginInfo.getPassword(), userAuth);
            if (!message.isSuccess()) {
                return ResultUtil.genFailedResult(message.getLashMessaage());
            }
        }

        //更新用户信息
        LoginMeta meta = updateUserAfterLogin(userAuth, loginInfo);

        if (null != meta && meta.getUser() != null) {
            bizLogService.login(meta.getUser().getId());
        }

        return ResultUtil.genSuccessResult(meta);
    }

    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionSupplyService.getPublishState(basicParam);
    }


    /**
     * 判断是否是密码登录
     *
     * @param loginInfo
     * @return
     */
    private boolean checkVerifyPwd(LoginInfo loginInfo) {
        return AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isNotBlank(loginInfo.getPassword())
                && !loginInfo.getResetPwd();
    }

    /**
     * 判断是否是找回密码
     *
     * @param loginInfo
     * @return
     */
    private boolean checkRetrievePwd(LoginInfo loginInfo) {
        return AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isNotBlank(loginInfo.getPassword())
                && null != loginInfo.getResetPwd()
                && loginInfo.getResetPwd();
    }

    /**
     * 找回密码进行的操作
     *
     * @param loginInfo
     * @param userAuth
     * @return
     */
    private Message retrievePwd(LoginInfo loginInfo, UserAuth userAuth, boolean flag) {
        if (null == userAuth) {
            return Message.build(false, "手机号码未注册，请前往注册");
        }

        if (!flag && !this.smsSupplyFeignService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_RETRIEVE_PWD, loginInfo.getLoginName(), loginInfo.getCode())) {
            return Message.build(false, "请输入正确的验证码");
        }

        String encodePwd = MD5Util.hgmd5(loginInfo.getPassword());
        userAuth.setCredential(encodePwd);
        this.userAuthMapper.updateByPrimaryKeySelective(userAuth);
        return Message.build(true);
    }

    /**
     * 验证账号密码登录
     *
     * @param password
     * @param userAuth
     * @return
     */
    private Message verifyPwd(String password, UserAuth userAuth) {
        String encodePwd = MD5Util.hgmd5(password);
        if (null == userAuth) {
            //若用户账号不存在
            return Message.build(false, "手机号码未注册，请前往注册");
        }
        if (!userAuth.getCredential().equals(encodePwd)) {
            //密码不一致则直接返回
            return Message.build(false, "账号或密码错误");
        }
        return Message.build(true);
    }

    /**
     * 在登录之后需要对用户信息进行更新
     *
     * @param userAuth
     * @param loginInfo
     * @return
     */
    private LoginMeta updateUserAfterLogin(UserAuth userAuth, LoginInfo loginInfo) {
        boolean isFirstLogin;
        User user = this.userMapper.selectByPrimaryKey(userAuth.getUserId());
        isFirstLogin = StringUtils.isBlank(user.getEquipment());

        //如果是微信登录，更新一次同步微信头像，昵称
        this.updateNicknameAndHeadImg(loginInfo, user);

        //如果是在站外H5微信授权登录，更新一次同步微信头像，昵称，并且在auth中插入openId和unionId
        this.updateWechat(loginInfo, user);

        //保存用户登录信息
        this.userLoginHistoryService.save(user,loginInfo);

        //更新用户位置，跟随静默用户的位置
        this.callAsyncUpdateLocation(loginInfo, user);

        //得到登录用户的包装类
        UserInfoDTO dto = this.getUpdateUser(loginInfo, user);

        //返回给客户端包装类
        LoginMeta meta = new LoginMeta(dto, isFirstLogin);

        //首次APP登录，设置用户为有效用户
        if (isFirstLogin) {
            this.pushRedPackageMsg(user.getId());
        }

        //缓存redis
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userAuth.getUserId().toString(), JSON.toJSONString(meta.getUser()));
        return meta;
    }

    /**
     * 得到更新后的用户包装类
     *
     * @param loginInfo
     * @param user
     * @return
     */
    private UserInfoDTO getUpdateUser(LoginInfo loginInfo, User user) {
        if (StringUtils.isNotBlank(loginInfo.getEquipment())) {
            user.setEquipment(loginInfo.getEquipment());
        }
        //更新用户token
        user.setToken(this.userService.getToken());
        user.setExpiretime(this.userService.getExpireTime());
        user.setRefreshtoken(user.getRefreshtoken());
        user.setLastLoginTime(new Date());
        user.setLastLoginIp(loginInfo.getLastLoginIp());
        this.userMapper.updateByPrimaryKeySelective(user);
        return this.userService.getUserToDTO(user);
    }

    /**
     * 如果是微信登录，更新一次同步微信头像，昵称
     *
     * @param loginInfo
     * @param user
     */
    private void updateNicknameAndHeadImg(LoginInfo loginInfo, User user) {
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {

            if (isTempNickName(loginInfo,user) || isTempHeadImg(loginInfo,user)) {
                if (isTempNickName(loginInfo,user)) {
                    user.setNickname(loginInfo.getNickname());
                }

                if (isTempHeadImg(loginInfo,user)) {
                    user.setHeadImg(loginInfo.getHeadImg());
                }

                this.newsReplyService.triggerUpdateInfo(user.getId(), user.getNickname(), user.getHeadImg());
                this.videoReplyFacadeService.triggerUpdateInfo(user.getId(), user.getNickname(), user.getHeadImg());
            }


        }
    }

    private boolean isTempNickName(LoginInfo loginInfo, User user){
        return user.getIsTempNickName()==1&&null != loginInfo.getNickname();
    }

    private boolean isTempHeadImg(LoginInfo loginInfo, User user){
        return user.getIsTempHeadImg()==1&&null != loginInfo.getHeadImg();
    }

    /**
     * 如果是在站外H5微信授权登录,则更新openId和unionId和昵称头像
     *
     * @param loginInfo
     * @param user
     */
    private void updateWechat(LoginInfo loginInfo, User user) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            if (null != loginInfo.getOpenId() && null != loginInfo.getUnionId()) {
                UserAuth userAuth = userAuthMapper.selectByUserId(WEIXIN, user.getId());
                //用户没有绑定第三方，则绑定第三方，并更新用户昵称和头像
                if (null == userAuth) {
                    Message isBindedMsg = checkBindExist(WEIXIN, loginInfo.getUnionId());
                    if (isBindedMsg.isSuccess()) {
                        user.setWeixin(loginInfo.getUnionId());
                        this.userAuthService.addUserAuth(WEIXIN, user.getId(), loginInfo.getUnionId(), null);
                        this.userAuthService.addUserAuth(WEIXIN_OPENID, user.getId(), loginInfo.getOpenId(), null);
                        if (null != loginInfo.getNickname()) {
                            user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
                        }
                        if (null != loginInfo.getHeadImg()) {
                            user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
                        }
                        //同步微信
                        this.newsReplyService.triggerUpdateInfo(user.getId(), user.getNickname(), user.getHeadImg());
                    }
                }

                //用户绑定第三方，则判断第三方验证是否相等，是的话判断openId是否存在，如果不存在则更新用户的openId
                if (null != userAuth) {
                    if (loginInfo.getUnionId().equals(userAuth.getIdentifier())) {
                        if (null == userAuthMapper.selectByUserId(WEIXIN_OPENID, user.getId())) {
                            this.userAuthService.addUserAuth(WEIXIN_OPENID, user.getId(), loginInfo.getOpenId(), null);
                        }
                    }

                }
            }
        }
    }


    private void callAsyncUpdateLocation(LoginInfo loginInfo, User user) {
        LoginService loginService = SpringContextHolder.getBean(LoginService.class);
        loginService.updateLocation(loginInfo,user);
    }

    @Override
    @Async
    public void updateLocation(LoginInfo loginInfo, User user) {
        if (null != loginInfo.getId()) {
            User originalUser = userMapper.selectByPrimaryKey(loginInfo.getId());
            if (null != originalUser && null != originalUser.getLocationCode() && null != originalUser.getLocationName()) {
                user.setLocationCode(originalUser.getLocationCode());
                user.setLocationName(originalUser.getLocationName());
            }
        }
    }

    /**
     * 设置为有效用户
     *
     * @param userId
     */
    private void pushRedPackageMsg(Long userId) {
        this.regSourceMapper.effective(userId);

    }

    private void copyUserInfo(LoginInfo loginInfo, User user) {
        Date copyStartDate = DateUtils.addField(new Date(), Calendar.DATE, -1);
        // 从访客账号拷贝标签权重信息和频道列表
        if (loginInfo.getId() == null || loginInfo.getId() == 0L) {
            this.logger.warn(LogMarker.BIZ, "用户注册，但是没有提供静默注册ID[H5站外注册例外]，loginInfo:[{}],user:[{}]", loginInfo, user);
            this.newsKindService.createUserDefaultKinds(user.getId());
            this.initTagService.createInitTags(user.getId());
        } else {
            this.userTagService.doCopyTags(loginInfo.getId(), user.getId(), copyStartDate);
            this.newsRecommendedService.copyRecommendeds(loginInfo.getId(), user.getId(), copyStartDate);
            this.userKindService.copyKinds(loginInfo.getId(), user.getId());
        }
    }



    @Override
    public Json<LoginMeta> registerAndBindingPhone(LoginInfo loginInfo, int platform,Long userId, String chnl) {
        if (!this.bandingAndRegisterCheckParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }

        //auth层面的验证  根据登录类型和登录标识去auth表里查 看看有没有被别人绑定
        //1.手机号注册：根据type=1和手机号去auth表里查是否#手机号#有被绑定的记录
        //2.第三方注册：根据type=3和微信unionId去auth表里查是否#微信#有被绑定的记录
        Message messageByCheckAuth = checkBindExist(loginInfo.getType(), loginInfo.getLoginName());
        if (!messageByCheckAuth.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth.getLashMessaage());
        }

        //user层面的验证  根据手机号去user表里查是否有相同手机号的绑定，如果有第三方绑定的话则视为为已有用户绑定第三方
        User user = getUserByType(loginInfo);
        Message messageByCheckUser = checkRepeatBind(user,loginInfo);
        if (!messageByCheckUser.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckUser.getLashMessaage());
        }

        //用户绑定手机号时
        if (null != userId) {
            user = userMapper.selectByPrimaryKey(userId);
            if (null != user && 0 == user.getState()) {
                user = null;
            }
        }

        //手机号绑定需要验证码校验
        if (PHONE == loginInfo.getType()) {
            if (StringUtils.isNotBlank(loginInfo.getCode())
                    && !this.smsSupplyFeignService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE, loginInfo.getPhone(), loginInfo.getCode())) {
                return ResultUtil.genFailedResult("请输入正确的验证码");
            }
        }

        //如果用户存在则进行绑定，如果用户不存在则添加用户并绑定第三方
        LoginMeta meta = registerAndBindingPhone(user, loginInfo, platform, chnl);

        return ResultUtil.genSuccessResult(meta);
    }

    private User getUserByType(LoginInfo loginInfo){
        User user = null;
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user = this.userMapper.findByWeixin(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user = this.userMapper.findByPhone(loginInfo.getPhone());
        }
        return user;
    }

    private Message checkRepeatBind(User user, LoginInfo loginInfo) {
        if (null != user) {
            if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()&& StringUtils.isNotEmpty(user.getPhone())) {
                return Message.build(false, "手机号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()&& StringUtils.isNotEmpty(user.getWeixin())) {
                return Message.build(false, "微信号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()&& StringUtils.isNotEmpty(user.getWeibo())) {
                return Message.build(false, "微博已被绑定");
            } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()&& StringUtils.isNotEmpty(user.getQq())) {
                return Message.build(false, "QQ号已被绑定");
            }
        }
        return Message.build(true);
    }

    private Message checkBindExist(Byte loginType, String loginName) {
        UserAuth auth = this.userAuthMapper.selectByUserAuth(loginType, loginName);
        if (null != auth) {
            if (AppConst.LOGIN_TYPE.PHONE == loginType) {
                return Message.build(false, "手机号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginType) {
                return Message.build(false, "微信号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginType) {
                return Message.build(false, "微博已被绑定");
            } else if (AppConst.LOGIN_TYPE.QQ == loginType) {
                return Message.build(false, "QQ号已被绑定");
            }
        }
        return Message.build(true);
    }

    /**
     * 注册并绑定第三方
     *
     * @param user
     * @param loginInfo
     * @param platform
     * @param chnl
     * @return
     */
    private LoginMeta registerAndBindingPhone(User user, LoginInfo loginInfo, int platform, String chnl) {
        //是否首次登录app
        boolean isFirstLogin = false;
        LoginMeta meta = new LoginMeta();
        UserInfoDTO dto;
        if (null != user) {
            if (StringUtils.isBlank(user.getEquipment())) {
                isFirstLogin = true;
            }
            //同步第三方信息(手机绑定就同步手机号，第三方绑定就同步第三方信息)
            this.syncThirdPartyInfo(user, loginInfo);

            //更新用户token
            this.updateTokenAfterBinding(user);

            //添加用户权限信息
            this.addUserAuth(loginInfo, user.getId());

            //判断用户是否待唤醒(似乎没有卵用)
            //this.inviteRelationService.updateUserAwakenState(user.getId());

            //返回包装类
            dto = this.userService.getUserToDTO(user);
        } else {
            //新用户第一次注册登陆
            isFirstLogin = true;
            //新增用户并绑定第三方信息
            DefaultUserWarper warper = this.addDefaultUser(loginInfo, platform, chnl);
            //处理vip邀请信息
            if (loginInfo.getType() == AppConst.LOGIN_TYPE.WEIXIN) {
                dealVipActivation(warper.user.getId(), loginInfo.getLoginName());
            }
            user = warper.user;
            dto = warper.dto;
            //设置为有效用户
            pushRedPackageMsg(user.getId());

        }
        //保存用户最后登录信息
        userLoginHistoryService.save(user,loginInfo);

        //缓存redis
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId().toString(), JSON.toJSONString(dto));

        meta.setUser(dto);
        meta.setFirstLogin(isFirstLogin);

        return meta;
    }

    /**
     * 第三方绑定后更新用户token
     *
     * @param user
     */
    private void updateTokenAfterBinding(User user) {
        //更新用户token
//        user.setToken(this.userService.getToken());
//        user.setExpiretime(this.userService.getExpireTime());
        user.setLastLoginTime(new Date());

        this.userMapper.updateByPrimaryKeySelective(user);
    }

    private DefaultUserWarper addDefaultUser(LoginInfo loginInfo, int platform, String chnl) {
        //新用户信息录入
        User user = this.userService.addUser(loginInfo, platform, chnl, AppConst.USER_STATE.NORMAL);
        UserInfoDTO dto = this.userService.getUserToDTO(user);
        dto.setCreateTime(new Date());
        //从访客账号拷贝标签权重信息和频道列表
        this.copyUserInfo(loginInfo, user);
        //注册成功
        this.userService.getRegisterReward(user.getId());
        //尝试绑定
        if (null != loginInfo.getInviteCode()) {
            this.inviteRelationService.addInviteRelation(user.getId(), loginInfo.getInviteCode());
        }
        return new DefaultUserWarper(user, dto);
    }

    /**
     * 根据用户的微信uid去查找vip的邀请信息
     *
     * @param userId 用户ID
     * @param wechat 微信uid
     */
    private void dealVipActivation(Long userId, String wechat) {
        vipFacadeService.saveVipViewRecord(userId, wechat);
    }

    /**
     * 添加三方登录标识
     *
     * @param loginInfo
     * @param userId
     */
    private void addUserAuth(LoginInfo loginInfo, Long userId) {
        UserAuth auth = new UserAuth();
        auth.setUserId(userId);
        auth.setType(loginInfo.getType());

        //如果是手机号码绑定
        if (StringUtils.equals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getPhone());
            //将密码通过MD5加密
            if (StringUtils.isNotEmpty(loginInfo.getPassword())) {
                auth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
            }
            this.userAuthMapper.insertSelective(auth);
        }

        //第三方登录绑定
        if (StringUtils.notEquals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getLoginName());
            this.userAuthMapper.insertSelective(auth);
        }
    }

    private boolean checkParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }

        //手机号标识登录，则验证码和密码不能为空
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isEmpty(loginInfo.getPassword())
                && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

    /**
     * 校验注册数据并绑定
     *
     * @param loginInfo
     * @return
     */
    private boolean bandingAndRegisterCheckParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //验证必填地段
        if (StringUtils.isBlank(loginInfo.getLoginName())
                && StringUtils.isBlank(loginInfo.getPhone())
                && StringUtils.isBlank(loginInfo.getCode())
                && StringUtils.isBlank(loginInfo.getPassword())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

    /**
     * 同步第三方信息
     *
     * @param user
     * @param loginInfo
     */
    private void syncThirdPartyInfo(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            if (null != loginInfo.getNickname()) {
                user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
            }
            if (null != loginInfo.getHeadImg()) {
                user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
            }
        }
    }

    private class DefaultUserWarper {

        User user;

        UserInfoDTO dto;

        DefaultUserWarper(User user, UserInfoDTO dto) {
            this.user = user;
            this.dto = dto;
        }

    }

}
