package com.bxm.localnews.user.service.impl;

import java.util.*;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.activity.service.InviteRelationService;
import com.bxm.localnews.activity.service.MissionService;
import com.bxm.localnews.activity.service.NoviceTaskRecordService;
import com.bxm.localnews.base.dto.LocationDTO;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.base.service.ChannelSupplyService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.*;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.feign.SmsSupplyFeignService;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.user.config.UserProperties;
import com.bxm.localnews.user.constant.CommonConstant;
import com.bxm.localnews.user.domain.RegSourceMapper;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserLocationHistoryMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.param.PushParam;
import com.bxm.localnews.user.param.UserParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.*;
import com.bxm.newidea.component.vo.Message;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import static com.bxm.localnews.common.constant.AppConst.*;
import static com.bxm.localnews.common.constant.TaskEnum.TASK_IMPROVE_USER_INFO;
import static com.bxm.localnews.common.constant.TaskEnum.TASK_INVITED_FRIEND;

@Service("userService")
@RefreshScope
public class UserServiceImpl extends BaseService implements UserService, UserInternalService {

    private final static Logger log = LoggerFactory.getLogger(UserServiceImpl.class);

    private final static long MAX_HEAD_IMAGE_SIZE = 1024 * 1024 * 5;

    private static final int AREA_CODE_LENGTH = 12;

    @Resource
    private SmsSupplyFeignService smsSupplyFeignService;

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private NoviceTaskRecordService noviceTaskRecordService;

    @Resource
    private PayFlowService payFlowService;

    @Resource
    private InitTagService initTagService;

    @Resource
    private NewsKindService newsKindService;

    @Resource
    private UserAmountService userAmountService;

    @Resource
    private UserMapper userMapper;

    @Resource
    private UserAuthMapper userAuthMapper;

    @Resource
    private UserProperties userProperties;

    @Resource
    private NewsRecommendedService newsRecommendedService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private RegSourceMapper regSourceMapper;

    @Resource
    private UserLocationHistoryMapper userLocationHistoryMapper;

    @Resource
    private LocationFacadeService locationService;

    @Resource
    private AliyunOSSService aliyunOSSService;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private ChannelSupplyService channelSupplyService;

    @Resource
    private MissionService missionService;

    @Resource
    private NewsReplyService newsReplyService;

    @Resource
    private VideoReplyFacadeService videoReplyFacadeService;

    @Resource
    private InviteRelationService inviteRelationService;

    @Resource
    private BizLogService bizLogService;

    @Resource
    private UserLoginHistoryService userLoginHistoryService;

    @Override
    public UserInfoDTO getUserFromRedisDB(long userId) {
        UserInfoDTO dto = this.getUserRedis(userId);
        if (null == dto) {
            dto = this.loadUserToRedis(userId);
        }
        return dto;
    }

    /**
     * 从redis中获取用户信息
     *
     * @param userId
     * @return
     */
    private UserInfoDTO getUserRedis(long userId) {
        String userJson = this.redisHashMapAdapter.get(RedisConfig.USER_INFO, userId + "", String.class);
        if (StringUtils.isNotBlank(userJson)) {
            try {
                return JSON.parseObject(userJson, UserInfoDTO.class);
            } catch (Exception e) {
                log.error("redis user parseObject error", e);
            }
        }
        return null;
    }

    /**
     * 从数据库中查找用户信息，并更新至redis中
     *
     * @param userId
     * @return
     */
    private UserInfoDTO loadUserToRedis(Long userId) {
        User user = this.userMapper.selectByPrimaryKey(userId);
        if (user != null) {
            UserInfoDTO dto = this.getUserToDTO(user);
            this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
            return dto;
        }
        return null;
    }

    @Override
    public User getUserByPhone(Long userId) {
        return this.userMapper.getPhone(userId);
    }

    @Override
    public User selectByPrimaryKey(Long userId) {
        return this.userMapper.selectByPrimaryKey(userId);
    }

    @Override
    public CoordinateDTO getUserCoordinate(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (user != null) {
            String adcode = user.getLocationCode();
            if (null != adcode) {
                LocationDTO locationDTO = locationService.getLocationByCode(adcode);
                if (locationDTO != null) {
                    CoordinateDTO coordinateDTO = new CoordinateDTO();
                    coordinateDTO.setLatitude(locationDTO.getLat());
                    coordinateDTO.setLongitude(locationDTO.getLng());
                    return coordinateDTO;
                }
            } else {
                //返回默认定位经纬度
                User newUser = new User();
                newUser.setLocationCode(bizConfigProperties.getLocationCode());
                newUser.setLocationName(bizConfigProperties.getLocationName());
                newUser.setId(userId);
                this.userMapper.updateByPrimaryKeySelective(user);
            }
        }
        return null;
    }

    @Override
    public UserBaseInfoWarper getUserBaseInfo(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        UserBaseInfoWarper userBaseInfoWarper = new UserBaseInfoWarper();
        userBaseInfoWarper.setUserId(user.getId());
        userBaseInfoWarper.setUserImg(user.getHeadImg());
        userBaseInfoWarper.setUserName(StringUtils.isEmpty(user.getNickname()) ? StringUtils.hideMobile(user.getPhone()) : user.getNickname());
        return userBaseInfoWarper;
    }

    @Override
    public String getLocationCode(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (user != null) {
            String adcode = user.getLocationCode();
            if (null != adcode) {
                return adcode;
            } else {
                return bizConfigProperties.getLocationCode();
            }
        }
        return null;
    }

    @Override
    public LocationDetailDTO getLocationByUser(Long userId) {
        LocationDetailDTO locationDetailDTO = null;
        User user = userMapper.selectByPrimaryKey(userId);
        if (user != null) {
            String adcode = user.getLocationCode();
            if (null != adcode) {
                locationDetailDTO = new LocationDetailDTO();
                BeanUtils.copyProperties(locationService.getLocationDetailByCode(adcode), locationDetailDTO);

            } else {
                locationDetailDTO = new LocationDetailDTO();
                BeanUtils.copyProperties(locationService.getLocationDetailByCode(bizConfigProperties.getLocationCode()), locationDetailDTO);
            }
        }
        return locationDetailDTO;
    }

    @Override
    public boolean checkUserExistByPhone(String phone, Integer flag) {
        User user = userMapper.findByPhone(phone);
        if (null == user) {
            return false;
        } else {
            UserAuth userAuth = userAuthMapper.selectByUserId((byte) 3, user.getId());
            if (flag == 1) {
                if (userAuth == null) {
                    return false;
                } else {
                    return true;
                }
            } else {
                return true;
            }
        }
    }

    @Override
    public Boolean checkUnionIdIsUser(String unionId) {
        return userAuthMapper.selectByUserAuth((byte) 3, unionId) == null ? false : true;

    }

    @Override
    public String getDefaultHeadImgUrl() {
        return userProperties.getDefaultHeadImgUrl();
    }

    @Override
    public UserInfoDTO getUserToDTO(User user) {
        UserInfoDTO dto = new UserInfoDTO();
        BeanUtils.copyProperties(user, dto);
        if (StringUtils.isBlank(dto.getNickname())) {
            dto.setNickname(StringUtils.hideMobile(user.getPhone()));
        }
        if (StringUtils.isBlank(dto.getHeadImg())) {
            dto.setHeadImg(userProperties.getDefaultHeadImgUrl());
        }
        return dto;
    }

    @Override
    public String getToken() {
        Long currentTime = System.currentTimeMillis();
        Random random = new Random();
        String appkey = userProperties.getAppKey();
        return MD5Util.hgmd5(currentTime + appkey + random.nextInt(1000));
    }

    @Override
    public Long getExpireTime() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, 30);
        return calendar.getTime().getTime();
    }

    @Override
    public Json<UserInfoDTO> refreshToken(Long userId, String refreshToken) {
        User user = this.userMapper.getUserByRefreshToken(userId, refreshToken);
        if (null == user) {
            return ResultUtil.genFailedResult(RespCode.UNAUTHORIZED, "登录失效");
        }
        user.setExpiretime(this.getExpireTime());
        user.setToken(this.getToken());
        this.userMapper.updateByPrimaryKeySelective(user);
        UserInfoDTO dto = this.getUserToDTO(user);
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
        return ResultUtil.genSuccessResult(dto);
    }

    @Override
    public Json<LoginMeta> register(LoginInfo loginInfo, BasicParam basicParam, String firstOpenType) {
        if (StringUtils.isEmpty(loginInfo.getEquipment())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少参数");
        }
        // 根据设备号查询
        User user = this.userMapper.findByEquipment(loginInfo.getEquipment());

        if (null == user) {
            //如果为空，则创建一条游客用户
            user = this.addUser(loginInfo, basicParam.getPlatform(), basicParam.getChnl(), AppConst.USER_STATE.VISITOR);
            //初始化用户默认分类
            //如果是IOS且处于提包状态则添加定制的默认频道
            if (2 == basicParam.getPlatform()) {
                this.newsKindService.createUserDefaultKindsForPublish(user.getId(), basicParam);
            } else {
                this.newsKindService.createUserDefaultKinds(user.getId());
            }

            //保存用户最后登录信息
            userLoginHistoryService.save(user, loginInfo);

            //初始化用户标签(重新计算标签逻辑转移到新闻推荐时执行)
            this.initTag(user.getId());
        }
        //首次登陆打开，清空用户去重的新闻
        if (CommonConstant.FIRST_OPEN_TYPE.equals(firstOpenType)) {
            this.newsRecommendedService.deleteByUserId(user.getId());
        }
        UserInfoDTO dto = this.getUserToDTO(user);
        //缓存redis
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId().toString(), JSON.toJSONString(dto));
        return ResultUtil.genSuccessResult(new LoginMeta(dto));
    }

    void initTag(Long userId) {
        this.initTagService.createInitTags(userId);
    }

    @Override
    public Json<User> doH5Register(LoginInfo loginInfo, Integer platform, String chnl) {
        if (!checkH5RegisterParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少参数");
        }

        if (!checkSmsCode(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "请输入正确的验证码");
        }

        String messageByCheckAuth = checkBindExist(loginInfo);
        if (StringUtils.isNotBlank(messageByCheckAuth)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth);
        }

        User user = this.userMapper.findByPhone(loginInfo.getLoginName());

        if (null != user) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "手机号已注册");
        }

        user = addUser(loginInfo, platform, chnl, AppConst.USER_STATE.NORMAL);

        if (user.getId() > 0) {
            if (null != loginInfo.getInviteCode()) {
                UserInfoDTO dto = this.getUserFromRedisDB(loginInfo.getInviteCode());
                if (null != dto) {
                    this.inviteRelationService.addInviteRelation(user.getId(), loginInfo.getInviteCode());
                }
            }
            //初始化用户标签(重新计算标签逻辑转移到新闻推荐时执行)
            this.initTag(user.getId());
            //初始化用户默认分类
            this.newsKindService.createUserDefaultKinds(user.getId());
            //注册成功，派发奖励
            this.getRegisterReward(user.getId());
            return ResultUtil.genSuccessResult(user);
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "注册失败");
    }

    /**
     * 检查验证码是否合理
     *
     * @param loginInfo
     * @return
     */
    private boolean checkSmsCode(LoginInfo loginInfo) {
        if (!this.smsSupplyFeignService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_REGISTER, loginInfo.getLoginName(), loginInfo.getCode())) {
            this.logger.debug("checkVcode failed,loginInfo[{}]", loginInfo);
            return false;
        }
        return true;
    }

    /**
     * 检查H5注册参数
     *
     * @param loginInfo
     * @return
     */
    private boolean checkH5RegisterParam(LoginInfo loginInfo) {
        if (StringUtils.isBlank(loginInfo.getLoginName())
                || AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                || StringUtils.isBlank(loginInfo.getCode())) {
            return false;
        }
        return true;
    }

    @Override
    public User addUser(LoginInfo loginInfo, Integer platform, String chnl, Byte userType) {
        //新增用户信息
        User user = this.saveUser(loginInfo, platform, chnl, userType);
        boolean notVisitor = AppConst.USER_STATE.VISITOR != userType;
        //判断用户是否是游客
        if (notVisitor) {
            //添加授权登录认证信息
            this.saveAuths(user.getId(), loginInfo);
            //生成计划新手计划
            this.noviceTaskRecordService.batchAdd(user.getId());
            //生成账户(外部模块)
            this.userAmountService.createAccount(user.getId());
            //如果是邀请注册,则添加注册来源
            this.saveRegSourceInfo(loginInfo, user, chnl);
            //如果是纸巾机渠道，则获取纸巾机奖励(如果不是纸巾机渠道则获得100金币)
            //this.getTissuePaperReward(user.getId(), chnl);

            bizLogService.newUser(user.getId(), chnl, loginInfo.getLoginName());
        }

        return user;
    }

    /**
     * 新用户获得纸巾机奖励
     *
     * @param userId
     * @param chnl
     */
    private void getTissuePaperReward(Long userId, String chnl) {
        //纸巾机下载注册不享受新用户100金币
        if (!UserRegChannelConstant.MACHINE.equals(chnl)) {
            //新用户获得金币100
//            PayFlow payFlowNewUser = PayFlow.initPayFlow(userId, REGISTER.getType(), REWARD_TYPE_GOLD, null);
//            this.payFlowService.modifyAccountFlowAndStatByGold(payFlowNewUser, BEEN_INVITED_NEW_USER_REWARD_GOLD_NUM, false);
        }
    }

    /**
     * 生成用户并新增入库
     *
     * @param loginInfo
     * @param platform
     * @param chnl
     * @param userType
     * @return
     */
    private User saveUser(LoginInfo loginInfo, Integer platform, String chnl, Byte userType) {
        Date date = new Date();
        User user = new User();
        user.setToken(this.getToken());
        user.setRefreshtoken(this.getToken());
        user.setRegisterClient(null == platform ? "0" : String.valueOf(platform));
        //不维护年龄字段，因为这个字段每年都在变
        user.setAge(0);
        user.setSex((byte) 1);
        user.setPhone(loginInfo.getPhone());
        user.setEquipment(loginInfo.getEquipment());
        user.setRegisteredAddress(loginInfo.getRegisteredaddress());
        user.setPhoneModel(loginInfo.getPhonemodel());
        user.setState(userType);
        user.setLevel(LevelEunm.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setExpiretime(this.getExpireTime());
        user.setLastLoginTime(date);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);
        if (null != loginInfo.getRegIp()) {
            user.setRegIp(loginInfo.getRegIp());
        }
        if (null != loginInfo.getLastLoginIp()) {
            user.setLastLoginIp(loginInfo.getLastLoginIp());
        }
        if (AppConst.USER_STATE.VISITOR != userType) {
            this.setLoginName(user, loginInfo);
        }

        user.setLocationCode(bizConfigProperties.getLocationCode());
        user.setLocationName(bizConfigProperties.getLocationName());
        if (null != loginInfo.getId()) {
            User originalUser = userMapper.selectByPrimaryKey(loginInfo.getId());
            if (null != originalUser && null != originalUser.getLocationCode() && null != originalUser.getLocationName()) {
                user.setLocationCode(originalUser.getLocationCode());
                user.setLocationName(originalUser.getLocationName());
            }
        }

        user.setChannelId(this.getChannelId(chnl));
        this.userMapper.insertSelective(user);
        return user;
    }

    /**
     * 添加授权登录认证信息
     *
     * @param userId
     * @param loginInfo
     */
    private void saveAuths(Long userId, LoginInfo loginInfo) {
        List<UserAuth> auths = new ArrayList<>();

        // 1.单纯手机号注册 2.手机号+微信注册 3.微信号注册
        UserAuth auth = new UserAuth();
        auth.setUserId(userId);
        auth.setType(loginInfo.getType());
        auth.setIdentifier(loginInfo.getLoginName());
        if (StringUtils.isNotBlank(loginInfo.getPassword())) {
            auth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
        }
        auths.add(auth);
        logger.warn("添加权限验证:{}", loginInfo.getLoginName());

        //如果三方注册
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()) {
            //绑定三方的时候绑定手机
            if (null != loginInfo.getPhone()) {
                UserAuth phoneAuth = new UserAuth();
                phoneAuth.setUserId(userId);
                phoneAuth.setType(AppConst.LOGIN_TYPE.PHONE);
                phoneAuth.setIdentifier(loginInfo.getPhone());
                if (StringUtils.isNotBlank(loginInfo.getPassword())) {
                    phoneAuth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
                }
                auths.add(phoneAuth);
            }

            //绑定三方的时候绑定openid
            if (loginInfo.getOpenId() != null) {
                UserAuth openIdAuth = new UserAuth();
                openIdAuth.setUserId(userId);
                openIdAuth.setType(LOGIN_TYPE.WEIXIN_OPENID);
                openIdAuth.setIdentifier(loginInfo.getOpenId());
                if (StringUtils.isNotBlank(loginInfo.getPassword())) {
                    openIdAuth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
                }
                auths.add(openIdAuth);
            }
        }

        this.userAuthMapper.batchAdd(auths);
    }

    /**
     * 保存用户注册来源信息
     *
     * @param loginInfo 注册信息
     * @param user      用户信息
     * @param chnl      注册渠道
     */
    private void saveRegSourceInfo(LoginInfo loginInfo, User user, String chnl) {
        Long inviteCode = loginInfo.getInviteCode();
        if (null == inviteCode) {
            return;
        }

        this.logger.debug(LogMarker.BIZ, "用户邀请记录,loginInfo:[{}],user:[{}],chnl:[{}]", loginInfo, user, chnl);

        //新增用户注册来源记录入库
        this.saveRegSourceCount(loginInfo, user, chnl);

        bizLogService.inviteSuccessed(inviteCode, user.getId());
        missionService.completeTask(inviteCode, TASK_INVITED_FRIEND, user.getId().toString());
    }

    /**
     * 统一调用missionService提供的方法
     */
    private void statisticsGold(Long inviteCode) {
        //统计当天完成任务获得的金币总数
        KeyGenerator userRewardSumKey = RedisConfig.USER_MISSION_REWARD_SUM_PER_DAY.copy()
                .setKey(inviteCode + ":" + DateUtils.formatDate(new Date()));
        redisStringAdapter.incrementWithDefault(userRewardSumKey, INVITE_NEW_USER_REWARD_GOLD_NUM.longValue(), INVITE_NEW_USER_REWARD_GOLD_NUM.intValue());
        redisStringAdapter.expire(userRewardSumKey, DateUtils.getCurSeconds());
    }

    /**
     * 统一调用missionService提供的方法
     */
    private void generatorInvitationReward(Long userId, Long inviteCode) {
        PayFlow payFlow = PayFlow.initPayFlow(inviteCode, TASK_INVITED_FRIEND.getType(), REWARD_TYPE_GOLD, userId.toString());
        this.payFlowService.modifyAccountFlowAndStatByGold(payFlow, INVITE_NEW_USER_REWARD_GOLD_NUM, false);
    }

    private void saveRegSourceCount(LoginInfo loginInfo, User user, String chnl) {
        RegSourceCounter counter = new RegSourceCounter();
        counter.setId(this.nextSequence().toString());
        counter.setRegTime(new Date());
        counter.setUserId(user.getId());
        //判断用户渠道
        counter.setChannel(this.getChannelId(chnl));
        counter.setInviteUserId(loginInfo.getInviteCode());
        this.regSourceMapper.insert(counter);
    }

    /**
     * 获取渠道
     *
     * @param chnl
     * @return
     */
    private Long getChannelId(String chnl) {
        //判断用户渠道
        return channelSupplyService.getChannelIdByCode(chnl);
    }

    /**
     * 设置登陆名称
     *
     * @param user
     * @param loginInfo
     */
    private void setLoginName(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
            user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
        } else {
            throw new RuntimeException("参数错误");
        }
    }

    @Override
    public void updateUserTokenExpireTime(Long uid) {
        User user = new User();
        user.setId(uid);
        user.setExpiretime(System.currentTimeMillis());
        this.userMapper.updateByPrimaryKeySelective(user);
    }

    private Json check(String phone, String type) {
        if (StringUtils.isBlank(phone) || StringUtils.isBlank(type)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数不能为空");
        } else if (!Validater.checkPhone(phone)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "手机号码格式有误");
        }
        return ResultUtil.genSuccessResult();
    }

    @Override
    public Json<UserInfoDTO> binding(LoginInfo loginInfo, Long userId) {
        if (!this.bindingCheckParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少必要参数");
        }

        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && !this.smsSupplyFeignService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE, loginInfo.getLoginName(), loginInfo.getCode())) {
            return ResultUtil.genFailedResult("请输入正确的验证码");
        }

        String messageByCheckAuth = checkBindExist(loginInfo);
        if (StringUtils.isNotBlank(messageByCheckAuth)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth);
        }

        User user = this.userMapper.selectByPrimaryKey(userId);

        String messageByCheckUser = checkBindRepeat(loginInfo.getType(), user);
        if (StringUtils.isNotBlank(messageByCheckUser)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckUser);
        }

        this.setLoginName(user, loginInfo);
        //更新用户信息(如果有旧的头像和旧的昵称存在，则不用微信的头像和昵称覆盖)
        this.userMapper.updateByPrimaryKeySelective(user);
        //添加登录认证
        this.userAuthService.addUserAuth(loginInfo.getType(), user.getId(), loginInfo.getLoginName(), null);

        UserInfoDTO dto = this.getUserToDTO(user);
        checkCompleteInfo(dto);

        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId().toString(), JSON.toJSONString(dto));

        return ResultUtil.genSuccessResult(dto);
    }

    /**
     * 检测微信号（第三方）是否已被其他账号绑定
     *
     * @param loginInfo
     * @return
     */
    private String checkBindExist(LoginInfo loginInfo) {
        UserAuth auth = this.userAuthMapper.selectByUserAuth(loginInfo.getType(), loginInfo.getLoginName());
        String message = "";
        if (null != auth) {
            if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
                message = "手机号已被绑定";
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
                message = "微信号已被绑定";
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
                message = "微博已被绑定";
            } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
                message = "QQ号已被绑定";
            }
        }
        return message;
    }

    /**
     * 只检查了当前要绑定第三方的用户有没有重复绑定，但是忽视了别的用户也可能已经绑定了第三方
     *
     * @param loginInfoType
     * @param user
     * @return
     */
    private String checkBindRepeat(Byte loginInfoType, User user) {
        String message = "";
        if (AppConst.LOGIN_TYPE.PHONE == loginInfoType && StringUtils.isNotBlank(user.getPhone())) {
            message = "请勿重复绑定手机号";
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfoType && StringUtils.isNotBlank(user.getWeixin())) {
            message = "请勿重复绑定微信账号";
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfoType && StringUtils.isNotBlank(user.getWeibo())) {
            message = "请勿重复绑定微博账号";
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfoType && StringUtils.isNotBlank(user.getQq())) {
            message = "请勿重复绑定QQ账号";
        }
        return message;
    }

    @Override
    public void getRegisterReward(long userId) {
//        PayFlow payFlow = PayFlow.initPayFlow(userId, TaskTypeEnum.REGISTER.getType(), AppConst.REWARD_TYPE_COIN, userId + "");
//        this.payFlowService.modifyAccountFlowAndStatByCoin(payFlow, this.sysConfigRedis.getBigDecimal(RedisContents.TYPE.BUSINESS, RedisContents.REGISTER_REWARD));
    }

    @Override
    public Message updateUserInfo(UserParam updateUser, Long userId, MultipartFile file) {
        User user = this.userMapper.selectByPrimaryKey(userId);
        if (1 != user.getState()) {
            return Message.build(false, "用户尚未登录");
        }
        if (null != updateUser.getNickname()) {
            if (null == user.getIsTempNickName() || user.getIsTempNickName() == 1) {
                user.setNickname(updateUser.getNickname());
                user.setIsTempNickName((byte) 0);
            } else {
                return Message.build(false, "该用户修改昵称已达上限");
            }
        }
        if (null != updateUser.getSex()) {
            user.setSex(updateUser.getSex());

            bizLogService.changeGender(user.getId(), updateUser.getSex());
        }
        if (null != updateUser.getAddress()) {
            user.setAddress(updateUser.getAddress());
        }
        if (null != updateUser.getBirthday()) {
            user.setBirthday(updateUser.getBirthday());
        }
        String headImg = null;
        if (null != file) {
            if (file.getSize() > MAX_HEAD_IMAGE_SIZE) {
                return Message.build(false, "头像图片过大,请重新上传");
            }
            headImg = upLoadHeadImage(userId, file);
            if (null != headImg) {
                user.setHeadImg(upLoadHeadImage(userId, file));
                user.setIsTempHeadImg((byte) 0);
            }
        }
        int i = this.userMapper.updateByPrimaryKeySelective(user);
        if (i > 0) {
            UserInfoDTO dto = this.getUserToDTO(user);
            UserService userService = SpringContextHolder.getBean(this.getClass());
            userService.updateHeadImgFromReply(updateUser.getUserId(), headImg, updateUser.getNickname());
            checkCompleteInfo(dto);
            this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
            return Message.build(true).addParam("user", dto);
        }
        return Message.build(false, "更新失败");
    }

    /**
     * 检查用户是否已经完善资料，如果已经完善则完成任务
     *
     * @param user
     */
    private void checkCompleteInfo(UserInfoDTO user) {
        if (null != user.getWeixin()
                && null != user.getBirthday()
                && StringUtils.isNotEmpty(user.getAddress())
                && StringUtils.isNotEmpty(user.getHeadImg())
                && StringUtils.isNotEmpty(user.getNickname())) {
            missionService.completeTask(user.getId(), TASK_IMPROVE_USER_INFO, null);
        }

    }

    /**
     * 异步更新用户的昵称和头像
     *
     * @param userId
     * @param headImg
     * @param nickname
     */

    @Override
    public void updateHeadImgFromReply(Long userId, String headImg, String nickname) {
        if (headImg != null || nickname != null) {
            this.newsReplyService.triggerUpdateInfo(userId, nickname, headImg);
            this.videoReplyFacadeService.triggerUpdateInfo(userId, nickname, headImg);
        }
    }


    @Override
    public void updateUserBaseInfo(Long userId) {
        User user = new User();
        user.setToken(RandomUtils.getRandomStr(32, true));
        user.setId(userId);
        this.redisHashMapAdapter.remove(RedisConfig.USER_INFO, user.getId().toString());
        this.userMapper.updateByPrimaryKeySelective(user);
    }

    @Override
    public PushInfo getPushInfo(Long userId) {
        return userMapper.getPushInfo(userId);
    }

    @Override
    public LocationUserInfoDTO getLocationUserInfo(Long userId) {
        return userMapper.getLocationUserInfo(userId);
    }

    /**
     * 上传头像图片到oss，路径为avator/对用户进行10取模/
     *
     * @param userId
     * @param file
     * @return
     */
    private String upLoadHeadImage(Long userId, MultipartFile file) {
        int hashCode = (int) (userId % 10);
        return aliyunOSSService.upload(file, "avatar/" + hashCode + "/");
    }

    /**
     * 账号绑定参数必要校验
     *
     * @param loginInfo
     * @return
     */
    private boolean bindingCheckParam(LoginInfo loginInfo) {
        //校验绑定类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }
        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }
        //绑定手机号标识
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }
        //绑定微信，则微信头像和昵称字段不能为空
        return AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                || !StringUtils.isEmpty(loginInfo.getHeadImg())
                || !StringUtils.isEmpty(loginInfo.getNickname());
    }

    @Override
    public LocationDTO doPostingLocation(String code, Long userId) {
        if (StringUtils.isEmpty(code)) {
            logger.error("定位时无法提供地区编码(定位失败)!!!:code:{},userId:{}", code, userId);
        }
        if (StringUtils.isNotEmpty(code) && ((code.length() != 6 && code.length() != 12)) || null == userId) {
            logger.error("参数错误(地区编码位数问题或用户id为空):code:{},userId:{}", code, userId);
        }
        LocationDTO locationDTO = locationService.getLocationByGeocode(code);

        //如果没有传用户id则不需要更新用户以及新增地理位置历史
        if (null != userId) {
            User user = this.userMapper.selectByPrimaryKey(userId);
            //测试环境在删除用户时，客户端还是有用户id的，当客户端传用户id时服务端查不到用户的信息会报服务器错误
            if (null != user) {
                user.setLocationCode(locationDTO.getCode());
                user.setLocationName(locationDTO.getName());
                //更新用户地理位置
                int i = this.userMapper.updateByPrimaryKeySelective(user);
                if (i > 0) {

                    bizLogService.changeArea(userId, code);

                    UserInfoDTO dto = this.getUserToDTO(user);
                    this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));

                    //位置历史表新增
                    UserLocationHistory userLocationHistory = new UserLocationHistory();
                    userLocationHistory.setAddTime(new Date());
                    userLocationHistory.setId(nextSequence());
                    userLocationHistory.setLocationCode(locationDTO.getCode());
                    userLocationHistory.setUserId(userId);
                    userLocationHistoryMapper.insert(userLocationHistory);
                }
            }
        }
        return locationDTO;
    }

    @Override
    public void savePush(PushParam pushParam) {
        User userForPush = userMapper.findByPushToken(pushParam.getPushToken());
        if (null != userForPush) {
            userMapper.clearPushInfo(userForPush.getId());
        }
        User user = userMapper.selectByUserId(pushParam.getUserId());
        if (null != user) {
            PushInfo pushInfo = new PushInfo();
            BeanUtils.copyProperties(pushParam, pushInfo);
            userMapper.savePushInfo(pushInfo);
        }

    }


}
