package com.bxm.localnews.user.account.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.user.account.ProcessorService;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.exception.UserAccountHandlerNotEnoughException;
import com.bxm.localnews.user.integration.BizLogIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.AccountActionParam;
import com.bxm.localnews.user.param.TransformParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import static com.bxm.localnews.user.enums.GoldFlowTypeEnum.FLOWER_PRESENT;

@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class ProcessorServiceImpl extends BaseService implements ProcessorService {

    private final RedisListAdapter redisListAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final UserAccountService userAccountService;

    private final UserService userService;

    private final DataSourceTransactionManager dstManager;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final AccountHandlerProxy accountHandlerProxy;

    @Override
    @Async
    public void doFlowerPresentConsume(Long userId, Long targetUserId) {
        String name = userId + ":" + targetUserId;
        log.debug("进入处理器，处理器名：[{}]", name);
        UserInfoDTO user = userService.getUserCache(userId);
        UserInfoDTO targetUser = userService.getUserCache(targetUserId);

        //若不存在目标用户则直接退出
        if (null == user || null == targetUser) {
            log.error("用户不存在，直接退出");
            return;
        }

        KeyGenerator queueKeyGenerator = RedisConfig.USER_PRESENT_FLOWER_QUEUE.copy().appendKey(name);
        KeyGenerator processorKeyGenerator = RedisConfig.USER_PRESENT_FLOWER_PROCESSOR.copy();
        //开始处理堆积着的数据
        List<TransformParam> transformParamList = new ArrayList<>();

        try {
            while (true) {
                //从redis队列中取出数据保存，等2秒无数据时进行消费！
                TransformParam t =
                        redisListAdapter.blockRightPop(queueKeyGenerator, TransformParam.class, 2, TimeUnit.SECONDS);
                if (null == t) {
                    log.debug("在等待2秒钟后，队列仍无数据，可以开始进行数据处理");
                    break;
                }
                log.debug("从redis中取出数据:[{}]", JSONObject.toJSON(t));
                transformParamList.add(t);
            }

            //转移小红花操作
            log.debug("开始转移小红花数量");

            //更新用户可用小红花数,得到红花明细
            transformParamList = SpringContextHolder.getBean(ProcessorService.class)
                    .calculateFlowerNum(transformParamList, user, targetUser);

            //生成小红花明细
            log.debug("开始生产小红花明细:[{}]", JSONObject.toJSON(transformParamList));
            if (!CollectionUtils.isEmpty(transformParamList)) {
                doDataReport(transformParamList);
            }

        } catch (Exception e) {
            //如果出现错误，需要把已经拿出的数据重新塞回去
            log.error("处理器出现错误,将数据重新放入redis，并向上级抛出");
            if (!CollectionUtils.isEmpty(transformParamList)) {
                redisListAdapter.leftPush(queueKeyGenerator, transformParamList);
            }
            throw e;
        } finally {
            //处理完毕，将处理器删除
            log.debug("处理完毕，将处理器删除");
            redisSetAdapter.remove(processorKeyGenerator, name);
        }
    }

    /**
     * 数据上报
     */
    private void doDataReport(List<TransformParam> transformParamList) {
        for (TransformParam param : transformParamList) {
            // 上报数据埋点
            // 业务事件触发-3034事件（EV） 112.{uid}：用户赠送小红花上报，uid：接收用户id
            PointReportParam reportParam = PointReportParam.build(param);
            reportParam.e("3034");
            reportParam.ev(StringUtils.join("112.", param.getNum(), ".", String.valueOf(param.getTargetUserId())));
            reportParam.put("uid", String.valueOf(param.getUserId()));
            reportParam.put("a", Objects.toString((param.getAreaCode())));
            bizLogIntegrationService.point(reportParam);

            if (log.isDebugEnabled()) {
                log.debug("上传userId: [{}] 领取小红花: [{}]事件，请求参数: {}", param.getTargetUserId(), param.getNum(),
                        reportParam);
            }
        }
    }

    /**
     * 计算小红花数
     */
    @Override
    public List<TransformParam> calculateFlowerNum(List<TransformParam> transformParamList,
                                                   UserInfoDTO user,
                                                   UserInfoDTO targetUser)
            throws UserAccountHandlerNotEnoughException {
        DefaultTransactionDefinition def = new DefaultTransactionDefinition();
        // 事物隔离级别，开启新事务，这样会比较安全些。
        def.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        // 获得事务状态
        TransactionStatus transaction = dstManager.getTransaction(def);

        Integer userAccount = userAccountService.getUserUsableGold(user.getId());

        final int[] sum = {0};
        List<TransformParam> finalTransformParamList = new ArrayList<>();
        transformParamList.forEach(e -> {
            if (sum[0] + e.getNum() <= userAccount) {
                sum[0] = sum[0] + e.getNum();
                finalTransformParamList.add(e);
            }
        });

        try {
            //赠送方消耗金币
            AccountActionParam sendParam = AccountActionParam.goldBuilder(AccountActionEnum.CONSUME_GOLD)
                    .userId(user.getId())
                    .goldNum(sum[0])
                    .goldFlowType(FLOWER_PRESENT)
                    .content("赠送【" + targetUser.getNickname() + "】消耗小红花")
                    .build();

            accountHandlerProxy.handle(sendParam);

            //接收方增加金币
            AccountActionParam receiveParam = AccountActionParam.goldBuilder(AccountActionEnum.ADD_USABLE_GOLD)
                    .userId(targetUser.getId())
                    .goldNum(sum[0])
                    .goldFlowType(FLOWER_PRESENT)
                    .content("收到【" + user.getNickname() + "】赠送的小红花")
                    .build();
            accountHandlerProxy.handle(receiveParam);
        } catch (Exception e) {
            dstManager.rollback(transaction);
            throw e;
        }

        dstManager.commit(transaction);

        return finalTransformParamList;

    }

}
