package com.bxm.localnews.user.account.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.localnews.user.account.CashFlowService;
import com.bxm.localnews.user.account.ProcessorService;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.account.UserWithdrawService;
import com.bxm.localnews.user.account.impl.callback.IAccountActionCallback;
import com.bxm.localnews.user.account.impl.context.AccountActionContext;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAccountMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.exception.UserAccountException;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.integration.MessageUserIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.integration.WithdrawIntegrationService;
import com.bxm.localnews.user.invite.InviteRecordService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.AccountActionParam;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.param.AccountGoldParam;
import com.bxm.localnews.user.param.TransformParam;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.redis.*;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.user.constant.RedisConfig.USER_ACCOUNT_CACHE;
import static com.bxm.localnews.user.enums.AccountActionEnum.ADD_WITHDRAWAL_CASH;
import static com.bxm.localnews.user.enums.AccountActionEnum.DRAWABLE_TO_WITHDRAWAL;
import static com.bxm.localnews.user.enums.CashFlowTypeEnum.WITHDRAWAL;
import static com.bxm.localnews.user.enums.CashFlowTypeEnum.WITHDRAWAL_SUCCESS;
import static com.bxm.localnews.user.enums.WithdrawEnum.DURING_PAYMENT;

/**
 * @author zhaoyadong 2019/3/6 9:25
 * @desc
 */
@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class UserAccountServiceImpl extends BaseService implements UserAccountService {

    private final UserAccountMapper userAccountMapper;

    private final DistributedLock distributedLock;

    private final CashFlowService cashFlowService;

    private final UserWithdrawService userWithdrawService;

    private final UserService userService;

    private final InviteRecordService inviteRecordService;

    private final MessageSender messageSender;

    private final RedisStringAdapter redisStringAdapter;

    private final PushMsgIntegrationService pushMsgIntegrationService;

    private final MessageUserIntegrationService messageUserIntegrationService;

    private final UserProperties userProperties;

    private final WithdrawIntegrationService withdrawIntegrationService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final RedisListAdapter redisListAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final LocationIntegrationService locationIntegrationService;

    private final AccountHandlerProxy accountHandlerProxy;

    @Override
    public BigDecimal getUserTotalCash(Long userId) {
        return getUserAccountDetail(userId).getTotalCash();
    }

    @Override
    public UserAccount getUserAccountDetail(Long userId) {
        logger.debug("从数据库获取用户账户信息：用户id:[{}]", userId);
        return getAndSetUserAccount(userId);
    }

    @Override
    public Message createUserAccount(Long userId) {
        logger.debug("创建用户账户信息：用户id:[{}]", userId);
        getAndSetUserAccount(userId);
        return Message.build();
    }

    @Override
    public void cleanCache(Long userId) {
        redisStringAdapter.remove(buildCacheKey(userId));
    }

    private KeyGenerator buildCacheKey(Long userId) {
        return USER_ACCOUNT_CACHE.copy().appendKey(userId);
    }

    @Override
    public UserAccountDTO getUserAccount(Long userId) {
        KeyGenerator key = buildCacheKey(userId);
        UserAccountDTO res = redisStringAdapter.get(key, UserAccountDTO.class);

        if (null == res) {
            res = convertUserAccount(getAndSetUserAccount(userId));
            redisStringAdapter.set(key, res);
        } else if (UserAccountDTO.CURRENT_VERSION > res.getV()) {
            res = convertUserAccount(getAndSetUserAccount(userId));
            redisStringAdapter.set(key, res);
        }

        return res;
    }

    @Override
    public UserAccountDTO getUserAccountAndEnableWithdraw(Long userId) {
        return getUserAccount(userId);
    }

    @Override
    public UserAccountDayCashDTO getUserAccountAndDetail(Long userId, String areaCode) {
        UserWithdrawDTO userWithdrawDTO = userWithdrawService.getUserRecentWithdrawInfo(userId);
        //判断用户最后一笔订单是否处于审核状态
        if (userWithdrawDTO == null || WithdrawEnum.SUCCESS_PAYMENT.getState().equals(userWithdrawDTO.getStatus())
                || WithdrawEnum.FAIL_PAYMENT.getState().equals(userWithdrawDTO.getStatus())) {
            userWithdrawDTO = null;
        }

        UserAccountDTO userAccountDTO = getUserAccount(userId);
        // 判断当前区域是否开启了 摇钱树/红包邀请好友的活动
        if (StringUtils.isNotBlank(areaCode)) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);

            // 如果未开启 则设置在路上金额为0
            if (!Objects.equals(locationDTO.getEnablePaidPromote(), (byte) 1)) {
                userAccountDTO.setNotArriveCash(BigDecimal.ZERO);
            }
        }

        return new UserAccountDayCashDTO(userAccountDTO,
                cashFlowService.getRecentSevenDayCashFlowByUser(userId),
                userWithdrawDTO);
    }

    @Override
    public UserCashGoldDTO getUserCashGold(Long userId) {
        logger.debug("获取用户账户在个人中心的信息：用户id:[{}]", userId);
        UserAccountDTO userAccount = getUserAccount(userId);

        UserCashGoldDTO userCashGoldDTO = new UserCashGoldDTO(userAccount.getEstimateCash(),
                userAccount.getUsableGold(),
                BigDecimal.valueOf(userAccount.getUsableGold())
                        .divide(userProperties.getGoldRate(), 2, BigDecimal.ROUND_DOWN),
                userAccount.getTotalGold());

        userCashGoldDTO.setCashFormat(convert(userCashGoldDTO.getCash()));
        return userCashGoldDTO;
    }

    private String convert(BigDecimal cash) {

        if (cash == null || cash.compareTo(new BigDecimal(0)) == 0) {
            return "";
        }

        BigDecimal multiple = new BigDecimal("10000");
        // 大于一万
        if (cash.compareTo(multiple) >= 0) {
            return NumberUtils.format(cash.divide(multiple)) + "W元";
        }

        // 大于一千
        multiple = new BigDecimal("1000");
        if (cash.compareTo(multiple) >= 0) {
            return NumberUtils.format(cash.divide(multiple)) + "K元";
        }

        return NumberUtils.format(cash) + "元";
    }

    @Override
    public Boolean addCash(AccountCashParam param) {
        if (null == param
                || null == param.getUserId()
                || null == param.getCashType()
                || null == param.getAddTotal()
                || null == param.getCash()
                || param.getCash().compareTo(BigDecimal.ZERO) == 0) {
            return false;
        }

        logger.debug("用户账户金额更新：{}", param);
        AccountActionParam.CashBuilder builder = AccountActionParam.cashBuilder()
                .amount(param.getCash())
                .userId(param.getUserId())
                .relationId(param.getRelationId());

        if (StringUtils.isNotBlank(param.getRemake())) {
            builder.remark(param.getRemake());
        }
        if (StringUtils.isNotBlank(param.getCashFlowType())) {
            builder.cashFlowType(CashFlowTypeEnum.valueOf(param.getCashFlowType()));
        }

        if (CashEnum.TEMP_CASH.name().equals(param.getCashType())) {
            builder.action(AccountActionEnum.ADD_TEMP_CASH);
        } else if (CashEnum.STABILIZE_CASH.name().equals(param.getCashType())) {
            builder.action(AccountActionEnum.ADD_STABILIZE_CASH);
        } else if (CashEnum.DRAWABLEL_CASH.name().equals(param.getCashType())) {
            builder.action(AccountActionEnum.ADD_DRAWABLE_CASH);
        } else if (CashEnum.SAVE_CASH.name().equals(param.getCashType())) {
            builder.action(AccountActionEnum.ADD_SAVE_CASH);
        }

        if (CashFlowTypeEnum.FAIL_WITHDRAWAL.name().equals(param.getCashFlowType())) {
            // 提现失败
            builder.action(AccountActionEnum.FAIL_WITHDRAWAL);
        } else if (CashFlowTypeEnum.WECHAT_WITHDRAWAL.name().equals(param.getCashFlowType())
                || CashFlowTypeEnum.ALIPAY_WITHDRAWAL.name().equals(param.getCashFlowType())) {
            // 可提现金额到提现中（提现审核）
            builder.action(DRAWABLE_TO_WITHDRAWAL);
        }

        try {
            accountHandlerProxy.handler(builder.build(), new IAccountActionCallback() {
                @Override
                public void postAction(AccountActionContext context) {
                    // 不发送推送  用户红包不发
                    if (!(CashFlowTypeEnum.RED_PACKET.name().equals(param.getCashFlowType())
                            || CashFlowTypeEnum.COMMAND_RED_PACKET.name().equals(param.getCashFlowType()))) {
                        //只有增加可提现金额时进行推送
                        if (AccountActionEnum.ADD_DRAWABLE_CASH.equals(context.getAction())) {
                            pushMsgIntegrationService.pushChangeAccountMsg(param);
                        }
                    }
                }
            });
        } catch (UserAccountException e) {
            return false;
        }

        return true;
    }

    @Override
    public Boolean transferCash(AccountCashParam param) {
        logger.debug("转换的账户信息：[{}]", JSON.toJSONString(param));
        if (null == param || null == param.getUserId() || null == param.getCashType() || null == param.getCash()
                || null == param.getConvertType() || param.getCash().compareTo(BigDecimal.ZERO) <= 0) {
            return false;
        }

        if (StringUtils.isNotBlank(param.getConvertType())) {
            AccountActionEnum actionEnum = AccountActionEnum.valueOf(param.getConvertType());

            accountHandlerProxy.handle(AccountActionParam.cashBuilder(actionEnum)
                    .userId(param.getUserId())
                    .cashFlowType(CashFlowTypeEnum.valueOf(param.getCashFlowType()))
                    .amount(param.getCash())
                    .relationId(param.getRelationId())
                    .build());
            return true;
        }

        return false;
    }

    @Override
    public Message updateUserWithdrawInfo(Long userId, BigDecimal amount, Byte state) {
        try {
            if (DURING_PAYMENT.getState().equals(state)) {
                // 更新为提现中的状态
                accountHandlerProxy.handle(AccountActionParam.cashBuilder(DRAWABLE_TO_WITHDRAWAL)
                        .amount(amount)
                        .userId(userId)
                        .cashFlowType(WITHDRAWAL)
                        .build());
            } else if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(state)) {
                //更新用户提现赏金转为已提现赏金
                accountHandlerProxy.handle(AccountActionParam.cashBuilder(ADD_WITHDRAWAL_CASH)
                        .amount(amount)
                        .userId(userId)
                        .cashFlowType(WITHDRAWAL_SUCCESS)
                        .build());

                //更新邀请人的赏金为可提现赏金
                updateInviteUserCash(userId, ExtraInviteStateEnum.WITHDRAW);
            } else if (WithdrawEnum.FAIL_PAYMENT.getState().equals(state)) {
                //提现失败
                AccountCashParam accountCashParam = new AccountCashParam(userId,
                        CashEnum.DRAWABLEL_CASH.name(),
                        Boolean.FALSE,
                        amount,
                        null,
                        CashFlowTypeEnum.FAIL_WITHDRAWAL.name(),
                        null);
                addCash(accountCashParam);
            }
        } catch (UserAccountException e) {
            logger.warn("提现操作失败，失败原因：" + e.getMessage());
            return Message.build(false, e.getMessage());
        }
        return Message.build();
    }

    @Override
    public BigDecimal getUserDrawableCash(Long userId) {
        return getUserAccount(userId).getDrawablelCash();
    }

    @Override
    public Boolean addGold(AccountGoldParam param) {
        logger.debug("用户账户更新金币的参数：[{}]", param);

        AccountActionParam.GoldBuilder builder = AccountActionParam.goldBuilder()
                .userId(param.getUserId())
                .goldNum(param.getGold())
                .content(param.getContent())
                .reason(param.getReason())
                .goldFlowType(GoldFlowTypeEnum.valueOf(param.getGoldFlowType()));

        if (StringUtils.isNotBlank(param.getReason())) {
            builder.remark(param.getReason());
        }

        if (GoldEnum.TEMP_GOLD.name().equals(param.getGoldType())) {
            builder.action(AccountActionEnum.ADD_TEMP_GOLD);
        } else if (GoldEnum.USABLE_GOLD.name().equals(param.getGoldType())) {
            builder.action(AccountActionEnum.ADD_USABLE_GOLD);
        } else if (GoldEnum.CONSUME_GOLD.name().equals(param.getGoldType())) {
            builder.action(AccountActionEnum.CONSUME_GOLD);
        }

        try {
            accountHandlerProxy.handler(builder.build(), new IAccountActionCallback() {
                @Override
                public void postAction(AccountActionContext context) {
                    //文章被赞,或点赞文章,分享内容获得红花不推送
                    boolean excludeGoldFlow = StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.ARTICLE_LIKE_ADD.getName(), param.getGoldFlowType()) ||
                            StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.ARTICLE_LIKE_EXPEND.getName(), param.getGoldFlowType()) ||
                            StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.FLOWER_PRESENT.getName(), param.getGoldFlowType()) ||
                            StringUtils.equalsIgnoreCase(GoldFlowTypeEnum.TASK_NEWS_SHARE.getName(), param.getGoldFlowType());
                    if (!excludeGoldFlow) {
                        //获得收益时的推送
                        GoldFlow goldFlow = context.getGoldFlow(context.getUserId());
                        if (null != goldFlow) {
                            messageUserIntegrationService.addGoldMsg(goldFlow, getUserUsableGold(param.getUserId()));
                        }
                    }
                }
            });
        } catch (UserAccountException e) {
            return false;
        }

        return true;
    }

    @Override
    public Integer getUserUsableGold(Long userId) {
        logger.debug("获取用户[{}]的账户金币可用余额", userId);
        return getUserAccountDetail(userId).getUsableGold();
    }

    @Override
    public void userWithdrawPop(Long userId) {
        redisStringAdapter.set(getUserWithdrawPop(userId), 1);
    }

    @Override
    public Json<BigDecimal> userGoldWithdraw(Long userId, String openId, BigDecimal amount, String devcId, String ip) {
        if (userProperties.getMixWithdrawBalance().compareTo(amount) > 0 ||
                amount.compareTo(userProperties.getMaxWithdrawBalance()) > 0) {
            logger.info("当前用户[{}]提现金额[{}], 不满足提现范围", userId, amount);
            return ResultUtil.genFailedResult("请输入正确的提现金额");
        }

        Integer usableGold = getUserUsableGold(userId);
        Integer transferGold = amount.multiply(userProperties.getGoldRate()).intValue();
        if (usableGold < transferGold) {
            logger.info("当前用户[{}]可用积分[{}]，小于提现积分[{}], 不允许提现", userId, usableGold, transferGold);
            return ResultUtil.genFailedResult("余额不足");
        }

        BigDecimal withdrawAmount = userWithdrawService.getMiniAppUserWithdraw(userId, openId, WithdrawEnum.SUCCESS_PAYMENT.getState());
        if (userProperties.getTodayLimitBalance().compareTo(withdrawAmount) <= 0) {
            logger.info("用户[{}]今日提现金额已达上限", userId);
            return ResultUtil.genFailedResult("单日兑换已达限额");
        }

        AccountGoldParam param = new AccountGoldParam(userId, "CONSUME_GOLD", false,
                transferGold, null, GoldFlowTypeEnum.MINI_APP_WITHDRAW.name());
        if (addGold(param)) {
            Message message = withdrawIntegrationService.appletWithdraw(userId, openId, amount, devcId, ip);
            if (message.isSuccess()) {
                //提现成功，记录流水
                CashFlow cashFlow = new CashFlow();
                cashFlow.setId(nextSequence());
                cashFlow.setUserId(userId);
                cashFlow.setAmount(amount);
                cashFlow.setType(CashFlowTypeEnum.WECHAT_WITHDRAWAL.getName());
                cashFlow.setCreateTime(new Date());
                cashFlowService.addCashFlow(cashFlow);
                return ResultUtil.genSuccessResult(amount);
            } else {
                //提现失败---金币退回
                logger.info("用户[{}]提现失败-----金币退回", userId);
                param = new AccountGoldParam(userId, "CONSUME_GOLD", false, -transferGold,
                        null, GoldFlowTypeEnum.WITHDRAW_FAIL_RETURN.name(), null);
                addGold(param);
                return ResultUtil.genFailedResult(message.getLastMessage());
            }
        } else {
            return ResultUtil.genFailedResult("积分扣除失败");
        }
    }

    @Override
    public UserCashInfoVo getUserCashInfo(Long userId) {
        UserCashInfoVo userCashInfoVo = new UserCashInfoVo();

        //获取用户账户金额
        userCashInfoVo.setUserId(userId);
        UserInfoDTO userInfoDTO = userService.getUserCache(userId);
        if (null == userInfoDTO) {
            return userCashInfoVo;
        }
        userCashInfoVo.setNickname(userInfoDTO.getNickname());
        userCashInfoVo.setHeadImg(userInfoDTO.getHeadImg());
        UserAccount userAccount = userAccountMapper.getUserAccount(userId);
        if (null != userAccount) {
            userCashInfoVo.setCash(userAccount.getTotalCash());
        }

        User user = userService.selectByPrimaryKey(userId);
        if (null == user || user.getInviteUserId() == null) {
            return userCashInfoVo;
        }

        //获取邀请人账户金额
        userCashInfoVo.setInviteUserId(user.getInviteUserId());
        UserInfoDTO inviteUserInfoDTO = userService.getUserCache(user.getInviteUserId());
        if (null == inviteUserInfoDTO) {
            return userCashInfoVo;
        }
        userCashInfoVo.setInviteUserNickname(inviteUserInfoDTO.getNickname());
        userCashInfoVo.setInviteUserHeadImg(inviteUserInfoDTO.getHeadImg());
        UserAccount inviteUserAccount = userAccountMapper.getUserAccount(user.getInviteUserId());
        if (null != inviteUserAccount) {
            userCashInfoVo.setInviteUserCash(inviteUserAccount.getTotalCash());
        }

        return userCashInfoVo;
    }

    /**
     * 构建一个用户账户信息
     */
    private UserAccount generateUserAccount(Long userId) {
        UserAccount userAccount = new UserAccount(userId);
        userAccount.setId(nextSequence());
        return userAccount;
    }

    /**
     * 转换账户信息
     */
    private UserAccountDTO convertUserAccount(UserAccount userAccount) {
        UserAccountDTO accountDTO = new UserAccountDTO();
        BeanUtils.copyProperties(userAccount, accountDTO);

        //需要特殊处理的字段
        accountDTO.setDrawablelCash(userAccount.getDrawableCash());

        //未到账金额 = 临时金额 + 固化现金
        accountDTO.setNotArriveCash(userAccount.getTempCash().add(userAccount.getStabilizeCash()));
        //预期收益 = 已结算+未结算
        if (userAccount.getSettledCash() == null) {
            userAccount.setSettledCash(BigDecimal.ZERO);
        }
        if (userAccount.getNoSettlementCash() == null) {
            userAccount.setNoSettlementCash(BigDecimal.ZERO);
        }
        accountDTO.setEstimateCash(BigDecimal.ZERO.add(userAccount.getSettledCash())
                .add(userAccount.getNoSettlementCash()));
        accountDTO.setSaveCash(userAccount.getSaveCash());

        return accountDTO;
    }

    /**
     * 获取用户账户信息（不存在则创建）
     */
    private UserAccount getAndSetUserAccount(Long userId) {
        UserAccount userAccount = this.userAccountMapper.getUserAccount(userId);
        if (null == userAccount) {
            logger.warn("用户[{}]账户信息不存在", userId);
            userAccount = generateUserAccount(userId);
            if (distributedLock.lock(RedisConfig.USER_ACCOUNT_LOCK.appendKey(userId).gen(), userAccount.getId().toString())) {
                this.userAccountMapper.addUserAccount(userAccount);
            }
        } else {
            BigDecimal usableGold = BigDecimal.valueOf(userAccount.getUsableGold());
            if (usableGold != null && usableGold.compareTo(BigDecimal.ZERO) < 0) {
                userAccount.setUsableGold(0);
            }
        }

        return userAccount;
    }

    /**
     * 添加用户账户变化事件
     */
    private void addUserEvent(Long userId, BigDecimal amount) {
        UserCashGoldDTO userCashGoldDTO = getUserCashGold(userId);
        PushPayloadInfo pushPayloadInfo = PushPayloadInfo.build(PushMessageEnum.USER_ACCOUNT_ENVET)
                .addExtend("userId", userId).addExtend("current", userCashGoldDTO.getCash())
                .addExtend("val", amount);

        messageSender.sendUserEvent(pushPayloadInfo);
    }

    @Override
    public void updateInviteUserCash(Long userId, ExtraInviteStateEnum currentStateEnum) {
        User user = userService.selectByPrimaryKey(userId);
        if (null == user) {
            logger.debug("用户不存在，id：[{}]", userId);
            return;
        }
        if (null == user.getInviteUserId()) {
            logger.debug("用户[{}]的邀请人为空", userId);
            return;
        }

        InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), userId);
        if (null == inviteRecord) {
            logger.debug("用户[{}]与邀请人[{}]的关系无效", userId, user.getInviteUserId());
            return;
        }

        if (!InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
            logger.debug("当前邀请关系不是处于登录app状态----不进行更新操作");
            return;
        }

        boolean changeInviteUserCashState = false;

        //只有面对面邀请需要同时判断受邀人是否提现且第二天登录
        if (InviteTypeEnum.FACEINVITE.getName().equals(inviteRecord.getType())) {
            Integer inviteState = redisHashMapAdapter.get(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), Integer.class);
            if (null != inviteState) {
                if (ExtraInviteStateEnum.WITHDRAW.equals(currentStateEnum) && ExtraInviteStateEnum.LOGIN_TWICE.getStatus()
                        .equals(inviteState)
                        || ExtraInviteStateEnum.LOGIN_TWICE.equals(currentStateEnum) && ExtraInviteStateEnum.WITHDRAW.getStatus()
                        .equals(inviteState)) {
                    changeInviteUserCashState = true;
                    redisHashMapAdapter.remove(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString());
                }
            } else {
                redisHashMapAdapter.put(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), currentStateEnum.getStatus());
            }
        } else {
            changeInviteUserCashState = true;
        }

        if (changeInviteUserCashState) {
            //更新邀请关系
            logger.info("邀请人[{}]增加可提现赏金[{}]", user.getInviteUserId(), inviteRecord.getAward());
            inviteRecord.setInviteState(InviteRecordStateEnum.SUCCESS_WITHDRAW.getName());
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecordService.updateInviteRecord(inviteRecord);

            //转换邀请人金额
            AccountCashParam accountCashParam = new AccountCashParam(inviteRecord.getUserId(),
                    CashEnum.DRAWABLEL_CASH.name(),
                    Boolean.FALSE,
                    inviteRecord.getAward(),
                    inviteRecord.getId(),
                    CashFlowTypeEnum.PROMOTE_BOUNTY.name(),
                    AccountActionEnum.STABILIZE_TO_DRAWABLE.name());
            transferCash(accountCashParam);
        }
    }

    @Override
    public BigDecimal getAccountTempCash(Long userId) {
        return userAccountMapper.getAccountTempCash(userId);
    }

    @Override
    public BigDecimal presentFlower(TransformParam transformParam) {
        String name = transformParam.getUserId() + ":" + transformParam.getTargetUserId();
        KeyGenerator queueKeyGenerator = RedisConfig.USER_PRESENT_FLOWER_QUEUE.copy().appendKey(name);
        KeyGenerator processorKeyGenerator = RedisConfig.USER_PRESENT_FLOWER_PROCESSOR.copy();
        redisListAdapter.leftPush(queueKeyGenerator, transformParam);
        if (!redisSetAdapter.exists(processorKeyGenerator, name)) {
            logger.debug("不存在处理器，创建处理器");
            //没有相应的处理器则调用处理器处理，如果有处理器正在处理，则不做操作
            redisSetAdapter.add(processorKeyGenerator, name);
            redisSetAdapter.expire(processorKeyGenerator, 30);
            SpringContextHolder.getBean(ProcessorService.class)
                    .doFlowerPresentConsume(transformParam.getUserId(), transformParam.getTargetUserId());
        }
        return BigDecimal.ONE;
    }

    private KeyGenerator getUserWithdrawPop(Long userId) {
        return RedisConfig.USER_WITHDRAW_POP.copy().appendKey(userId);
    }
}
