package com.bxm.localnews.user.account.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.auth.constant.AuthRedisKey;
import com.bxm.localnews.auth.enums.AuthCodeEnum;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.user.account.VirtualService;
import com.bxm.localnews.user.attribute.UserPersonalInfoService;
import com.bxm.localnews.user.auth.UserAuthCodeService;
import com.bxm.localnews.user.domain.AdminVirtualMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.domain.UserVirtualLoginHistoryMapper;
import com.bxm.localnews.user.domain.VirtualUserMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserVirtualAdminDTO;
import com.bxm.localnews.user.dto.UserVirtualDTO;
import com.bxm.localnews.user.dto.UserVirtualLoginMetaDTO;
import com.bxm.localnews.user.enums.ActivationVipStatusEnum;
import com.bxm.localnews.user.enums.AdminVirtualTypeEnum;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.vo.AdminVirtual;
import com.bxm.localnews.user.param.UserVirtualAddParam;
import com.bxm.localnews.user.param.UserVirtualAdminParam;
import com.bxm.localnews.user.param.UserVirtualLoginParam;
import com.bxm.localnews.user.param.UserVirtualParam;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.support.ChannelService;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserVip;
import com.bxm.localnews.user.vo.UserVirtualLoginHistory;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.github.pagehelper.PageHelper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @author: gengyy
 * @create: 2020-10-26 09:17
 */
@Service
@Slf4j
@AllArgsConstructor
public class VirtualServiceImpl implements VirtualService {
    private final AdminVirtualMapper adminVirtualMapper;

    private final UserService userService;

    private final UserAuthCodeService userAuthCodeService;

    private final ChannelService channelService;

    private final VirtualUserMapper virtualUserMapper;

    private final SequenceCreater sequenceCreater;

    private final UserMapper userMapper;

    private final UserVipService userVipService;

    private final UserVirtualLoginHistoryMapper userVirtualLoginHistoryMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserProperties userProperties;

    @Override
    public Message virtualToLogin(UserVirtualLoginParam userVisitParam) {
        if (Objects.equals(userVisitParam.getCurrentUserId(), userVisitParam.getTargetUserId())) {
            return Message.build(false, "当前用户已登录");
        }
        //是否是虚拟用户
        boolean isVirtualUser = true;
        //是否切换到主账号
        if (Objects.equals(userVisitParam.getTargetUserId(), userVisitParam.getVirtualAdminId())) {
            isVirtualUser = false;
        }
        if (userVisitParam.getVirtualAdminId() == null || userVisitParam.getVirtualAdminId() == 0) {
            //说明是主账号首次切换登陆
            userVisitParam.setVirtualAdminId(userVisitParam.getCurrentUserId());
        }
        //如果管理员ID和切换的账号ID相同，则说明要切到主账号
        //否则查询要切换的账号是否是此管理员下的马甲号
        if (!Objects.equals(userVisitParam.getVirtualAdminId(), userVisitParam.getTargetUserId())) {
            Boolean exist = adminVirtualMapper.existInfoByAdminAndType(userVisitParam.getVirtualAdminId(),
                    AdminVirtualTypeEnum.APP.getType(), userVisitParam.getTargetUserId());
            if (!exist) {
                return Message.build(false, "请切换正确的账号");
            }
        }
        UserVirtualLoginMetaDTO login = login(userVisitParam, isVirtualUser);
        return Message.build(true).addParam("meta", login);
    }

    private UserVirtualLoginMetaDTO login(UserVirtualLoginParam userVirtualLoginParam, Boolean isVirtualUser) {
        //查找用户信息
        User user = this.userMapper.selectByPrimaryKey(userVirtualLoginParam.getTargetUserId());
        //登录
        UserInfoDTO dto = this.virtualLogin(userVirtualLoginParam, user, isVirtualUser);
        dto.setToken(generateToken(user.getId(), null));
        //如果是虚拟用户，需要给一个虚拟的手机号
        if (isVirtualUser) {
            dto.setPhone(userProperties.getVirtualUserPhone());
        }
        //马甲号也判断是否是vip
        UserVip userVip = userVipService.getUserVipByUid(userVirtualLoginParam.getTargetUserId());
        //用户是否为VIP
        dto.setIsVip(Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.USEFUL.getStatus()) ? 1 : 0);
        //vip剩余激活次数
        if (Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.USEFUL.getStatus())) {
            dto.setLeftActiveNum(userVip.getLeftOpenNum());
        } else {
            dto.setLeftActiveNum(0);
        }

        //返回给客户端包装类
        UserVirtualLoginMetaDTO meta = new UserVirtualLoginMetaDTO(dto);
        meta.setVirtualAdminId(userVirtualLoginParam.getVirtualAdminId());
        meta.setIsVirtualUser(isVirtualUser);
        //保存用户登录信息
        this.saveVirtualLoginLog(userVirtualLoginParam);

        return meta;
    }

    /**
     * 创建用户token，并存储到redis，在网关层会根据存储的token进行用户登录状态判断
     */
    private String generateToken(Long userId, String token) {
        if (null == token) {
            token = JwtTokenUtil.generateToken(userId);
        }
        redisHashMapAdapter.put(UserRedisKey.HASH_USER_TOKEN, String.valueOf(userId), token);
        return token;
    }

    @Async
    public void saveVirtualLoginLog(UserVirtualLoginParam virtualLoginParam) {
        String phoneType = null;
        if (PlatformEnum.ANDROID.getCode() == virtualLoginParam.getPlatform()) {
            phoneType = PlatformEnum.ANDROID.name();
        }
        if (PlatformEnum.IOS.getCode() == virtualLoginParam.getPlatform()) {
            phoneType = PlatformEnum.IOS.name();
        }
        UserVirtualLoginHistory loginHistory = new UserVirtualLoginHistory();
        loginHistory.setId(sequenceCreater.nextLongId());
        loginHistory.setLastLoginIp(virtualLoginParam.getIp());
        loginHistory.setVirtualId(virtualLoginParam.getTargetUserId());
        loginHistory.setVirtualAdminId(virtualLoginParam.getVirtualAdminId());
        loginHistory.setLastLoginTime(new Date());
        loginHistory.setPhoneModel(phoneType);
        loginHistory.setClientVersion(virtualLoginParam.getCurVer());
        loginHistory.setServerVersion(virtualLoginParam.getVersion());
        loginHistory.setOperatingSystem(phoneType);
        loginHistory.setCreateTime(new Date());
        loginHistory.setUpdateTime(new Date());
        userVirtualLoginHistoryMapper.insertSelective(loginHistory);
    }

    /**
     * 登录马甲号用户
     */
    private UserInfoDTO virtualLogin(UserVirtualLoginParam userVirtualLoginParam, User user, Boolean isVirtualUser) {
        //更新用户token
        user.setRefreshtoken(JwtTokenUtil.generateToken(user.getId()));
        user.setLastLoginIp(userVirtualLoginParam.getIp());
        user.setLastLoginTime(new Date());
        user.setChannelId(user.getChannelId() == 0
                ? channelService.getChannelByCode(userVirtualLoginParam.getChnl()).getId()
                : user.getChannelId());
        user.setInfoCompleteState(SpringContextHolder.getBean(UserPersonalInfoService.class).initUserInfoCompleteStatus(user));
        user.setRegisterClient(userVirtualLoginParam.getPlatform() + "");
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]对应的状态：[{}]", user.getId(), user.getState());
        }
        UserInfoDTO userInfoDetail = this.userService.loadUserToRedis(user.getId());
        addAuthInfo(userInfoDetail, isVirtualUser);
        return userInfoDetail;
    }

    /**
     * 填充用户拥有的权限编码信息
     *
     * @param userInfo 用户信息
     */
    private void addAuthInfo(UserInfoDTO userInfo, Boolean isVirtualUser) {
        if (isVirtualUser) {
            String combineCode = redisHashMapAdapter.get(AuthRedisKey.USER_AUTH_CODE, userInfo.getId().toString(), String.class);
            if (combineCode == null) {
                Long[] authCodeArray = BitOperatorUtil.setBitToArray(new Long[]{0L}, AuthCodeEnum.VIRTUAL_ADMIN_MANAGE.index, true);
                String authCombineCode = StringUtils.join(authCodeArray, ",");
                redisHashMapAdapter.put(AuthRedisKey.USER_AUTH_CODE, userInfo.getId().toString(), authCombineCode);
            }
            userInfo.setCombineAuthCode(userAuthCodeService.getAuthCombineCode(userInfo.getId()));
            userInfo.setAuthResources(userAuthCodeService.getAuthResources(userInfo.getId()));
        } else {
            userInfo.setCombineAuthCode(userAuthCodeService.getAuthCombineCode(userInfo.getId()));
            userInfo.setAuthResources(userAuthCodeService.getAuthResources(userInfo.getId()));
        }
    }

    @Override
    public List<UserVirtualAdminDTO> getAdminVirtualList(UserVirtualAdminParam userVisitParam) {
        if (userVisitParam.getVirtualAdminId() == null || userVisitParam.getVirtualAdminId() == 0) {
            userVisitParam.setVirtualAdminId(userVisitParam.getUserId());
        }
        //管理员的账户信息不在马甲库中，需要单独构建
        User user = userService.selectByPrimaryKey(userVisitParam.getVirtualAdminId());
        if (user == null) {
            return Collections.emptyList();
        }
        UserVirtualAdminDTO userVirtualAdminDTO = new UserVirtualAdminDTO();
        userVirtualAdminDTO.setVirtualHeadImg(user.getHeadImg());
        userVirtualAdminDTO.setVirtualNickName(user.getNickname());
        userVirtualAdminDTO.setVirtualId(user.getId());
        if (userVisitParam.getUserId().equals(userVisitParam.getVirtualAdminId())) {
            userVirtualAdminDTO.setIsLogin(true);
        } else {
            userVirtualAdminDTO.setIsLogin(false);
        }
        List<UserVirtualAdminDTO> adminVirtualList = new ArrayList<>();
        //list逻辑，主账号放第一位，马甲号已添加顺序倒叙，如果是马甲号登陆
        //则登陆的马甲号放第二位，马甲号已添加顺序倒叙
        //只有第一页数据才加入主账号信息
        if (userVisitParam.getPageNum() != null && userVisitParam.getPageNum() == 1) {
            adminVirtualList.add(userVirtualAdminDTO);
        }
        List<UserVirtualAdminDTO> userVirtualList = PageHelper.startPage(userVisitParam).doSelectPage(() -> adminVirtualMapper.getAdminVirtualList(userVisitParam));
        Iterator<UserVirtualAdminDTO> adminDTOIterator = userVirtualList.iterator();
        while (adminDTOIterator.hasNext()) {
            UserVirtualAdminDTO item = adminDTOIterator.next();
            if (userVisitParam.getUserId().equals(item.getVirtualId())) {
                item.setIsLogin(true);
                adminVirtualList.add(item);
                adminDTOIterator.remove();
            } else {
                item.setIsLogin(false);
            }
        }
        adminVirtualList.addAll(userVirtualList);
        return adminVirtualList;
    }

    @Override
    public List<UserVirtualDTO> getVirtualList(UserVirtualParam userVisitParam) {
        if (userVisitParam.getVirtualAdminId() == null || userVisitParam.getVirtualAdminId() == 0) {
            userVisitParam.setVirtualAdminId(userVisitParam.getUserId());
        }

        List<UserVirtualDTO> userVirtualList = PageHelper.startPage(userVisitParam)
                .doSelectPage(() -> adminVirtualMapper.getVirtualList(userVisitParam));

        return userVirtualList;
    }

    @Override
    public Boolean addVirtualToAdminAccount(UserVirtualAddParam userVisitParam) {
        if (userVisitParam.getVirtualAdminId() == null || userVisitParam.getVirtualAdminId() == 0) {
            userVisitParam.setVirtualAdminId(userVisitParam.getUserId());
        }
        boolean existVirtualUser = virtualUserMapper.existVirtualUser(userVisitParam.getVirtualId());
        if (!existVirtualUser) {
            log.error("马甲号信息不存在 id：{}", userVisitParam.getVirtualId());
            return Boolean.FALSE;
        }
        UserInfoDTO userCache = userService.getUserCache(userVisitParam.getVirtualAdminId());
        if (userCache == null || userCache.getState().equals(AppConst.USER_STATE.VIRTUAL)) {
            log.error("马甲号管理员信息不存在 id：{}", userVisitParam.getVirtualAdminId());
            return Boolean.FALSE;
        }
        AdminVirtual adminVirtual = new AdminVirtual();
        adminVirtual.setAddTime(new Date());
        adminVirtual.setAdminType(AdminVirtualTypeEnum.APP.getType());
        adminVirtual.setAdminUserId(userVisitParam.getVirtualAdminId());
        adminVirtual.setVirtualUserId(userVisitParam.getVirtualId());
        adminVirtual.setId(sequenceCreater.nextLongId());
        adminVirtual.setOperateUserId(userVisitParam.getVirtualAdminId());
        adminVirtual.setOperateUserAccount(userCache.getNickname());
        adminVirtual.setAdminUserAccount(userCache.getNickname());
        int result = 0;
        try {
            result = adminVirtualMapper.insertSelective(adminVirtual);
        } catch (Exception e) {
            log.error("管理员添加马甲号失败 请求json：{}", JSON.toJSONString(userVisitParam), e);
        }
        if (result > 0) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }
}
