package com.bxm.localnews.user.account.impl.context;

import com.bxm.localnews.user.account.impl.callback.IAccountActionCallback;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.param.AccountActionParam;
import com.bxm.localnews.user.vo.CashFlow;
import com.bxm.localnews.user.vo.GoldFlow;
import com.bxm.localnews.user.vo.UserAccount;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import java.math.BigDecimal;
import java.util.Map;

/**
 * 用户转化信息转化与记录的上下文对象 用于承载处理过程中的相关信息
 * @author liujia
 * @date 2020/05/04 10:40
 */
public class AccountActionContext {

    public AccountActionContext(AccountActionParam param) {
        this.originParam = param;
    }

    /**
     * 当前操作的用户账号，用于获取数据
     */
    private UserAccount account;

    /**
     * 复制用户操作账号，需要落库的属性操作都操作此对象 方便后续使用Mybatis的空值判定对需要的更新的字段进行更新
     */
    private UserAccount cloneAccount;

    /**
     * 回调接口，调用方提供，用于账号处理的生命周期中自定义业务逻辑
     */
    private IAccountActionCallback callback;

    /**
     * 原始的请求参数，不可变更
     */
    private final AccountActionParam originParam;

    /**
     * 账号变更过程中产生的金币流水记录，有一些历史业务需要使用到
     * 可能一次账号变更，产生多个用户的流水
     * key : 用户ID
     * value : 该用户的流水
     */
    private Map<Long, GoldFlow> genGoldFlowMap;

    /**
     * 账号变更过程中产生的现金流水记录，有一些历史业务需要使用到
     * 可能一次账号变更，产生多个用户的流水
     * key : 用户ID
     * value : 该用户的流水
     */
    private Map<Long, CashFlow> genCashFlowMap;

    public void setAccount(UserAccount account) {
        this.account = account;
        this.cloneAccount = new UserAccount();

        cloneAccount.setId(account.getId());
        cloneAccount.setUserId(account.getUserId());
        cloneAccount.setVersion(account.getVersion());
    }

    public void putGoldFlow(Long userId, GoldFlow flow) {
        if (genGoldFlowMap == null) {
            genGoldFlowMap = Maps.newHashMap();
        }
        genGoldFlowMap.put(userId, flow);
    }

    public void putCashFlow(Long userId, CashFlow flow) {
        if (genCashFlowMap == null) {
            genCashFlowMap = Maps.newHashMap();
        }
        genCashFlowMap.put(userId, flow);
    }

    public GoldFlow getGoldFlow(Long userId) {
        if (null != genGoldFlowMap) {
            return genGoldFlowMap.get(userId);
        }
        return null;
    }

    public CashFlow getCashFlow(Long userId) {
        if (null != genCashFlowMap) {
            return genCashFlowMap.get(userId);
        }
        return null;
    }

    public Long getUserId() {
        return originParam.getUserId();
    }

    public AccountActionEnum getAction() {
        return originParam.getAction();
    }

    public Integer getGoldNum() {
        Integer goldNum = originParam.getGoldNum();
        //历史数据存在根据负数进行处理了逻辑，在新的处理逻辑中全部按照正数处理，这里做一下统一的纠正
        if (null != goldNum && goldNum < 0) {
            return Math.abs(goldNum);
        }
        return goldNum;
    }

    public BigDecimal getAmount() {
        BigDecimal amount = originParam.getAmount();
        //历史数据存在根据负数进行处理了逻辑，在新的处理逻辑中全部按照正数处理，这里做一下统一的纠正
        if (amount != null && BigDecimal.ZERO.compareTo(amount) > 0) {
            amount = amount.abs();
        }
        return amount;
    }

    public UserAccount getAccount() {
        return account;
    }

    public UserAccount getCloneAccount() {
        return cloneAccount;
    }

    public IAccountActionCallback getCallback() {
        return callback;
    }

    public void setCallback(IAccountActionCallback callback) {
        this.callback = callback;
    }

    public AccountActionParam getOriginParam() {
        return originParam;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.JSON_STYLE);
    }
}
