package com.bxm.localnews.user.account.impl.handler;

import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.account.impl.context.AccountActionContext;
import com.bxm.localnews.user.account.impl.rule.IRule;
import com.bxm.localnews.user.domain.UserAccountMapper;
import com.bxm.localnews.user.exception.UserAccountException;
import com.bxm.localnews.user.vo.UserAccount;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;

/**
 * 抽象的账号处理类，对一些公共的行为进行抽象
 *
 * @author liujia
 * @date 2020/05/04 14:34
 */
public abstract class AbstractAccountActionHandler implements IAccountActionHandler {

    private Map<Class<? extends IRule>, IRule> ruleMap;

    @Autowired
    protected UserAccountMapper userAccountMapper;

    protected UserAccountService userAccountService;

    @Autowired
    public void setRules(List<IRule> rules) {
        if (CollectionUtils.isNotEmpty(rules)) {
            ruleMap = Maps.newHashMap();
            rules.forEach(rule -> ruleMap.put(rule.getClass(), rule));
        }
    }

    @Override
    public final void handle(AccountActionContext context) {
        if (null == userAccountService) {
            userAccountService = SpringContextHolder.getBean(UserAccountService.class);
        }
        //前置处理，主要用来做校验
        preAction(context);
        //处理中，进行逻辑处理，数据入库，流水记录等
        postAction(context);
    }

    /**
     * 前置处理，执行绑定的验证规则，执行通过后调用自定义回调
     *
     * @param context 上下文信息
     */
    private void preAction(AccountActionContext context) {
        List<Class<? extends IRule>> bindRules = bindRules();

        if (bindRules != null) {
            for (Class<? extends IRule> ruleClass : bindRules) {
                IRule rule = ruleMap.get(ruleClass);

                if (!rule.apply(context)) {
                    throw new UserAccountException(
                            ruleClass.getSimpleName() + "验证失败",
                            rule.retryOnFailed());
                }
            }
        }

        //规则全部校验通过后执行自定义回调
        context.getCallback().preAction(context);
    }

    private void postAction(AccountActionContext context) {
        //执行具体的处理逻辑，负责将入库账号中的字段进行变更处理
        execAction(context, context.getAccount(), context.getCloneAccount());

        //记录流水信息
        saveFlow(context);

        //数据入库
        syncAccount(context);

        //处理完成后触发回调
        context.getCallback().postAction(context);

        // 正常处理完成后的回调，方便具体的子类扩展，默认不需要实现
        afterPost(context);

        //清理用户账号缓存
        userAccountService.cleanCache(context.getUserId());
    }

    protected void afterPost(AccountActionContext context) {

    }

    /**
     * 保存操作产生的流水信息
     *
     * @param context 上下文信息
     */
    protected abstract void saveFlow(AccountActionContext context);

    /**
     * 执行账号对应的业务动作的具体逻辑处理
     *
     * @param context 上下文信息
     * @param source  原始账号信息
     * @param storage 待入库信息
     */
    protected abstract void execAction(AccountActionContext context, UserAccount source, UserAccount storage);

    /**
     * 同步账号信息，将操作结果序列号到数据库
     * 乐观锁更新失败后，进行重试
     *
     * @param context 上下文
     */
    private void syncAccount(AccountActionContext context) {
        int result = userAccountMapper.updateUserAccountWithVersion(context.getCloneAccount());

        if (result <= 0) {
            throw new UserAccountException("version与数据库中的不一致，进行重试", true);
        }
    }
}
