package com.bxm.localnews.user.auth.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.auth.constant.AuthRedisKey;
import com.bxm.localnews.auth.enums.AuthCodeEnum;
import com.bxm.localnews.auth.enums.RoleCodeEnum;
import com.bxm.localnews.mq.common.constant.UserEventEnum;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.localnews.user.auth.UserAuthCodeService;
import com.bxm.localnews.user.domain.auth.UserAuthCodeMapper;
import com.bxm.localnews.user.domain.auth.UserAuthIdMapper;
import com.bxm.localnews.user.domain.auth.UserRoleAuthMapper;
import com.bxm.localnews.user.domain.auth.UserRoleMapper;
import com.bxm.localnews.user.dto.auth.UserAuthResourceDto;
import com.bxm.localnews.user.param.RemoveAuthCodeParam;
import com.bxm.localnews.user.timer.RemoveUserAuthTask;
import com.bxm.localnews.user.vo.auth.UserAuthCodeBean;
import com.bxm.localnews.user.vo.auth.UserAuthIdBean;
import com.bxm.localnews.user.vo.auth.UserRoleAuthBean;
import com.bxm.localnews.user.vo.auth.UserRoleBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTask;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
public class UserAuthCodeServiceImpl extends BaseService implements UserAuthCodeService {

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserAuthCodeMapper userAuthCodeMapper;

    private final UserAuthIdMapper userAuthIdMapper;

    private final UserRoleAuthMapper userRoleAuthMapper;

    private final UserRoleMapper userRoleMapper;

    private final MessageSender messageSender;

    private final ScheduleService scheduleService;

    private final RemoveUserAuthTask removeUserAuthTask;


    private TypeReference<List<String>> typeReference = new TypeReference<List<String>>() {
    };

    @Autowired
    public UserAuthCodeServiceImpl(RedisHashMapAdapter redisHashMapAdapter,
                                   UserAuthCodeMapper userAuthCodeMapper,
                                   UserAuthIdMapper userAuthIdMapper,
                                   UserRoleAuthMapper userRoleAuthMapper,
                                   UserRoleMapper userRoleMapper,
                                   MessageSender messageSender,
                                   ScheduleService scheduleService,
                                   RemoveUserAuthTask removeUserAuthTask) {
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userAuthCodeMapper = userAuthCodeMapper;
        this.userAuthIdMapper = userAuthIdMapper;
        this.userRoleAuthMapper = userRoleAuthMapper;
        this.userRoleMapper = userRoleMapper;
        this.messageSender = messageSender;
        this.scheduleService = scheduleService;
        this.removeUserAuthTask = removeUserAuthTask;
    }

    @Override
    public Long[] getAuthCombineCode(Long userId) {
        String combineCode = redisHashMapAdapter.get(AuthRedisKey.USER_AUTH_CODE, userId.toString(), String.class);

        Long[] authCodeArray;

        if (null == combineCode) {
            authCodeArray = new Long[]{0L};
        } else {
            String[] authCodeStrArray = combineCode.split(",");
            authCodeArray = new Long[authCodeStrArray.length];

            for (int i = 0; i < authCodeStrArray.length; i++) {
                authCodeArray[i] = Long.valueOf(authCodeStrArray[i]);
            }
        }

        return authCodeArray;
    }

    @Override
    public List<UserAuthResourceDto> getAuthResources(Long userId) {
        Map<String, List<String>> authResourceMap = redisHashMapAdapter.entries(buildKey(userId), typeReference);

        if (null == authResourceMap) {
            authResourceMap = reloadResourceIdCache(userId);
        }

        return authResourceMap.entrySet().stream().map(entry ->
                new UserAuthResourceDto(Integer.valueOf(entry.getKey()), entry.getValue()))
                .collect(Collectors.toList());
    }

    /**
     * 重新加载权限ID缓存数据
     *
     * @param userId 用户ID
     * @return 缓存数据
     */
    private Map<String, List<String>> reloadResourceIdCache(Long userId) {
        Map<String, List<String>> cacheMap = Maps.newHashMap();

        List<UserAuthIdBean> userAuthIdBeans = userAuthIdMapper.queryAuthIds(userId);

        userAuthIdBeans.forEach(item -> {
            String key = item.getAuthCode() + "";
            List<String> resourceIds = cacheMap.get(key);
            if (null == resourceIds) {
                resourceIds = Lists.newArrayList();
            }
            resourceIds.add(item.getAuthId());
            cacheMap.put(key, resourceIds);
        });

        KeyGenerator key = buildKey(userId);
        redisHashMapAdapter.remove(key);
        redisHashMapAdapter.putAll(key, cacheMap);

        return cacheMap;
    }

    /**
     * 添加用户事件
     *
     * @param userId 用户ID
     */
    private void sendUserReloadEvent(Long userId) {
        messageSender.sendUserEvent(PushPayloadInfo.build()
                .setType(UserEventEnum.AUTH_CHANGE.type)
                .addExtend("userId", userId));
    }

    @Override
    public Message addRole(Long userId, RoleCodeEnum role) {
        Preconditions.checkArgument(null != role);

        //设置用户拥有的角色
        UserRoleBean roleBean = new UserRoleBean();
        roleBean.setId(nextId());
        roleBean.setCreateTime(new Date());
        roleBean.setRoleCode(role.name());
        roleBean.setUserId(userId);

        userRoleMapper.insert(roleBean);

        //获取角色对应的操作权限列表
        List<UserRoleAuthBean> authBeans = userRoleAuthMapper.queryByCode(role.name());
        //设置用户拥有的权限集合
        Long[] authCombineCode = getAuthCombineCode(userId);
        for (UserRoleAuthBean userRoleAuth : authBeans) {
            AuthCodeEnum codeEnum = AuthCodeEnum.getByIndex(userRoleAuth.getAuthCode());
            authCombineCode = changeUserAuthCode(authCombineCode, codeEnum, true);
        }

        Message message = saveAuthCode(authCombineCode, userId);

        sendUserReloadEvent(userId);
        return message;
    }

    @Override
    public Message removeRole(Long userId, RoleCodeEnum role) {
        Preconditions.checkArgument(null != role);
        Preconditions.checkArgument(null != userId);

        //删除用户拥有的角色
        userRoleMapper.removeUserRole(userId, role.name());

        //获取角色对应的操作权限列表
        List<UserRoleAuthBean> authBeans = userRoleAuthMapper.queryByCode(role.name());
        //设置用户拥有的权限集合
        Long[] authCombineCode = getAuthCombineCode(userId);
        for (UserRoleAuthBean userRoleAuth : authBeans) {
            AuthCodeEnum codeEnum = AuthCodeEnum.getByIndex(userRoleAuth.getAuthCode());
            authCombineCode = changeUserAuthCode(authCombineCode, codeEnum, false);
        }

        //移除用户拥有的权限-资源信息
        MybatisBatchBuilder.create(UserAuthIdMapper.class, authBeans).run((mapper, auth) ->
                mapper.removeAuth(userId, auth.getAuthCode()));

        reloadResourceIdCache(userId);

        sendUserReloadEvent(userId);

        return saveAuthCode(authCombineCode, userId);
    }

    @Override
    public Message addAuthCode(Long userId, AuthCodeEnum authCode) {
        Long[] authCombineCode = getAuthCombineCode(userId);
        authCombineCode = changeUserAuthCode(authCombineCode, authCode, true);

        return saveAuthCode(authCombineCode, userId);
    }

    private Message saveAuthCode(Long[] authCodeArray, Long userId) {
        String authCombineCode = StringUtils.join(authCodeArray, ",");
        redisHashMapAdapter.put(AuthRedisKey.USER_AUTH_CODE, userId.toString(), authCombineCode);

        int rows;
        UserAuthCodeBean entity = userAuthCodeMapper.queryByUserId(userId);
        if (entity == null) {
            entity = new UserAuthCodeBean();
            entity.setId(nextId());
            entity.setCreateTime(new Date());
            entity.setAuthCombineCode(authCombineCode);
            entity.setUserId(userId);

            rows = userAuthCodeMapper.insert(entity);
        } else {
            entity.setAuthCombineCode(authCombineCode);
            entity.setModifyTime(new Date());

            rows = userAuthCodeMapper.updateByPrimaryKey(entity);
        }

        sendUserReloadEvent(userId);

        return Message.build(rows);
    }

    /**
     * 变更用户持有的权限合集
     *
     * @param authCombineCode 权限编码合集
     * @param authCode        变更的授权编码
     * @param isAdd           true表示添加，false表示移除
     * @return 变更后的权限合集
     */
    private Long[] changeUserAuthCode(Long[] authCombineCode, AuthCodeEnum authCode, boolean isAdd) {
        return BitOperatorUtil.setBitToArray(authCombineCode, authCode.index, isAdd);
    }

    @Override
    public Message removeAuthCode(Long userId, AuthCodeEnum authCode) {
        Long[] authCombineCode = getAuthCombineCode(userId);
        authCombineCode = changeUserAuthCode(authCombineCode, authCode, false);

        userAuthIdMapper.removeAuth(userId, authCode.index);
        redisHashMapAdapter.remove(buildKey(userId), authCode.index + "");

        return saveAuthCode(authCombineCode, userId);
    }

    @Override
    public Message addResource(Long userId, AuthCodeEnum authCode, String resourceId) {
        return addResource(userId, authCode, ImmutableList.of(resourceId));
    }

    @Override
    public Boolean hasAuth(Long userId, AuthCodeEnum codeEnum) {
        int authCode = codeEnum.index;
        Long[] parseAuthCodeArray = getAuthCombineCode(userId);

        return BitOperatorUtil.hasBit(parseAuthCodeArray, authCode);
    }

    @Override
    public Message removeResource(Long userId, AuthCodeEnum authCode, String resourceId) {
        KeyGenerator key = buildKey(userId);
        String subKey = String.valueOf(authCode.index);

        List<String> resources = redisHashMapAdapter.get(key, subKey, typeReference);

        if (null != resources) {
            resources.remove(resourceId);
            redisHashMapAdapter.put(key, subKey, resources);
        }

        Message message = Message.build(userAuthIdMapper.removeAuthId(userId, authCode.index, resourceId));
        sendUserReloadEvent(userId);

        return message;
    }

    @Override
    public Message addResource(Long userId, AuthCodeEnum authCode, List<String> resources) {
        Preconditions.checkArgument(null != resources);

        KeyGenerator key = buildKey(userId);
        String subKey = String.valueOf(authCode.index);

        List<String> existsResources = redisHashMapAdapter.get(key, subKey, typeReference);

        if (null == existsResources) {
            existsResources = Lists.newArrayList();
        }
        existsResources.addAll(resources);

        redisHashMapAdapter.put(key, subKey, existsResources);

        List<UserAuthIdBean> userAuthIdBeans = Lists.newArrayList();
        Date now = new Date();
        for (String resource : resources) {
            UserAuthIdBean entity = new UserAuthIdBean();
            entity.setAuthCode(authCode.index);
            entity.setId(nextId());
            entity.setAuthId(resource);
            entity.setUserId(userId);
            entity.setCreateTime(now);

            userAuthIdBeans.add(entity);
        }

        boolean success = MybatisBatchBuilder.create(UserAuthIdMapper.class, userAuthIdBeans).run(UserAuthIdMapper::insert);

        sendUserReloadEvent(userId);

        return Message.build(success);
    }

    @Override
    public Message addRemoveAuthCodeTask(RemoveAuthCodeParam param) {
        OnceTask task = OnceTaskBuilder
                .builder(RemoveUserAuthTask.generateTaskName(param.getUserId()), param.getExpireTime(), removeUserAuthTask)
                .callbackParam(param).build();
        return Message.build(scheduleService.push(task));
    }

    @Override
    public Message removeAuthCodeTask(Long userId) {
        logger.debug("删除移除用户权限定时任务成功,taskName:[{}]", RemoveUserAuthTask.generateTaskName(userId));
        return Message.build(scheduleService.remove(RemoveUserAuthTask.generateTaskName(userId)));
    }


    private KeyGenerator buildKey(Long userId) {
        return AuthRedisKey.USER_AUTH_RESOURCE.copy().appendKey(userId.toString());
    }
}
