/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.user.invite.bind;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.domain.UserInviteHistoryMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.UserInviteBindDTO;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.InviteBindMethodEnum;
import com.bxm.localnews.user.facade.ForumPostFacadeService;
import com.bxm.localnews.user.facade.MerchantFeignService;
import com.bxm.localnews.user.invite.InviteRecordService;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.vo.ForumPostVo;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserInviteHistoryBean;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Objects;

/**
 * @author : jieli.xu
 * @date 2020/7/13 11:51
 * 不同方式绑定邀请关系的处理方式
 */
@Component
@Slf4j
public abstract class AbstractBindInviteStrategy {

    @Autowired
    private UserInviteHistoryMapper userInviteHistoryMapper;

    @Autowired
    private UserAttributeService userAttributeService;

    @Autowired
    private ForumPostFacadeService forumPostFacadeService;

    @Autowired
    private UserMapper userMapper;

    @Autowired
    private UserService userService;

    @Autowired
    private InviteRecordService inviteRecordService;

    @Autowired
    private MerchantFeignService merchantFeignService;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    private UserInviteService userInviteService;


    private UserInviteService getUserInviteService() {
        if (userInviteService == null) {
            userInviteService = SpringContextHolder.getBean(UserInviteService.class);
        }
        return userInviteService;
    }

    /**
     * 公共检查方法
     * 初始化用户信息以及初始化用户的邀请关系
     *
     * @param userInviteBindDTO ： userId肯定不为空
     * @return ： 是否通过校验
     */
    private Message preCheckCommon(UserInviteBindDTO userInviteBindDTO) {
        //查看用户关系中的inviteUsers是否为空
        if (null == userInviteBindDTO.getUser()) {
            userInviteBindDTO.setUser(userMapper.selectByPrimaryKey(userInviteBindDTO.getUserId()));
            if (null == userInviteBindDTO.getInviteUserId()) {
                userInviteBindDTO.setInviteUserId(userInviteBindDTO.getUser().getInviteUserId());
            }
        }
        //校验徒弟和师傅不能是同一个人
        if (null == userInviteBindDTO.getInviteUserId()
                || Objects.equals(userInviteBindDTO.getUserId(), userInviteBindDTO.getInviteUserId())) {
            return Message.build(false);
        }

        //是否应存在记录
        UserInviteHistoryBean inviteBean = userInviteHistoryMapper.selectByUserId(userInviteBindDTO.getUserId());
        if (null == inviteBean) {
            userInviteBindDTO.setAlreadyHas(false);
            this.setInviteBean(userInviteBindDTO);
        } else {
            userInviteBindDTO.setAlreadyHas(true);
            userInviteBindDTO.setUserInviteHistoryBean(inviteBean);
            //师徒关系已确定，不走后续逻辑
            if (Objects.equals(inviteBean.getState(), AppConst.USER_STATE.NORMAL)) {
                //有些已经存在的关系，需要更新VIP状态或者其他操作
                this.beNormalInviteAfter(userInviteBindDTO);
                return Message.build(false);
            }

        }
        return Message.build(true);
    }

    /**
     * 参数检查与基础赋值操作
     *
     * @param userInviteBindDTO ： 绑定关系的入参
     * @return ：成功或者失败，失败不需要继续往下走
     */
    protected abstract Message preInvoke(UserInviteBindDTO userInviteBindDTO);

    /**
     * 徒弟从未激活update为已激活状态的方法
     * 绑定师徒关系以后的后续操作
     *
     * @param userInviteBindDTO
     * @return : 处理结果
     */
    protected Message afterInviteNormal(UserInviteBindDTO userInviteBindDTO) {
        //删除假VIP弹窗
        redisHashMapAdapter.remove(UserRedisKey.ACTIVE_VIP_HAS_NOT_CHANCE, String.valueOf(userInviteBindDTO.getUserId()));
        //增加推广金
        merchantFeignService.insertMerchantInvite(userInviteBindDTO.getUserInviteHistoryBean());
        return Message.build(true);
    }

    /**
     * 徒弟insert的是待激活状态
     *
     * @param userInviteBindDTO
     * @return 处理结果
     */
    protected Message afterInviteNotActive(UserInviteBindDTO userInviteBindDTO) {
        //设置用户的师徒信息
        User user = new User();
        user.setId(userInviteBindDTO.getUser().getId());
        user.setInviteUserId(userInviteBindDTO.getInviteUserId());
        userMapper.updateByPrimaryKeySelective(user);
        userService.loadUserToRedis(user.getId());
        return Message.build(true);
    }

    /**
     * 徒弟直接是insert的激活状态
     *
     * @param userInviteBindDTO
     * @return
     */
    private Message afterInsertInviteNormal(UserInviteBindDTO userInviteBindDTO) {
        afterInviteNotActive(userInviteBindDTO);
        return afterInviteNormal(userInviteBindDTO);
    }

    /**
     * 具体的执行绑定的方法
     *
     * @param userInviteBindDTO ： 绑定参数入参
     * @return : 返回绑定的结果
     */
    public final UserInviteHistoryBean doBindInvite(UserInviteBindDTO userInviteBindDTO) {
        //公共校验参数是否合法
        Message message = preCheckCommon(userInviteBindDTO);

        if (!message.isSuccess()) {
            if (null != userInviteBindDTO.getUserInviteHistoryBean() && userInviteBindDTO.getAlreadyHas()) {
                return userInviteBindDTO.getUserInviteHistoryBean();
            }
            return null;
        }
        //参数校验与基础赋值
        message = this.preInvoke(userInviteBindDTO);
        if (message.isSuccess()) {
            //补充师傅的师傅相关信息
            this.replenishInviteHistory(userInviteBindDTO.getUserInviteHistoryBean());
            //新增或者修改用户邀请关系
            if (getUserInviteService().insertOrUpdate(userInviteBindDTO)) {
                //后置方法:修改和新增的后续操作不同，新增直接为normal和新增直接为notactive也不同
                if (userInviteBindDTO.getAlreadyHas()) {//修改
                    this.afterInviteNormal(userInviteBindDTO);
                } else {//是新增
                    if (Objects.equals(AppConst.USER_STATE.NORMAL, userInviteBindDTO.getUserInviteHistoryBean().getState())) {
                        //直接是正常
                        this.afterInsertInviteNormal(userInviteBindDTO);
                    } else if (Objects.equals(AppConst.USER_STATE.NOT_ACTIVE, userInviteBindDTO.getUserInviteHistoryBean().getState())) {
                        //还有待激活
                        this.afterInviteNotActive(userInviteBindDTO);
                    }
                }
                return userInviteBindDTO.getUserInviteHistoryBean();
            }
        }
        return null;
    }

    protected abstract InviteBindMethodEnum getBindMethod();

    /**
     * 已经是确定关系的后续操作：支付后的需要更新状态
     *
     * @param userInviteBindDTO
     */
    protected void beNormalInviteAfter(UserInviteBindDTO userInviteBindDTO) {
        //默认啥也不做，需要做的子类自己实现
    }

    /**
     * * 补充邀请人信息
     *
     * @param entity 邀请历史记录bean
     */
    protected void replenishInviteHistory(UserInviteHistoryBean entity) {
        if (null == entity.getInviteUserId()) {
            return;
        }

        log.info("能否进入到创作者的信息[{}]", null != entity.getRelationId() && StringUtils.equals(entity.getType(), InviteTypeEnum.FORUM.getName()));

        //设置上级创作者（双重邀请关系），目前只有帖子的话才会有创作者
        if (null != entity.getRelationId() && StringUtils.equals(entity.getType(), InviteTypeEnum.FORUM.getName())) {
            ForumPostVo forumPostVo = forumPostFacadeService.getForumPostById(entity.getRelationId()).getBody();

            log.info("进来，并且获取到信息：" + JSONObject.toJSONString(forumPostVo));

            if (null != forumPostVo && null != forumPostVo.getUserId()) {
                entity.setCreatedUserId(forumPostVo.getUserId());
                entity.setRelationTitle(StringUtils.substring(forumPostVo.getTitle(), 0, 20));
            }
        }


        // 获取上级用户的邀请信息，用于设置上上级信息
        UserInviteHistoryBean inviteHistoryBean = userInviteHistoryMapper.selectByUserId(entity.getInviteUserId());
        if (null == inviteHistoryBean || null == inviteHistoryBean.getUserId()) {
            return;
        }
        //设置上上级用户（二级关系）
        entity.setInviteSuperUserId(inviteHistoryBean.getInviteUserId());
        // 设置上上级创建者（二级关系）
        entity.setInviteCreatedUserId(inviteHistoryBean.getCreatedUserId());

    }

    /**
     * 初始化inviteHistoryBean的信息
     *
     * @param userInviteBindDTO
     */
    private void setInviteBean(UserInviteBindDTO userInviteBindDTO) {
        //补充完善邀请关系
        UserInviteHistoryBean bean = new UserInviteHistoryBean();
        bean.setUserId(userInviteBindDTO.getUserId());
        bean.setInviteUserId(userInviteBindDTO.getInviteUserId());
        bean.setRelationId(userInviteBindDTO.getUser().getRelationId());
        bean.setRelationTitle(userInviteBindDTO.getRelationTitle());
        bean.setCreateTime(new Date());
        bean.setInviteHeadImg(userInviteBindDTO.getUser().getHeadImg());
        bean.setInviteUserNickname(userInviteBindDTO.getUser().getNickname());
        bean.setType(this.getBindMethod().name());
        bean.setState(userInviteBindDTO.getUser().getState());
        bean.setBindTime(new Date());
        userInviteBindDTO.setUserInviteHistoryBean(bean);
    }
}
