package com.bxm.localnews.user.invite.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.constant.CommonConstants;
import com.bxm.localnews.user.domain.UserInviteHistoryMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.InviteBindMethodEnum;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.invite.InviteProcessService;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.invite.bind.BindInviteManager;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.dto.InviteCounterDTO;
import com.bxm.localnews.user.model.param.UserInviteConterParam;
import com.bxm.localnews.user.param.UserInvitePageParam;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.support.UserRankService;
import com.bxm.localnews.user.support.VirtualUserService;
import com.bxm.localnews.user.utils.NumberConvertUtils;
import com.bxm.localnews.user.vo.Tuple;
import com.bxm.localnews.user.vo.UserInviteHistoryBean;
import com.bxm.localnews.user.vo.UserInviteOrderVO;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

@Service
public class UserInviteServiceImpl extends BaseService implements UserInviteService {

    private final UserInviteHistoryMapper userInviteHistoryMapper;

    private InviteProcessService inviteProcessService;

    private UserAttributeService userAttributeService;

    private final UserProperties userProperties;

    private final BindInviteManager bindInviteManager;

    private final UserRankService userRankService;

    private UserService userService;

    private final LocationIntegrationService locationIntegrationService;

    private final VirtualUserService virtualUserService;

    @Autowired
    public UserInviteServiceImpl(UserInviteHistoryMapper userInviteHistoryMapper,
                                 UserAttributeService userAttributeService,
                                 UserProperties userProperties,
                                 BindInviteManager bindInviteManager,
                                 UserRankService userRankService,
                                 LocationIntegrationService locationIntegrationService,
                                 VirtualUserService virtualUserService) {
        this.userInviteHistoryMapper = userInviteHistoryMapper;
        this.userAttributeService = userAttributeService;
        this.userProperties = userProperties;
        this.bindInviteManager = bindInviteManager;
        this.userRankService = userRankService;
        this.locationIntegrationService = locationIntegrationService;
        this.virtualUserService = virtualUserService;
    }

    private InviteProcessService getInviteProcessService() {
        if (inviteProcessService == null) {
            inviteProcessService = SpringContextHolder.getBean(InviteProcessService.class);
        }
        return inviteProcessService;
    }

    public UserService getUserService() {
        if (userService == null) {
            userService = SpringContextHolder.getBean(UserService.class);
        }

        return userService;
    }

    @Override
    public UserInviteCollectDTO getInviteCollect(Long userId) {
        UserInvitePageParam param = new UserInvitePageParam();
        param.setUserId(userId);
        param.setType(1);

        UserInviteCollectDTO result = getCollect(userId);
        result.setFirstPage(getInviteByPage(param));

        return result;
    }

    /**
     * 获取用户邀请徒弟的统计信息，包括今日、昨日、总共的数量
     *
     * @param userId 当前用户
     * @return 统计结果
     */
    private UserInviteCollectDTO getCollect(Long userId) {
        // 获取最近两天的收徒数据，包括实际的师徒、潜在的师徒
        List<Tuple> collectList = userInviteHistoryMapper.getCollect(userId);


        String todayFormat = DateUtils.formatDate(new Date());

        Calendar yesterdayCalendar = Calendar.getInstance();
        yesterdayCalendar.setTime(new Date());
        yesterdayCalendar.add(Calendar.DAY_OF_YEAR, -1);
        String yesterdayFormat = DateUtils.formatDate(yesterdayCalendar.getTime());

        int total = 0, today = 0, yesterday = 0;

        for (Tuple tuple : collectList) {
            total += tuple.getTotal();

            if (StringUtils.equals(todayFormat, tuple.getLabel())) {
                today = tuple.getTotal();
            }
            if (StringUtils.equals(yesterdayFormat, tuple.getLabel())) {
                yesterday = tuple.getTotal();
            }
        }

        // 统计一级、二级的有效徒弟
        Integer levelOne = userInviteHistoryMapper.getCollectLevel(userId, 1);
        Integer levelTwo = userInviteHistoryMapper.getCollectLevel(userId, 2);


        UserInviteCollectDTO result = new UserInviteCollectDTO();
        result.setTodayTotal(today);
        result.setYesterdayTotal(yesterday);
        result.setTotal(total);
        result.setOneLevel(levelOne);
        result.setTwoLevel(levelTwo);

        // 收徒总数包括潜在徒弟 - 一级有效徒弟 - 二级有效徒弟 = 潜在徒弟
        int nonActivated = total - levelOne - levelTwo;
        result.setNonActivated(nonActivated < 0 ? 0 : nonActivated);
        return result;
    }

    @SuppressWarnings("unchecked")
    @Override
    public PageWarper<UserInviteDTO> getInviteByPage(UserInvitePageParam param) {
        List<UserInviteHistoryBean> histories = userInviteHistoryMapper.queryByPage(param);

        PageWarper page = new PageWarper(histories);
        page.setList(convert(param.getUserId(), histories));
        return page;
    }

    @Override
    public List<String> getLastInviteUserHeadImgList(Long userId) {
        return userInviteHistoryMapper.getLastInviteHeadImg(userId, 5);
    }

    private List<UserInviteDTO> convert(Long userId, List<UserInviteHistoryBean> histories) {
        List<UserInviteDTO> result = new ArrayList<>(histories.size());

        histories.forEach(item -> result.add(UserInviteDTO.builder()
                .activeVip(Objects.equals(item.getActiveVip(), (byte) 1))
                .inviteDate(item.getCreateTime())
                .inviteHeadImg(item.getInviteHeadImg())
                .inviteNickName(item.getInviteUserNickname())
                .inviteUserId(item.getUserId())
                .relationTitle(getInviteProcessService().getLevelTwoFinalInviteTitle(userId, item))
                .build()));

        return result;
    }

    @Override
    public UserInviteHistoryBean selectByUserId(Long userId) {
        UserInviteHistoryBean bean = userInviteHistoryMapper.selectByUserId(userId);
        return null == bean ? new UserInviteHistoryBean() : bean;
    }

    /**
     * 补充师傅新增收徒的数量
     */
    private void addInviteNum(UserInviteHistoryBean bean) {
        if (null == bean.getUserId() || null == bean.getInviteUserId()) {
            return;
        }
        // B = inviteUser(邀请人)
        // C = inviteSuperUser（上上级（邀请人的上级）） ,
        // D = createUserId （当前用户的内容邀请人（通过哪一条内容进行的注册））
        // E = inviteCreate （上级的内容邀请人，也可理解为内容维度的上上级）

        // 更新上级邀请人数
        if (null != bean.getInviteUserId()) {
            userAttributeService.addInviteNum(bean.getInviteUserId());
        }

        // 更新上上级邀请人数
        if (null != bean.getInviteSuperUserId()) {
            userAttributeService.addInviteNum(bean.getInviteSuperUserId());
        }

        // E 不等于D/C
        if (null != bean.getInviteCreatedUserId()
                && !bean.getInviteCreatedUserId().equals(bean.getCreatedUserId())
                && !bean.getInviteCreatedUserId().equals(bean.getInviteSuperUserId())) {
            userAttributeService.addInviteNum(bean.getInviteCreatedUserId());
        }
        // D 不等于B/C/E
        if (null != bean.getCreatedUserId()
                && !bean.getCreatedUserId().equals(bean.getInviteUserId())
                && !bean.getCreatedUserId().equals(bean.getInviteSuperUserId())
                && !bean.getCreatedUserId().equals(bean.getInviteCreatedUserId())) {
            userAttributeService.addInviteNum(bean.getCreatedUserId());
        }
    }

    @Override
    public boolean insertOrUpdateUserInviteNotActive(Long userId, Long inviteUserId) {
        UserInviteBindDTO userInviteBindDTO = new UserInviteBindDTO();
        userInviteBindDTO.setIsCallback(false);
        userInviteBindDTO.setUserId(userId);
        userInviteBindDTO.setInviteUserId(inviteUserId);
        userInviteBindDTO.setInviteBindMethodEnum(InviteBindMethodEnum.WELFARE_VIP);
        bindInviteManager.bindInvite(userInviteBindDTO);
        return true;
    }


    /**
     * 新增或者更新的操作 --
     */
    @Override
    public Boolean insertOrUpdate(UserInviteBindDTO userInviteBindDTO) {
        if (!userInviteBindDTO.getAlreadyHas()) {
            logger.info("新增邀请关建立[{}]", JSONObject.toJSONString(userInviteBindDTO.getUserInviteHistoryBean()));
            if (null == userInviteBindDTO.getUserInviteHistoryBean().getId()) {
                userInviteBindDTO.getUserInviteHistoryBean().setId(nextId());
            }

            if (Objects.equals(userInviteBindDTO.getUserInviteHistoryBean().getState(), AppConst.USER_STATE.NORMAL)) {
                userInviteBindDTO.getUserInviteHistoryBean().setBindTime(new Date());
            }

            addInviteNum(userInviteBindDTO.getUserInviteHistoryBean());

            return userInviteHistoryMapper.insert(userInviteBindDTO.getUserInviteHistoryBean()) > 0;
        } else {
            if (userInviteBindDTO.getNeedUpdate()) {
                logger.info("修改邀请关系[{}]", JSONObject.toJSONString(userInviteBindDTO));
                userInviteHistoryMapper.updateInviteSelective(userInviteBindDTO.getUserInviteHistoryBean());
            } else {
                return false;
            }
        }
        return true;
    }

    @Override
    public List<UserInviteOrderVO> getUserInviteOrderTen(Integer num, String locationCode, List<Long> userIds) {
        if (null == num || num > CommonConstants.INVITE_ORDER_NUM_LIMIT) {
            num = CommonConstants.INVITE_ORDER_NUM_LIMIT;
        }
        List<UserInviteOrderVO> list = userInviteHistoryMapper.getInviteOrderTen(num, locationCode, userIds);
        for (UserInviteOrderVO entity : list) {
            if (StringUtils.isEmpty(entity.getHeadImg())) {
                entity.setHeadImg(userProperties.getDefaultHeadImgUrl());
            }
        }
        return list;
    }

    @Override
    public List<UserInviteHistoryBean> getListByUid(Long userId) {
        return userInviteHistoryMapper.queryListAllByUid(userId);
    }

    @Override
    public InviteCounterDTO getInviteCounter(UserInviteConterParam param) {
        InviteCounterDTO inviteCounterDTO = new InviteCounterDTO();

        Long total = userRankService.getTotal(param.getAreaCode());
        Long inviteNumRank = userRankService.getInviteNumRank(param.getUserId(), param.getAreaCode());

        UserInfoDTO userCache = getUserService().getUserCache(param.getUserId());

        Long inviteNum;

        if (userCache.getInviteNum() == null) {
            inviteNum = 0L;
        } else {
            inviteNum = Long.valueOf(userCache.getInviteNum());
        }

        inviteCounterDTO.setTotal(inviteNum);
        inviteCounterDTO.setPercent(NumberConvertUtils.formatInvitePercent(inviteNumRank, total, inviteNum));

        // 滚动弹幕
        inviteCounterDTO.setScollNotes(getScrollNotes(param.getAreaCode()));
        return inviteCounterDTO;
    }

    /**
     * 获取某一个地区的收徒信息滚动信息
     *
     * @param areaCode 地区编码
     * @return 地区的滚动信息
     */
    private List<String> getScrollNotes(String areaCode) {
        List<String> scrollMessages = Lists.newArrayList();


        int queryNum = userProperties.getInviteScrollList().size();

        if (queryNum == 0) {
            return scrollMessages;
        }

        LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);
        List<UserInfoDTO> virtualUsers = virtualUserService.listVirtualUser(queryNum);

        int index = 0;

        for (String message : userProperties.getInviteScrollList()) {
            UserInfoDTO virtualUser = virtualUsers.get(index);

            String scrollMsg = StringUtils.replace(message, "{areaName}", locationDTO.getName());
            scrollMsg = StringUtils.replace(scrollMsg, "{userName}", virtualUser.getNickname());

            scrollMessages.add(scrollMsg);

            index++;
        }

        return scrollMessages;
    }

    @Override
    public void changeHistoryStatus(Long userId, Byte state) {
        UserInviteHistoryBean userInviteHistoryBean = userInviteHistoryMapper.selectByUserId(userId);

        if (userInviteHistoryBean != null) {
            if (!Objects.equals(userInviteHistoryBean.getState(), state)) {
                logger.warn("用户邀请状态异常，经过绑定逻辑后仍错误，用户ID：{}", userId);
            }

            UserInviteHistoryBean modifyBean = new UserInviteHistoryBean();
            modifyBean.setUserId(userInviteHistoryBean.getUserId());
            modifyBean.setState(state);

            userInviteHistoryMapper.updateInviteSelective(modifyBean);
        }
    }

    @Override
    public UserInviteOrderVO getInviteNumByUserId(Long userId) {
        Tuple tuple = userInviteHistoryMapper.getCollectAll(userId);
        if (null != tuple) {
            return UserInviteOrderVO.builder()
                    .userId(userId)
                    .inviteNum(tuple.getTotal())
                    .build();
        }
        return UserInviteOrderVO.builder()
                .userId(userId)
                .inviteNum(0)
                .build();
    }
}
