package com.bxm.localnews.user.invite.processor;

import com.bxm.localnews.auth.enums.AuthCodeEnum;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.user.auth.UserAuthCodeService;
import com.bxm.localnews.user.domain.UserEquipmentMapper;
import com.bxm.localnews.user.dto.LocationUserInfoDTO;
import com.bxm.localnews.user.dto.h5.ClientVersionDto;
import com.bxm.localnews.user.integration.BizLogIntegrationService;
import com.bxm.localnews.user.integration.IMIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.invite.InviteProcessorContext;
import com.bxm.localnews.user.vo.ImChatroom;
import com.bxm.localnews.user.vo.InviteRecord;
import com.bxm.localnews.user.vo.UserInviteHistoryBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 通过本地群聊室邀请的处理逻辑
 * @author liujia
 * @date 2020/05/11 19:13
 */
@Component
@Slf4j
public class ChatRoomInviteTypeProcessor extends AbstractInviteTypeProcessor {

    private UserAuthCodeService userAuthCodeService;

    private UserEquipmentMapper userEquipmentMapper;

    private IMIntegrationService imIntegrationService;

    private PushMsgIntegrationService pushMsgIntegrationService;

    private BizLogIntegrationService bizLogIntegrationService;

    @Autowired
    public ChatRoomInviteTypeProcessor(UserAuthCodeService userAuthCodeService,
                                       UserEquipmentMapper userEquipmentMapper,
                                       IMIntegrationService imIntegrationService,
                                       PushMsgIntegrationService pushMsgIntegrationService,
                                       BizLogIntegrationService bizLogIntegrationService) {
        this.userAuthCodeService = userAuthCodeService;
        this.userEquipmentMapper = userEquipmentMapper;
        this.imIntegrationService = imIntegrationService;
        this.pushMsgIntegrationService = pushMsgIntegrationService;
        this.bizLogIntegrationService = bizLogIntegrationService;
    }

    @Override
    public String obtainTitle(InviteProcessorContext context) {
        return "聊天室";
    }

    @Override
    public InviteTypeEnum support() {
        return InviteTypeEnum.CHATROOM;
    }

    @Override
    public String decorateTitle(UserInviteHistoryBean inviteHistory) {
        return "聊天室邀请";
    }

    /**
     * 聊天室邀请成功后需要给邀请人开通聊天室的相关权限
     * 上报相关埋点
     * @param context 邀请记录上下文信息
     * @return 回调处理结果
     */
    @Override
    Message callback(InviteProcessorContext context) {
        InviteRecord inviteRecord = context.getRecord();
        Long inviteUserId = inviteRecord.getUserId();

        log.info("解锁用户抢红包的权限,userId:[{}]", inviteUserId);

        // 邀请人版本低于3.0.0不解锁，也不发推送
        //TODO [liujia]获取设备信息优化
        ClientVersionDto clientVersionDto = userEquipmentMapper.queryClientVersion(inviteUserId);
        if (null != clientVersionDto && StringUtils.compareVersion(clientVersionDto.getVersion(), "3.0.0") >= 0) {
            //邀请关系建立时，邀请人不存在抢红包的权限，则添加权限、发送push
            if (!userAuthCodeService.hasAuth(inviteUserId, AuthCodeEnum.PICK_RED_PACKAGE)) {
                LocationUserInfoDTO location = userService.getLocationUserInfo(inviteUserId);

                //添加用户抢红包的权限
                userAuthCodeService.addAuthCode(inviteUserId, AuthCodeEnum.PICK_RED_PACKAGE);

                //只有邀请人所在地区开启了抢红包才发送推送
                ImChatroom chatroom = imIntegrationService.getChatRoom(location.getLocationCode());
                if (chatroom.getEnableTimingRedPacket() == 1) {
                    //发送PUSH,通知用户解锁成功
                    pushMsgIntegrationService.pushUnlockPickAuth(inviteUserId, context.getInvitedUser().getNickname(), location);
                }

                //上报数据埋点
                PointReportParam reportParam = PointReportParam.build(context.getBasicParam());
                reportParam.e("3034");
                reportParam.ev(StringUtils.join("109.", String.valueOf(inviteUserId)));
                reportParam.put("uid", String.valueOf(context.getInvitedUser().getId()));

                bizLogIntegrationService.point(reportParam);
            }
        }

        //上报埋点
        PointReportParam reportParam = PointReportParam.build(context.getBasicParam());
        reportParam.e("3034");
        reportParam.ev(StringUtils.join("110.", String.valueOf(inviteUserId)));
        reportParam.put("uid", String.valueOf(context.getInvitedUser().getId()));

        bizLogIntegrationService.point(reportParam);
        return Message.build();
    }
}
