package com.bxm.localnews.user.login;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import com.bxm.localnews.user.model.dto.RegisterUserCountDTO;
import com.bxm.localnews.user.param.TempUserParam;
import com.bxm.localnews.user.param.UserBindPhoneParam;
import com.bxm.localnews.user.param.UserClipboardParam;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserBindPhoneVO;
import com.bxm.newidea.component.vo.Message;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDate;
import java.util.Collection;
import java.util.List;

public interface UserService {

    /**
     * 封装了UserMapper的insert方法，方便统一在添加用户的时候做一些公共操作
     * @param user 添加的用户信息
     * @param basicParam 基础参数 用来做版本判断之类的
     */
    int createUser(User user, BasicParam basicParam);

    /**
     * 用途：从redis中取得用户扩展信息，如果redis中不存在，则在数据库中查找并更新至redis中
     */
    UserInfoDTO getUserCache(Long userId);

    /**
     * 移除用户缓存
     *
     * @param userId 用户ID
     */
    void removeUserCache(Long userId);

    /**
     * 用途：user->userInfoDTO
     */
    UserInfoDTO convertUserToDTO(User user);

    /**
     * 用途：重置用户token
     */
    Json<UserInfoDTO> refreshToken(Long userId, String refreshToken);

    /**
     * 用途：判断该手机号是否注册
     *
     * @param flag 标记是否验证微信绑定 0不用验证/1需要验证
     * @return true已存在绑定记录，不允许重复绑定
     */
    boolean checkUserExistByPhone(String phone, Integer flag);

    /**
     * 判断手机号是否存在已绑定的用户
     * @param phone 手机号
     * @return true已存在绑定的用户
     */
    boolean checkUserExistByPhone(String phone);

    /**
     * 获取用户定位区域、基本信息
     */
    LocationUserInfoDTO getLocationUserInfo(Long userId);

    /**
     * 从数据库读取用户信息
     */
    User selectByPrimaryKey(Long userId);

    /**
     * 判断unionId是否被使用
     */
    Boolean checkUnionIdIsUser(String unionId);

    /**
     * 创建临时用户
     */
    Message createTempUser(TempUserParam userParam);

    /**
     * 更新用户信息【万事通小程序】
     */
    Message updateTempUser(TempUserParam userParam);

    /**
     * 用户是否是临时用户
     */
    Boolean isTempUser(Long userId);

    /**
     * 用户是否是风险用户
     */
    Boolean isRiskUser(Long userId);


    /**
     * 用户是否是正常用户
     *
     * @param userId 用户id
     * @return true正常用户 false非正常用户
     */
    Boolean isNormalUser(Long userId);

    /**
     * 添加万事通小程序的静默用户
     */
    Long addWSTAppletUser();

    /**
     * 检查用户是否已经完善资料，如果已经完善则完成任务 地区编码位数问题或用户id为空
     */
    void checkCompleteInfo(User user);

    /**
     * 获取微信用户信息
     */
    Message getWxUserInfo(TempUserParam tempUserParam, HttpServletRequest request);

    /**
     * 取得用户缓存
     */
    UserInfoDTO loadUserToRedis(Long userId);

    /**
     * 更新用户缓存
     */
    UserInfoDTO putUserToRedis(User user);

    /**
     * 只获取用户的头像昵称信息
     */
    BaseUserInfo getNickNameAndHeadImg(Long userId);

    /**
     * 批量获取用户信息
     */
    List<UserInfoDTO> getBatchUserInfo(Collection<Long> userIds);

    /**
     * 检查手机号是否有风险
     */
    void isRisk(Long userId, String phone, BasicParam basicParam, String deviceToken, String ip);

    /**
     * 根据生日获取年龄段
     */
    String getGeneration(LocalDate birthday);

    /**
     * 根据淘宝客 relation id获取用户淘宝客信息
     *
     * @param relationId 淘宝客 relation id
     * @return 淘宝客信息
     */
    UserTbkInfo selectUserFromCacheByRelationId(String relationId);

    /**
     * 更新淘宝客绑定信息
     *
     * @param userId     userId
     * @param relationId relationId
     * @param specialId  specialId
     */
    void updateTbkInfo(Long userId, String relationId, String specialId);

    /**
     * 根据userId 获取用户淘宝客信息
     *
     * @param userId
     * @return
     */
    UserTbkInfo selectUserFromCacheByUserId(Long userId);

    /**
     * @param userId   : 用户id
     * @param userName : 用户名
     * @return 用户列表信息
     */
    List<UserBaseInfo> selectUserByIdName(Long userId, String userName);

    /**
     * 校验用户id和手机号是否存在
     *
     * @param phoneParam
     * @return
     */
    UserBindPhoneVO checkUserAndPhoneExists(UserBindPhoneParam phoneParam);

    /**
     * 绑定手机号码
     *
     * @param phoneParam
     * @return 是否绑定成功
     */
    UserBindPhoneVO bindUserPhone(UserBindPhoneParam phoneParam);

    /**
     * 更新用户标志位
     *
     * @param userId      userId
     * @param judgeMarker {@link LocalNewsUserJudgeMarkerEnum}
     * @param add         true 增加 false 移除
     */
    void updateUserJudgeMarker(Long userId, LocalNewsUserJudgeMarkerEnum judgeMarker, boolean add);


    /**
     * 获取评论排行  V3.3.0
     *
     * @param areaCode
     * @param limitNum
     * @return
     */
    List<InteractRankInfo> getInteractRankInfo(String areaCode, int limitNum, List<Long> userIds);

    /**
     * 获取个人评论数
     *
     * @param userId
     * @return
     */
    Integer replyNumForUser(Long userId);

    /**
     * 上传粘贴板方法
     *
     * @param param ： 粘贴板参数
     * @return
     */
    Boolean clipboardUpload(UserClipboardParam param);

    /**
     * 根据脱敏的手机号码获取用户信息
     *
     * @param sensitivePhone 脱敏的手机号码
     * @return 模糊匹配的用户信息列表
     */
    List<UserPhoneDTO> getUserInfoBySensitivePhone(String sensitivePhone);

    /**
     * 根据用户id 获取用户的最新定位信息
     *
     * @param userId userId
     * @return 用户的最新定位信息
     */
    UserLocationInfoDTO getUserLocationCodeById(Long userId);

    /**
     * 根据用户手机号码获取用户信息
     *
     * @param phone 手机号码
     * @return 用户id和电话号码信息
     */
    UserPhoneDTO getUserInfoByPhone(String phone);


    /**
     * 获取用户的注册信息
     * 注册时间、首次登录时间等等
     *
     * @param userId userId
     * @return 注册信息 如果用户不存在则返回null
     */
    UserRegisterInfoDTO userRegisterInfo(Long userId);

    /**
     * 批量删除用户的缓存
     * @param userIds ： 用户的id集合
     * @return 删除结果
     */
    Boolean delUserCacheBatch(List<Long> userIds);

    /**
     * 获取areaCode对应地区的用户注册数
     * 可以不用很精准
     * @param areaCode 地区编码
     * @return 对应的用户注册数
     */
    RegisterUserCountDTO getRegisterUserCount(String areaCode);
}