package com.bxm.localnews.user.login.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.msg.integration.SmsIntegrationService;
import com.bxm.localnews.user.account.CashFlowService;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.attribute.UserPersonalInfoService;
import com.bxm.localnews.user.auth.UserAuthCodeService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserEquipmentMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.EquipmentDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.exception.UserRepeatCreateException;
import com.bxm.localnews.user.integration.AppVersionIntegrationService;
import com.bxm.localnews.user.integration.BizIntegrationService;
import com.bxm.localnews.user.integration.BizLogIntegrationService;
import com.bxm.localnews.user.invite.InviteProcessService;
import com.bxm.localnews.user.login.*;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.param.InviteProcessParam;
import com.bxm.localnews.user.support.ChannelService;
import com.bxm.localnews.user.support.UserEquipmentService;
import com.bxm.localnews.user.timer.InviteRelationshipTask;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

/**
 * Created by mars on 2018/2/23. 登录
 */
@Service("loginService")
@RefreshScope
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class LoginServiceImpl extends BaseService implements LoginService {

    private final SmsIntegrationService smsIntegrationService;

    private final UserAuthMapper userAuthMapper;

    private final UserMapper userMapper;

    private final UserService userService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final RedisStringAdapter redisStringAdapter;

    private final AppVersionIntegrationService appVersionIntegrationService;

    private final UserAuthService userAuthService;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final UserLoginHistoryService userLoginHistoryService;

    private final UserAccountService userAccountService;

    private final CashFlowService cashFlowService;

    private final UserEquipmentService userEquipmentService;

    private final ChannelService channelService;

    private final BizIntegrationService bizIntegrationService;

    private final UserParamCheckService userParamCheckService;

    private final UserEquipmentMapper userEquipmentMapper;

    private final UserRegistrationService userRegistrationService;

    private final UserAuthCodeService userAuthCodeService;

    private final ScheduleService scheduleService;

    private final InviteProcessService inviteProcessService;

    private final UserVipService userVipService;

    @Override
    @Retryable(value = RetryException.class, backoff = @Backoff(300))
    public Message doLogin(LoginInfo loginInfo, BasicParam basicParam) {

        //根据类型找出授权信息
        UserAuth userAuth = this.userAuthMapper.selectByUserAuth(loginInfo.getType(), loginInfo.getLoginName());

        //根据登录类型进行判断，是否登录成功，如果没有这个用户就选择注册并直接返回
        Message message;
        try {
            message = preJudgeBeforeLogin(loginInfo, basicParam, userAuth);
        } catch (UserRepeatCreateException e) {
            logger.warn("重复创建用户，进行重试，请求参数：{}", loginInfo);
            throw new RetryException("用户已存在，重新创建");
        }
        if (!message.isSuccess()) {
            return message;
        } else {
            //判断是否及时退出
            Boolean isReturn = message.getParam("return");
            if (null != isReturn && isReturn) {
                return message;
            }
        }

        //更新用户信息
        LoginMeta meta = login(userAuth, loginInfo, basicParam);

        //数据埋点
        if (meta.getUser() != null) {
            bizLogIntegrationService.login(meta.getUser().getId(), basicParam.getPlatform());
        }

        return Message.build(true).addParam("meta", meta);
    }

    /**
     * 根据登录类型与auth判断是否存在用户
     */
    private Message preJudgeBeforeLogin(LoginInfo loginInfo, BasicParam basicParam, UserAuth userAuth) throws UserRepeatCreateException {
        //是否提审状态
        boolean isPublishState = appVersionIntegrationService.getPublishState(basicParam);
        //第三方登录,如果未绑定手机号码，提示进行绑定
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()) {
            if (null == userAuth) {
                if (!isPublishState) {
                    // 微信注册
                    Message meta = registerByWechat(loginInfo, basicParam);
                    return meta.addParam("return", true);
                    //提审时返回绑定手机号提示
                } else {
                    LoginMeta meta = new LoginMeta();
                    meta.setCode(RespCode.BIND_PHONE);
                    return Message.build(true).addParam("meta", meta).addParam("return", true);
                }
            }
            //如果是手机登录，目前只在ios提审时用
        } else if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {

            //ios提测时用的测试账号
            if (isPublishState && this.appVersionIntegrationService.inWhiteList(loginInfo.getLoginName())) {
                loginInfo.setCode("123456");
            }

            //判断参数是否正确
            if (!this.userParamCheckService.checkParam(loginInfo)) {
                this.logger.info("doLogin failed:" + loginInfo);
                return Message.build(false, "参数错误");
            }

            //密码登录(ios提审时用)
            if (isPublishState && userParamCheckService.checkVerifyPwd(loginInfo)) {
                Message message = userParamCheckService.verifyPwd(loginInfo.getPassword(), userAuth);
                if (!message.isSuccess()) {
                    return message;
                }
            }
        }
        return Message.build(true);
    }

    /**
     * 微信注册
     */
    private Message registerByWechat(LoginInfo loginInfo, BasicParam basicParam) throws UserRepeatCreateException {

        //user层面的验证  根据手机号去user表里查是否有相同手机号的绑定，如果有第三方绑定的话则视为为已有用户绑定第三方
        User user = getUserByType(loginInfo);
        Message messageByCheckUser = userParamCheckService.checkRepeatBind(user, loginInfo);
        if (!messageByCheckUser.isSuccess()) {
            return messageByCheckUser;
        }

        //如果用户存在则进行绑定，如果用户不存在则添加用户并绑定第三方
        LoginMeta meta = createAndBinding(user, loginInfo, basicParam);

        return Message.build(true).addParam("meta", meta);
    }

    /**
     * 在登录之后需要对用户信息进行更新 优化不了，尽量别动吧，加点注释是我唯一能做的，希望后来者能重构
     */
    private LoginMeta login(UserAuth userAuth, LoginInfo loginInfo, BasicParam basicParam) {

        //根据设备信息查询该用户在该设备是否登录过，得到设备唯一id
        Message deviceExistMessage = userLoginHistoryService.isDeviceExist(userAuth.getUserId(), basicParam.getDevcId());
        String equipment = deviceExistMessage.getParamMap().get("equipment") == null
                ? null
                : deviceExistMessage.getParamMap().get("equipment").toString();

        //1.查找用户信息
        User user = this.userMapper.selectByPrimaryKey(userAuth.getUserId());

        //2.如果是在站外H5微信授权登录，更新一次同步微信头像，昵称，并且在auth中插入openId和unionId（大转盘和摇钱树）
        this.addWechatLoginInfo(loginInfo, user);

        //4.登录或激活用户
        UserInfoDTO dto = this.activeAndLogin(loginInfo, user, basicParam, equipment);

        //如果找不到设备id，则证明用户是新注册的
        if (!deviceExistMessage.isSuccess()) {
            // 5.首次APP登录，设置用户为有效用户（有些在大转盘或者是摇钱树创建过临时用户的）
            saveNewUserCache(userAuth.getUserId());
        }

        //7.返回给客户端包装类
        LoginMeta meta = new LoginMeta(dto, !deviceExistMessage.isSuccess());

        //8.保存用户登录信息(客户端时可以获取到设备信息的，另外的不行)异步
        userLoginHistoryService.save(equipment, basicParam, user.getId(), loginInfo.getRegIp());

        return meta;
    }

    /**
     * 创建用户token，并存储到redis，在网关层会根据存储的token进行用户登录状态判断
     */
    private String generateToken(Long userId, String token) {
        if (null == token) {
            token = JwtTokenUtil.generateToken(userId);
        }
        redisHashMapAdapter.put(UserRedisKey.HASH_USER_TOKEN, String.valueOf(userId), token);
        return token;
    }

    /**
     * 登录或激活用户
     */
    private UserInfoDTO activeAndLogin(LoginInfo loginInfo, User user, BasicParam basicParam, String equipment) {
        if (logger.isDebugEnabled()) {
            logger.debug("登录->开始更新用户数据，参数：user:[{}],设备号：[{}]", JSONObject.toJSON(user), equipment);
        }

        //更新用户token
        user.setToken(generateToken(user.getId(), null));
        user.setRefreshtoken(JwtTokenUtil.generateToken(user.getId()));
        user.setEquipment(equipment);
        user.setLastLoginIp(loginInfo.getLastLoginIp());
        user.setLastLoginTime(new Date());
        user.setChannelId(user.getChannelId() == 0
                ? channelService.getChannelByCode(basicParam.getChnl()).getId()
                : user.getChannelId());
        user.setInfoCompleteState(SpringContextHolder.getBean(UserPersonalInfoService.class).initUserInfoCompleteStatus(user));
        user.setRegisterClient(basicParam.getPlatform() + "");

        if (logger.isDebugEnabled()) {
            logger.debug("用户[{}]对应的状态：[{}]", user.getId(), user.getState());
        }

        if (Objects.equals(user.getChannelId(), 0L)) {
            logger.info("用户的注册渠道未定义，用户ID:[{}],原始渠道：[{}]", user.getId(), basicParam.getChnl());
        }

        // 将未激活用户更新用户为正常用户
        updateUserActive(user, basicParam, equipment);

        this.userMapper.updateByPrimaryKeySelective(user);

        UserInfoDTO userInfoDetail = this.userService.loadUserToRedis(user.getId());

        addAuthInfo(userInfoDetail);

        //vip信息补充
        this.replenishVipInfo(userInfoDetail);

        return userInfoDetail;
    }

    /**
     * 补充完善vip信息
     *
     * @param userInfoDetail ： 用户信息
     */
    private void replenishVipInfo(UserInfoDTO userInfoDetail) {
        UserVip userVip = userVipService.getUserVipByUid(userInfoDetail.getId());
        //用户是否为VIP
        userInfoDetail.setIsVip(Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.USEFUL.getStatus()) ? 1 : 0);
        //vip剩余激活次数
        if (Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.USEFUL.getStatus())) {
            userInfoDetail.setLeftActiveNum(userVip.getLeftOpenNum());
        } else {
            userInfoDetail.setLeftActiveNum(0);
        }
    }

    /**
     * 将未激活用户更新用户为正常用户
     */
    @Override
    public void updateUserActive(User user, BasicParam basicParam, String equipment) {

        if (AppConst.USER_STATE.NOT_ACTIVE == user.getState()) {
            UserAccount userAccount = userAccountService.getUserAccountDetail(user.getId());
            // 设置状态
            user.setState(AppConst.USER_STATE.NORMAL);

            //判断用户是否领取过红包，未领取的设备才认为是有效邀请（针对新人红包活动而言）
            boolean effective = false;
            if (StringUtils.isNotBlank(equipment)) {
                effective = userEquipmentMapper.countEquipment(equipment, "DRAW") == 0;
            }

            //处理邀请关系建立后的其他逻辑，根据不同的邀请类型进行不同的处理
            InviteProcessParam param = new InviteProcessParam();
            param.setInviteUserId(user.getInviteUserId());
            param.setUserId(user.getId());
            param.setBasicParam(basicParam);
            param.setEquipmentNum(equipment);
            param.setEffective(effective);

            //用户进行有效激活后，表示邀请成功，处理后续逻辑
            inviteProcessService.execAsyncActiveInvite(param);

            //更新用户的新人红包状态
            updateUserCashState(userAccount.getUserId(), effective, equipment);


            //移除邀请关系失效的定时任务
            scheduleService.remove(InviteRelationshipTask.generateTaskName(user.getInviteUserId(), user.getId()));

            //上报用户注册事件
            bizLogIntegrationService.createUser(user, basicParam, userAccount);
        }
    }

    /**
     * 新用户首次登陆赏金状态更改
     */
    private void updateUserCashState(Long userId, boolean isEffective, String equipment) {
        logger.debug("新用户[{}]首次登陆临时赏金转换为可提现赏金", userId);

        BigDecimal cash = cashFlowService.getMySelfBounty(userId);
        if (cash.compareTo(BigDecimal.ZERO) <= 0) {
            return;
        }

        if (isEffective) {
            //如果是有效用户，则将新人红包更新为可提现金额
            AccountCashParam param = new AccountCashParam(userId,
                    CashEnum.TEMP_CASH.name(),
                    false,
                    cash,
                    null,
                    null,
                    CashFlowConvertTypeEnum.TEMP_TO_DRAWABLE.name());
            userAccountService.transferCash(param);

            //记录该设备已经领取过现金
            UserEquipment userEquipment = new UserEquipment(nextId(), equipment, "DRAW");
            userEquipmentService.addUserEquipment(userEquipment);
        } else {
            //设备已经领取过现金，则转化为临时金额
            AccountCashParam accountCashParam = new AccountCashParam();
            accountCashParam.setUserId(userId);
            accountCashParam.setCashType(CashEnum.TEMP_CASH.name());
            accountCashParam.setAddTotal(false);
            accountCashParam.setCashFlowType(CashFlowTypeEnum.INVALID_PACKET.name());
            accountCashParam.setCash(cash.negate());
            userAccountService.addCash(accountCashParam);
        }
    }

    /**
     * 填充用户拥有的权限编码信息
     *
     * @param userInfo 用户信息
     */
    private void addAuthInfo(UserInfoDTO userInfo) {
        userInfo.setCombineAuthCode(userAuthCodeService.getAuthCombineCode(userInfo.getId()));
        userInfo.setAuthResources(userAuthCodeService.getAuthResources(userInfo.getId()));
    }

    /**
     * 如果是在站外H5微信授权登录,则更新openId和unionId和昵称头像
     * 2.0.0版本 没有h5授权登陆 openId是APP端传入保证用户能够正常提现
     */
    private void addWechatLoginInfo(LoginInfo loginInfo, User user) {
        if (logger.isDebugEnabled()) {
            logger.debug("用户登陆传入的参数：[{}]", toJSONString(loginInfo));
        }

        // 如果是微信登录 设置微信uid
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            loginInfo.setUnionId(loginInfo.getLoginName());
        }
        if (null != loginInfo.getOpenId() && null != loginInfo.getUnionId()) {
            //查看该用户有无unionId绑定
            List<UserAuth> userAuthList = userAuthMapper.listByUserId(user.getId());
            if (!CollectionUtils.isEmpty(userAuthList)) {
                if (logger.isDebugEnabled()) {
                    logger.debug("当前用户创建unionId鉴权信息:用户id：[{}], unionId:[{}]", user.getId(), loginInfo.getUnionId());
                }

                boolean isBindMsg = userAuthList.stream().anyMatch(e -> e.getIdentifier().equals(loginInfo.getUnionId()));

                if (!isBindMsg) {
                    user.setWeixin(loginInfo.getUnionId());

                    //增加一个union auth
                    this.userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN, user.getId(),
                            loginInfo.getUnionId(), null);
                }
            }

            //增加一个openid auth
            boolean isBindMsg = userAuthList.stream().anyMatch(e -> e.getIdentifier().equals(loginInfo.getOpenId()));
            if (!isBindMsg) {
                logger.debug("当前用户创建APP微信授权登陆openId鉴权信息:用户id：[{}], openId:[{}]", user.getId(),
                        loginInfo.getOpenId());
                this.userAuthService.addUserAuth(AppConst.LOGIN_TYPE.APP_OPENID, user.getId(),
                        loginInfo.getOpenId(), null);
            }
        }
    }

    /**
     * 更新用户的邀请状态信息
     */
    private void updateUserInviteHistory() {

    }

    @Override
    @Retryable(value = RetryException.class, backoff = @Backoff(300))
    public Json<LoginMeta> registerAndBindingPhone(LoginInfo loginInfo, Long userId, BasicParam basicParam) {
        if (!this.userParamCheckService.bindingAndRegisterCheckParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }

        //auth层面的验证  根据登录类型和登录标识去auth表里查 看看有没有被别人绑定
        //1.手机号注册：根据type=1和手机号去auth表里查是否#手机号#有被绑定的记录
        //2.第三方注册：根据type=3和微信unionId去auth表里查是否#微信#有被绑定的记录
        Message messageByCheckAuth = userAuthService.checkBindExist(loginInfo.getType(), loginInfo.getLoginName());
        if (!messageByCheckAuth.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth.getLastMessage());
        }

        //user层面的验证  根据手机号去user表里查是否有相同手机号的绑定，如果有第三方绑定的话则视为为已有用户绑定第三方
        User user = getUserByType(loginInfo);
        Message messageByCheckUser = userParamCheckService.checkRepeatBind(user, loginInfo);
        if (!messageByCheckUser.isSuccess()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckUser.getLastMessage());
        }

        //用户绑定手机号时
        if (null != userId) {
            user = userMapper.selectByPrimaryKey(userId);
            if (null != user && 0 == user.getState()) {
                user = null;
            }
        }

        //手机号绑定需要验证码校验
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            if (StringUtils.isNotBlank(loginInfo.getCode()) &&
                    !this.smsIntegrationService.verifyByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE,
                            loginInfo.getPhone(), loginInfo.getCode())) {
                return ResultUtil.genFailedResult("请输入正确的验证码");
            }
        }

        //如果用户存在则进行绑定，如果用户不存在则添加用户并绑定第三方
        LoginMeta meta;
        try {
            meta = createAndBinding(user, loginInfo, basicParam);
        } catch (UserRepeatCreateException e) {
            logger.warn("重复创建用户，进行重试，请求参数：{}", loginInfo);
            throw new RetryException("用户已存在，重新创建");
        }

        return ResultUtil.genSuccessResult(meta);
    }

    /**
     * 根据登录类型的不同获取用户
     */
    private User getUserByType(LoginInfo loginInfo) {
        User user = null;
        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user = this.userMapper.findByWeixin(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user = this.userMapper.findByPhone(loginInfo.getPhone());
        }
        return user;
    }

    /**
     * 创建用户，绑定第三方信息
     */
    private LoginMeta createAndBinding(User user, LoginInfo loginInfo, BasicParam basicParam) throws UserRepeatCreateException {
        LoginMeta meta = new LoginMeta();

        UserInfoDTO dto;

        if (null == user) {
            //新增用户并绑定第三方信息
            user = this.addDefaultUser(loginInfo, basicParam);
            //生成token放入redis库中
            generateToken(user.getId(), user.getToken());
        } else {
            //同步第三方信息(手机绑定就同步手机号，第三方绑定就同步第三方信息)
            syncThirdPartyInfo(user, loginInfo);

            //更新用户token
            changeToken(user, loginInfo, basicParam);

            //添加用户权限信息
            addUserAuth(loginInfo, user.getId());
        }

        //返回包装类
        dto = this.userService.convertUserToDTO(user);

        //是否首次登录app
        boolean isFirstLogin = false;

        String equipment = basicParam.getDevcId();

        if (!NumberUtils.isDigits(basicParam.getDevcId())) {
            EquipmentDTO equipmentDTO = bizIntegrationService.getDevice(basicParam.getDevcId());
            equipment = equipmentDTO == null ? null : equipmentDTO.getId();
        }

        Message deviceExistMessage = userLoginHistoryService.isDeviceExistByDeviceId(user.getId(), equipment);
        if (!deviceExistMessage.isSuccess()) {
            isFirstLogin = true;
        }

        //首次APP登录，设置用户为有效用户
        //第一次登陆绑定VIP邀请关系
        if (isFirstLogin) {
            saveNewUserCache(user.getId());
        }

        //添加用户权限信息
        addAuthInfo(dto);

        meta.setUser(dto);
        meta.setFirstLogin(isFirstLogin);

        //保存用户登录信息(客户端时可以获取到设备信息的，另外的不行)异步
        userLoginHistoryService.save(equipment, basicParam, user.getId(), loginInfo.getRegIp());

        //缓存redis
        this.userService.putUserToRedis(user);

        return meta;
    }

    /**
     * 记录首次登录用户缓存 后续将用于判断用户是否第一天登录的新用户
     */
    private void saveNewUserCache(Long id) {
        KeyGenerator key = RedisConfig.NEW_USER_LOGIN.copy().appendKey(id);
        redisStringAdapter.set(key, true);
        redisStringAdapter.expire(key, DateUtils.getCurSeconds());
    }

    /**
     * 第三方绑定后更新用户token
     */
    private void changeToken(User user, LoginInfo loginInfo, BasicParam basicParam) {
        //更新用户token
        user.setToken(generateToken(user.getId(), null));
        if (loginInfo.getSex() != null) {
            user.setSex(loginInfo.getSex());
        }
        user.setLastLoginTime(new Date());

        this.userMapper.updateByPrimaryKeySelective(user);

    }

    /**
     * 创建用户以及相应信息
     */
    private User addDefaultUser(LoginInfo loginInfo, BasicParam basicParam) throws UserRepeatCreateException {
        //新用户信息录入
        return this.userRegistrationService.addUser(loginInfo, basicParam, AppConst.USER_STATE.NORMAL);
    }

    /**
     * 用户首次登录的处理流程
     *
     * @param userId 用户ID
     * @param wechat 微信uid
     */
    private void firstLogin(Long userId, String wechat, Integer platform) {
        //v3.0.0 VIP相关逻辑下线，代码保留
        //vipIntegrationService.saveVipViewRecord(userId, wechat, platform);
    }

    /**
     * 添加三方登录标识
     */
    private void addUserAuth(LoginInfo loginInfo, Long userId) {
        logger.info("添加用户权限认证");
        UserAuth auth = new UserAuth();
        auth.setUserId(userId);
        auth.setType(loginInfo.getType());

        //如果是手机号码绑定
        if (StringUtils.equals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getPhone());
            //将密码通过MD5加密
            if (StringUtils.isNotEmpty(loginInfo.getPassword())) {
                auth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
            }
            this.userAuthMapper.insertSelective(auth);
        }

        //第三方登录绑定
        if (StringUtils.notEquals(loginInfo.getPhone(), loginInfo.getLoginName())) {
            auth.setIdentifier(loginInfo.getLoginName());
            this.userAuthMapper.insertSelective(auth);
        }
    }

    /**
     * 使用第三方登录时，将第三方信息更新到用户属性中
     */
    private void syncThirdPartyInfo(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            if (null != loginInfo.getNickname()) {
                user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
            }
            if (null != loginInfo.getHeadImg()) {
                user.setHeadImg(user.getHeadImg() == null ? loginInfo.getHeadImg() : user.getHeadImg());
            }
        }
    }
}
