package com.bxm.localnews.user.support.impl;

import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.base.model.dto.AvailableDomainInfo;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.user.attribute.UserFollowService;
import com.bxm.localnews.user.attribute.UserTagService;
import com.bxm.localnews.user.attribute.UserVisitService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserLoginHistoryMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.MerchantUserDTO;
import com.bxm.localnews.user.properties.BizConfigProperties;
import com.bxm.localnews.user.properties.HomePageConfigProperties;
import com.bxm.localnews.user.properties.NativeUserProperties;
import com.bxm.localnews.user.support.HomePageService;
import com.bxm.localnews.user.utils.ConstellationUtils;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vip.relation.MerchantRelationService;
import com.bxm.localnews.user.vo.UserTag;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.AllArgsConstructor;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import static java.lang.Boolean.TRUE;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class HomePageServiceImpl extends BaseService implements HomePageService {

    private final String RESOURCE = "homepage:systemmessage:lock:";

    private final UserMapper userMapper;

    private final HomePageConfigProperties homePageConfigProperties;

    private final BizConfigProperties bizConfigProperties;

    private final RedisStringAdapter redisStringAdapter;

    private final UserLoginHistoryMapper userLoginHistoryMapper;

    private final PushMsgIntegrationService pushMsgIntegrationService;

    private final DistributedLock distributedLock;

    private final UserTagService userTagService;

    private final UserVisitService userVisitService;

    private final NativeUserProperties nativeUserProperties;

    private final UserService userService;

    private final MerchantRelationService merchantRelationService;

    private final UserVipService userVipService;

    private final DomainIntegrationService domainIntegrationService;

    @Override
    public OnselfHomePageDTO getOnselfHomePage(Long userId) {
        OnselfHomePageDTO result = new OnselfHomePageDTO();
        BaseHomePage baseHomePage = this.getBaseHomePage(userId);
        BeanUtils.copyProperties(baseHomePage, result);
        //用户最近访问的头像
        result.setInterviewHeadImgs(userVisitService.listRecentVisitorHeadImg(userId));
        //今天增加的访问数量
        result.setTodayInterviewNum(userVisitService.getTodayVisitorNum(userId));
        //总访问量
        result.setAllInterviewNum(userVisitService.getInterviewNumByUserId(userId));
        //添加vip和关联商家信息
        addRelateMerchant(result, userId, true);
        return result;
    }

    @Override
    public CustomerHomePageDTO getCustomerHomePage(Long targetUserId, Long userId) {

        CustomerHomePageDTO result = new CustomerHomePageDTO();
        BaseHomePage baseHomePage = this.getBaseHomePage(targetUserId);
        BeanUtils.copyProperties(baseHomePage, result);
        //判断是否关注
        result.setIsAttention(SpringContextHolder.getBean(UserFollowService.class).isFollowed(userId, targetUserId));
        //调用访客接口
        SpringContextHolder.getBean(UserVisitService.class).visit(userId, targetUserId);

        //添加vip和关联商家信息
        addRelateMerchant(result, targetUserId, false);
        return result;
    }

    /**
     * 增加用户vip和关联商家信息
     *
     * @param result
     * @param userId
     */
    private void addRelateMerchant(BaseHomePage result, Long userId, boolean isSelf) {
        result.setIsVip(Objects.equals(userVipService.isVip(userId), TRUE) ? 1 : 0);

        // 商家信息
        MerchantUserDTO merchantUserDTO = merchantRelationService.getRelationMerchant(userId);

        if (Objects.nonNull(merchantUserDTO)) {
            if (!Objects.equals(merchantUserDTO.getIsShelf(), 1)) {
                // 如果非上架状态
                if (isSelf) {
                    // 主态访问 返回商家信息
                    BeanUtils.copyProperties(merchantUserDTO, result);
                }
                // 客态态 就不返回了
                return;
            }
            // 上架状态 则直接copy
            BeanUtils.copyProperties(merchantUserDTO, result);
        }
    }

    /**
     * 获取个人主页基本信息
     */
    private BaseHomePage getBaseHomePage(Long userId) {
        UserService userService = SpringContextHolder.getBean(UserService.class);
        //缓存获取用户信息
        UserInfoDTO user = userService.getUserCache(userId);

        BaseHomePage baseHomePage = new BaseHomePage();
        if (Objects.nonNull(user)) {
            BeanUtils.copyProperties(user, baseHomePage);

            //填充默认个人简介
            if (Objects.equals(user.getIsDefaultPersonalProfile(), TRUE)) {
                baseHomePage.setPersonalProfile(nativeUserProperties.getDefaultPersonalProfile());
            }

            //处理星座年龄
            if (null == user.getBirthday()) {
                baseHomePage.setAge(null);
            } else {
                LocalDate birthday = user.getBirthday().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                if (birthday.isAfter(LocalDate.now())) {
                    baseHomePage.setAge(null);
                } else {
                    baseHomePage.setConstellation(ConstellationUtils.calculateConstellation(birthday));
                    baseHomePage.setBirthday(birthday);
                    baseHomePage.setGeneration(userService.getGeneration(birthday));
                    baseHomePage.setAge(LocalDate.now().getYear() - birthday.getYear() + 1);
                }
            }

            // 图片不存在随机选择背景图片,并保存到数据库
            if (StringUtils.isBlank(baseHomePage.getBackgroundImgUrl())) {
                List<String> urls = homePageConfigProperties.getDefaultBackgroundImgUrls();
                int index = ThreadLocalRandom.current().nextInt(homePageConfigProperties.getDefaultBackgroundImgUrls().size());
                String backgroundImgUrl = urls.get(index);
                userMapper.updateBackgroundImgUrlById(userId, backgroundImgUrl);
                baseHomePage.setBackgroundImgUrl(backgroundImgUrl);
                //更新用户缓存
                userService.loadUserToRedis(userId);
            }

            //设置分享副标题
            if (Objects.equals(user.getIsDefaultPersonalProfile(), TRUE)) {
                baseHomePage.setShareSubTitle(homePageConfigProperties.getShareSubTitle());
            } else {
                baseHomePage.setShareSubTitle(baseHomePage.getPersonalProfile());
            }

            //{nickName} 分享标题中用户昵称的占位符
            if (isNotBlank(homePageConfigProperties.getShareTitle()) && isNotBlank(user.getNickname())) {
                baseHomePage.setShareTitle(homePageConfigProperties.getShareTitle().replace("{nickName}", user.getNickname()));
            }

            // 分享链接 域名服务获取base url 进行拼接
            Optional<AvailableDomainInfo> domainInfo = domainIntegrationService.getBaseUrlInfoByScene(DomainScene.OUTSIDE_SHARE_VIEW,
                    DomainScene.DomainViewScene.CONTENT_VIEW, null);
            baseHomePage.setShareUrl(StringUtils.join(domainInfo.isPresent() ? domainInfo.get().getBaseUrl() : "",
                    "/shareUserHome.html"));

            //填充兴趣标签列表
            List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(userId);
            baseHomePage.setUserTagList(
                    userTagList.stream().filter(e -> e.getDeleteFlag() == 0).map(UserTag::getLabel).collect(Collectors.toList()));

        }
        return baseHomePage;
    }


    @Override
    public Boolean sendSystemMessage(Long userId) {
        KeyGenerator keyGenerator = RedisConfig.USER_ACCESS_MESSAGE.copy().appendKey(userId);
        String stringId = String.valueOf(nextId());
        String lockkey = RESOURCE + userId;
        //分布式锁,接口幂等
        if (distributedLock.lock(lockkey, stringId)) {
            try {
                Long timestamp = redisStringAdapter.get(keyGenerator, Long.class);
                boolean flag = Objects.isNull(timestamp)
                        || !new Date(timestamp).toInstant().atZone(ZoneId.systemDefault()).toLocalDate().equals(LocalDate.now());
                //执行发系统消息逻辑
                if (flag) {
                    Date sendDate;
                    if (Objects.isNull(timestamp)) {
                        sendDate = userLoginHistoryMapper.selectMinLoginTime(userId);
                    } else {
                        sendDate = new Date(timestamp);
                    }
                    int count = userVisitService.getInterviewNumByTime(userId, sendDate);
                    if (count > 0) {
                        logger.debug("homepage systemmessage ,userId={},sendTime={}", userId,
                                DateFormatUtils.format(new Date(), DateUtils.DATE_TIME_FORMAT));
                        LocationUserInfoDTO location = userService.getLocationUserInfo(userId);
                        pushMsgIntegrationService.sendSystemMessage(userId, count, location);
                        redisStringAdapter.set(keyGenerator, System.currentTimeMillis());
                    }
                }
            } finally {
                distributedLock.unlock(lockkey, stringId);
            }
        }
        return TRUE;
    }

}
