package com.bxm.localnews.user.support.impl;

import com.bxm.localnews.common.constant.InfoCombineStateEnum;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.user.attribute.UserTagService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.integration.UserNewsIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.NativeParam;
import com.bxm.localnews.user.param.NativeRecommendContext;
import com.bxm.localnews.user.properties.NativeUserProperties;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.service.NativeRecommendService;
import com.bxm.localnews.user.support.NativeUserService;
import com.bxm.localnews.user.vo.RecommendNative;
import com.bxm.localnews.user.vo.UserTag;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSON;
import static com.alibaba.fastjson.JSON.toJSONString;
import static org.apache.commons.lang3.RandomUtils.nextInt;

@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class NativeUserServiceImpl extends BaseService implements NativeUserService {

    private final NativeRecommendService nativeRecommendService;

    private final NativeUserProperties nativeUserProperties;

    private final UserService userService;

    private final UserTagService userTagService;

    private final LocationIntegrationService locationIntegrationService;

    private final UserNewsIntegrationService userNewsIntegrationService;

    private final RedisSetAdapter redisSetAdapter;

    private final UserProperties userProperties;

    @Override
    public NativeDTO listNative(Long userId, NativeParam nativeParam, BasicParam basicParam) {
        logger.debug("用户进行本地人推荐，用户id:[{}],参数:[{}]", userId, toJSON(nativeParam));
        List<NativeInfoDTO> nativeDtoList = new ArrayList<>();

        //组装参数
        NativeRecommendContext nativeRecommendContext = NativeRecommendContext.builder().userId(userId)
                .industryId(StringUtils.isEmpty(nativeParam.getIndustryId()) ? null : Long.parseLong(nativeParam.getIndustryId())).build();
        BeanUtils.copyProperties(nativeParam, nativeRecommendContext);

        //如果是下拉，需要用户信息做推荐
        if (1 == nativeParam.getActionType()) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(nativeParam.getCurrentAreaCode());
            if (null != locationDTO) {
                //客户端传的code是6位的，需要转成12位
                nativeRecommendContext.setCurrentAreaCode(locationDTO.getCode());
            }
            UserInfoDTO user = userService.getUserCache(userId);
            nativeRecommendContext.addParam("user", user);
        }

        //调用推荐逻辑获取数据
        NativeRecommendContext recommendNativeWarper = nativeRecommendService.listRecommendUser(nativeRecommendContext);
        List<RecommendNative> recommendNativeList = recommendNativeWarper.getResult();

        logger.debug("得到用户推荐列表:[{}]", toJSON(recommendNativeList));

        if (!CollectionUtils.isEmpty(recommendNativeList)) {
            //得到推荐出来的用户id列表
            List<Long> userIdList = recommendNativeList.stream().map(RecommendNative::getUserId).collect(Collectors.toList());

            //可做多线程获取
            //得到用户map
            Map<Long, UserInfoDTO> userMap = getUserMap(userIdList);
            //得到用户标签map
            Map<Long, List<UserTag>> userTagMap = userTagService.getBatchUserTag(userIdList);
            //得到用户帖子图片map
            Map<Long, List<ImgDTO>> imgMap = userNewsIntegrationService.batchGetUserImg(userIdList);

            //开始组装
            recommendNativeList.forEach(e -> {

                UserInfoDTO user = userMap.get(e.getUserId());
                if (user == null) {
                    return;
                }

                //标签组装
                List<String> tagNameList = new ArrayList<>();
                List<UserTag> userTagList = userTagMap.get(e.getUserId());
                if (!CollectionUtils.isEmpty(userTagList)) {
                    tagNameList = userTagList.stream().filter(t -> t.getDeleteFlag() == 0).map(UserTag::getLabel)
                            .collect(Collectors.toList());
                }

                //图片组装
                List<ImgDTO> imgDtoList = new ArrayList<>();
                List<ImgDTO> imgList = imgMap.get(e.getUserId());
                if (!CollectionUtils.isEmpty(imgList)) {
                    imgDtoList = imgList;
                }

                //距离组装
                String distance = "";
                if (null != e.getDistance() && 0 != e.getDistance()) {
                    distance = getDistance(e.getDistance()) + "km内";
                }

                //年龄段组装(想不出什么好办法。。先写着)
                String generationStr = getGeneration(user);

                //本地人简介
                String personalProfile = user.getPersonalProfile();
                if (user.getIsDefaultPersonalProfile()) {
                    personalProfile = nativeUserProperties.getDefaultPersonalProfile();
                }

                //对象组装
                nativeDtoList.add(NativeInfoDTO.builder()
                        .distance(distance)
                        .generation(generationStr)
                        .headImg(user.getHeadImg())
                        .hobbyList(tagNameList)
                        .imgList(imgDtoList)
                        .industry(user.getIndustry())
                        .nickName(user.getNickname())
                        .personalProfile(personalProfile)
                        .recommendType(e.getRecommendType())
                        .sex(user.getSex())
                        .userId(e.getUserId())
                        .build());
            });
        }

        return NativeDTO.builder().recommendCategory(recommendNativeWarper.getRecommendCategory()).nativeInfoList(nativeDtoList).build();

    }

    @Override
    public NativeDTO listNativeForV2(Long userId, NativeParam nativeParam, BasicParam basicParam) {
        logger.debug("用户进行本地人推荐，用户id:[{}],参数:[{}]", userId, toJSON(nativeParam));
        List<NativeInfoDTO> nativeDtoList = new ArrayList<>();

        //组装请求参数
        NativeRecommendContext nativeRecommendContext = NativeRecommendContext.builder()
                .userId(userId)
                //客户端非要老子给个String
                .industryId(StringUtils.isEmpty(nativeParam.getIndustryId()) ? null : Long.parseLong(nativeParam.getIndustryId()))
                .build();
        BeanUtils.copyProperties(nativeParam, nativeRecommendContext);

        //如果是下拉，需要用户信息做推荐
        UserInfoDTO currentUser = userService.getUserCache(userId);
        if (1 == nativeParam.getActionType()) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(nativeParam.getCurrentAreaCode());
            if (null != locationDTO) {
                //客户端传的code是6位的，需要转成12位
                nativeRecommendContext.setCurrentAreaCode(locationDTO.getCode());
            }
            nativeRecommendContext.addParam("user", currentUser);
        }

        //调用推荐逻辑获取数据
        NativeRecommendContext recommendNativeWarper = nativeRecommendService.listRecommendUser(nativeRecommendContext);
        List<RecommendNative> recommendNativeList = recommendNativeWarper.getResult();
        logger.debug("得到用户推荐列表:[{}]", toJSON(recommendNativeList));
        if (!CollectionUtils.isEmpty(recommendNativeList)) {
            //得到推荐出来的用户id列表
            List<Long> userIdList = recommendNativeList.stream()
                    .map(RecommendNative::getUserId)
                    .collect(Collectors.toList());
            //推荐出来的列表如果包含自己
            Long currentUserId = null;
            if (userIdList.contains(userId)) {
                currentUserId = userId;
            }

            //得到用户map
            Map<Long, UserInfoDTO> userMap = getUserMap(userIdList);
            //得到用户标签map
            Map<Long, List<UserTag>> userTagMap = userTagService.getBatchUserTag(userIdList);
            //得到用户帖子图片map
            Map<Long, List<ImgDTO>> imgMap = userNewsIntegrationService.batchGetAllUserImg(userIdList, currentUserId);
            //得到用户纸条map
            Map<Long, NoteDTO> noteMap = userNewsIntegrationService.batchGetUserNote(userIdList, currentUserId);

            //开始组装
            nativeDtoList = recommendNativeList.parallelStream()
                    .map(e -> generateRecommendNativeInfo(userId, currentUser, userMap, userTagMap, imgMap, noteMap, e))
                    .filter(Objects::nonNull).collect(Collectors.toList());
        }

        NativeDTO nativeDTO = NativeDTO.builder().recommendCategory(recommendNativeWarper.getRecommendCategory())
                .nativeInfoList(nativeDtoList).build();
        logger.debug("本地人推荐最终结果:[{}]", toJSONString(nativeDTO));
        return nativeDTO;

    }

    @Override
    public NativeDTO listNativeForV3(Long userId, NativeParam nativeParam, BasicParam basicParam) {
        logger.debug("用户进行本地人推荐，用户id:[{}],参数:[{}]", userId, toJSON(nativeParam));
        List<NativeInfoDTO> nativeDtoList = new ArrayList<>();

        //如果是下拉，需要用户信息做推荐
        UserInfoDTO currentUser = userService.getUserCache(userId);

        //组装请求参数
        NativeRecommendContext nativeRecommendContext = assemblyParam(userId, nativeParam, currentUser);

        //调用推荐逻辑获取数据
        NativeRecommendContext recommendNativeWarper = nativeRecommendService.listRecommendUser(nativeRecommendContext);
        List<RecommendNative> recommendNativeList = recommendNativeWarper.getResult();
        logger.debug("得到用户推荐列表:[{}]", toJSON(recommendNativeList));

        //开始根据id组装数据
        if (!CollectionUtils.isEmpty(recommendNativeList)) {
            //得到推荐出来的用户id列表
            List<Long> userIdList = recommendNativeList.stream()
                    .map(RecommendNative::getUserId)
                    .collect(Collectors.toList());
            //推荐出来的列表如果包含自己
            Long currentUserId = null;
            if (userIdList.contains(userId)) {
                currentUserId = userId;
            }

            //得到用户map
            AtomicReference<Map<Long, UserInfoDTO>> userMap = new AtomicReference<>(Maps.newHashMap());
            //得到用户标签map
            AtomicReference<Map<Long, List<UserTag>>> userTagMap = new AtomicReference<>(Maps.newHashMap());
            //得到用户帖子图片map
            AtomicReference<Map<Long, List<ImgDTO>>> imgMap = new AtomicReference<>(Maps.newHashMap());
            //得到用户纸条map
            AtomicReference<Map<Long, NoteDTO>> noteMap = new AtomicReference<>(Maps.newHashMap());

            //异步填充准备数据(并行)
            preProcessData(userIdList, currentUserId, userMap, userTagMap, imgMap, noteMap);

            //开始组装(并行流)
            nativeDtoList = recommendNativeList.parallelStream()
                    .map(e -> generateRecommendNativeInfo(userId, currentUser, userMap.get(), userTagMap.get(), imgMap.get(), noteMap.get(), e))
                    .filter(Objects::nonNull).collect(Collectors.toList());

            //开始穿插引导资料卡片
            interspersedNewbieGuide(userId, nativeDtoList, currentUser, null != currentUserId);
        }
        NativeDTO nativeDTO = NativeDTO.builder().recommendCategory(recommendNativeWarper.getRecommendCategory())
                .nativeInfoList(nativeDtoList).build();
        logger.debug("本地人推荐最终结果:[{}]", toJSONString(nativeDTO));
        return nativeDTO;
    }

    /**
     * 并行准备填充数据
     */
    private void preProcessData(List<Long> userIdList, Long currentUserId, AtomicReference<Map<Long, UserInfoDTO>> userMap,
                                AtomicReference<Map<Long, List<UserTag>>> userTagMap, AtomicReference<Map<Long, List<ImgDTO>>> imgMap,
                                AtomicReference<Map<Long, NoteDTO>> noteMap) {
        CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> {
            userMap.set(getUserMap(userIdList));
        });
        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> {
            userTagMap.set(userTagService.getBatchUserTag(userIdList));
        });
        CompletableFuture<Void> future3 = CompletableFuture.runAsync(() -> {
            imgMap.set(userNewsIntegrationService.batchGetAllUserImg(userIdList, currentUserId));
        });

        CompletableFuture<Void> future4 = CompletableFuture.runAsync(() -> {
            noteMap.set(userNewsIntegrationService.batchGetUserNote(userIdList, currentUserId));
        });

        CompletableFuture<Void> all = CompletableFuture.allOf(future1, future2, future3, future4);
        try {
            all.get();
            logger.info("【帖子】、【纸条】、【标签】准备数据就绪");
        } catch (InterruptedException | ExecutionException e) {
            logger.error("【帖子】、【纸条】、【标签】准备数据获取出错[{}]", e.getMessage());
            logger.error(e.getMessage(), e);
            Thread.currentThread().interrupt();
        }
    }

    /**
     * 填充请求参数
     */
    private NativeRecommendContext assemblyParam(Long userId, NativeParam nativeParam, UserInfoDTO currentUser) {
        NativeRecommendContext nativeRecommendContext = NativeRecommendContext.builder()
                .userId(userId)
                //客户端非要老子给个String
                .industryId(StringUtils.isEmpty(nativeParam.getIndustryId()) ? null : Long.parseLong(nativeParam.getIndustryId()))
                .build();
        BeanUtils.copyProperties(nativeParam, nativeRecommendContext);

        if (1 == nativeParam.getActionType()) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(nativeParam.getCurrentAreaCode());
            if (null != locationDTO) {
                //客户端传的code是6位的，需要转成12位
                nativeRecommendContext.setCurrentAreaCode(locationDTO.getCode());
            }
            nativeRecommendContext.addParam("user", currentUser);
        }
        return nativeRecommendContext;
    }

    /**
     * 填充本地人列表
     */
    private NativeInfoDTO generateRecommendNativeInfo(Long userId,
                                                      UserInfoDTO currentUser,
                                                      Map<Long, UserInfoDTO> userMap,
                                                      Map<Long, List<UserTag>> userTagMap,
                                                      Map<Long, List<ImgDTO>> imgMap,
                                                      Map<Long, NoteDTO> noteMap,
                                                      RecommendNative e) {
        UserInfoDTO user = userMap.get(e.getUserId());
        if (user == null) {
            return null;
        }

        //标签组装
        List<String> tagNameList = new ArrayList<>();
        List<UserTag> userTagList = userTagMap.get(e.getUserId());
        if (!CollectionUtils.isEmpty(userTagList)) {
            tagNameList = userTagList.stream()
                    .filter(t -> t.getDeleteFlag() == 0)
                    .map(UserTag::getLabel)
                    .collect(Collectors.toList());
        }

        //动态信息组装
        List<ImgDTO> imgDtoList = new ArrayList<>();
        DynamicDTO dynamicDTO = getDynamic(noteMap, imgMap, imgDtoList, e.getUserId());

        //距离组装,如果列表显示自己，则不显示距离
        String distance = "";
        if (null != e.getDistance() && 0 != e.getDistance() && !userId.equals(e.getUserId())) {
            distance = getDistance(e.getDistance()) + "km内";
        }

        //年龄段组装(想不出什么好办法。。先写着)
        String generationStr = getGeneration(user);

        //本地人简介
        String personalProfile = getPersonalProfile(user, currentUser);

        //对象组装
        return NativeInfoDTO.builder()
                .distance(distance)
                .generation(generationStr)
                .headImg(user.getHeadImg())
                .hobbyList(tagNameList)
                .imgList(imgDtoList)
                .dynamic(dynamicDTO)
                .industry(user.getIndustry())
                .nickName(user.getNickname())
                .personalProfile(personalProfile)
                .recommendType(e.getRecommendType())
                .sex(user.getSex())
                .userId(e.getUserId())
                .build();
    }

    /**
     * 如果是当天首次浏览本地人列表，判断资料完善度是否有80%，如果没有，就穿插一条引导资料完善的卡片
     */
    private void interspersedNewbieGuide(Long userId, List<NativeInfoDTO> nativeDtoList, UserInfoDTO currentUser, boolean hasCurrentUser) {
        //如果当天首次浏览本地人列表，穿插一个引导资料的卡片
        //判断资料完善度是否有80%
        KeyGenerator keyGenerator = RedisConfig.USER_RECOMMEND.copy()
                .appendKey("newbie-guide").appendKey(DateUtils.formatDate(new Date()));

        logger.debug("穿插引导完善资料卡片-判断今日是否已引导");
        if (!redisSetAdapter.exists(keyGenerator, userId)) {
            logger.debug("开始穿插引导完善资料卡片-判断是否达到引导阈值");
            if (InfoCombineStateEnum.getInfoCompletePercent(currentUser.getInfoCompleteState()) >= userProperties
                    .getNativeNewbieGuideThreshold()) {
                return;
            }
            logger.debug("开始穿插引导完善资料卡片-开始");
            //如果列表中本来就有当前用户的feed，就将他删掉
            if (hasCurrentUser) {
                nativeDtoList.removeIf(e -> e.getUserId().equals(currentUser.getId()));
            }

            //得到当前用户标签
            List<String> tagNameList = userTagService.getUserTagFromRedisDb(userId)
                    .stream()
                    .filter(t -> t.getDeleteFlag() == 0)
                    .map(UserTag::getLabel)
                    .collect(Collectors.toList());

            //得到当前用户年龄段
            String generationStr = getGeneration(currentUser);

            //组装当前用户实体
            NativeInfoDTO nativeInfoDTO = NativeInfoDTO.builder()
                    .distance("")
                    .generation(generationStr)
                    .headImg(currentUser.getHeadImg())
                    .hobbyList(tagNameList)
                    .imgList(new ArrayList<>())
                    .dynamic(DynamicDTO.buildGuideDynamic(GuideDTO.builder().text(userProperties.getNativeNewbieGuideText()).build()))
                    .industry(currentUser.getIndustry())
                    .nickName(currentUser.getNickname())
                    .recommendType((byte) 2)
                    .sex(currentUser.getSex())
                    .userId(userId)
                    .build();

            if (currentUser.getIsDefaultPersonalProfile()) {
                nativeInfoDTO.setPersonalProfile(userProperties.getNativeNewbieGuidePersonalProfile());
            }

            //得到穿插的位置
            int position = userProperties.getNativeNewbieGuidePosition();
            if (nativeDtoList.size() >= position) {
                nativeDtoList.add(position - 1, nativeInfoDTO);
            } else {
                nativeDtoList.add(nativeInfoDTO);
            }

            redisSetAdapter.add(keyGenerator, currentUser.getId());
            redisSetAdapter.expire(keyGenerator, DateUtils.getCurSeconds());

            logger.debug("开始穿插引导完善资料卡片-结束");
        }

    }

    /**
     * 计算动态信息
     */
    private DynamicDTO getDynamic(Map<Long, NoteDTO> noteMap,
                                  Map<Long, List<ImgDTO>> imgMap,
                                  List<ImgDTO> imgDtoList,
                                  Long userId) {
        DynamicDTO dynamicDTO = DynamicDTO.buildEmptyDynamic();

        //得到动态信息
        NoteDTO note = noteMap.get(userId);
        if (null == note) {
            List<ImgDTO> imgList = imgMap.get(userId);
            if (!CollectionUtils.isEmpty(imgList)) {
                imgDtoList = imgList;
            }
        }

        //填充动态信息
        if (null != note) {
            dynamicDTO = DynamicDTO.buildNoteDynamic(note);
        } else if (!CollectionUtils.isEmpty(imgDtoList)) {
            dynamicDTO = DynamicDTO.buildPostDynamic(PostDTO.builder().imgList(imgDtoList).build());
        }
        return dynamicDTO;
    }

    /**
     * 得到用户map
     */
    private Map<Long, UserInfoDTO> getUserMap(List<Long> userIdList) {
        //得到用户列表
        List<UserInfoDTO> userList = userService.getBatchUserInfo(userIdList);

        //得到用户map
        Map<Long, UserInfoDTO> userMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(userList)) {
            userMap = userList.stream()
                    .collect(Collectors.toMap(UserInfoDTO::getId, Function.identity(), (key1, key2) -> key1));
        }
        return userMap;
    }

    /**
     * 个人简介组装
     */
    private String getPersonalProfile(UserInfoDTO user, UserInfoDTO currentUser) {
        String he = "TA";
        if (1 == user.getSex()) {
            he = "他";
        } else if (2 == user.getSex()) {
            he = "她";
        }
        String personalProfile = user.getPersonalProfile();
        boolean defaultFlag = false;
        if (user.getIsDefaultPersonalProfile()) {
            defaultFlag = true;
            //如果没有简介，则拼凑简介
            //是否同行：判断跟当前用户二级行业是否相同，相同则显示：他/她是你同行呢
            if (null != user.getJobCategory()
                    && null != currentUser.getJobCategory()
                    && user.getJobCategory().equals(currentUser.getJobCategory())) {
                personalProfile = "TA是你的同行呢，一起交流交流";
                defaultFlag = false;
                //对方是否单身：单身则显示他/她现在单身呢
            } else if (null != user.getRelationshipStatus() && 1 == user.getRelationshipStatus()) {
                personalProfile = "TA现在是单身，要不要打个招呼？";
                defaultFlag = false;
                //根据是否家乡相同：提示他/她跟你是老乡哦；如果当前用户选择城市和家乡一样，则不需要提醒对方跟你们老乡
            } else if (null != user.getHometownCode() && null != currentUser.getHometownCode()) {
                if (user.getHometownCode().equals(currentUser.getHometownCode()) &&
                        !user.getHometownCode().equals(user.getLocationCode()) || null == user.getLocationCode()) {
                    personalProfile = "TA和你是老乡哦！快去打个招呼~";
                    defaultFlag = false;
                }
            }
        }

        if (defaultFlag) {
            //随机取个人简介
            int randomIndex = nextInt(0, nativeUserProperties.getDefaultPersonalProfiles().size());
            personalProfile = nativeUserProperties.getDefaultPersonalProfiles().get(randomIndex);
        }

        personalProfile = personalProfile.replace("TA", he);

        return personalProfile;
    }

    /**
     * 年龄段组装
     */
    private String getGeneration(UserInfoDTO user) {
        String generationStr = "";
        if (null != user.getBirthday()) {
            LocalDate birthday = user.getBirthday().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            String gen = userService.getGeneration(birthday);
            if (!StringUtils.isEmpty(gen)) {
                generationStr = gen;
            }
        }
        return generationStr;
    }

    /**
     * 不超过5公里，就是1公里以内，2公里以内，3公里以内，4公里以内，5公里以内； 超过5公司就显示10公里以内，20公里以内，30公里以内，依次类推。
     */
    private Integer getDistance(Integer originalDistance) {
        if (originalDistance == 0) {
            return 0;
        } else if (originalDistance <= 1) {
            return 1;
        } else if (originalDistance <= 2) {
            return 2;
        } else if (originalDistance <= 3) {
            return 3;
        } else if (originalDistance <= 4) {
            return 4;
        } else if (originalDistance <= 5) {
            return 5;
        } else if (originalDistance <= 10) {
            return 10;
        } else if (originalDistance <= 20) {
            return 20;
        } else if (originalDistance <= 30) {
            return 30;
        } else if (originalDistance <= 40) {
            return 40;
        } else if (originalDistance <= 50) {
            return 50;
        }
        return originalDistance;
    }
}
