package com.bxm.localnews.user.support.impl;

import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.domain.UserCountMapper;
import com.bxm.localnews.user.dto.UserAccountDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.event.LocationActionEvent;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.dto.UserInviteRankDTO;
import com.bxm.localnews.user.model.dto.UserSaveCashDTO;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.support.UserRankService;
import com.bxm.localnews.user.vo.Tuple;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.data.redis.core.DefaultTypedTuple;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.user.constant.RedisConfig.*;

@SuppressWarnings("unchecked")
@Service
public class UserRankServiceImpl implements UserRankService {

    private final UserCountMapper userCountMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final RedisZSetAdapter redisZSetAdapter;

    private UserAccountService userAccountService;

    private final UserService userService;

    private final UserProperties userProperties;

    private static final int LIMIT = 500;

    @Autowired
    public UserRankServiceImpl(UserCountMapper userCountMapper,
                               RedisHashMapAdapter redisHashMapAdapter,
                               RedisZSetAdapter redisZSetAdapter,
                               UserService userService,
                               UserProperties userProperties) {
        this.userCountMapper = userCountMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisZSetAdapter = redisZSetAdapter;
        this.userService = userService;
        this.userProperties = userProperties;
    }


    private UserAccountService getUserAccountService() {
        if (userAccountService == null) {
            userAccountService = SpringContextHolder.getBean(UserAccountService.class);
        }
        return userAccountService;
    }

    @Override
    public Long getTotal(String areaCode) {
        if (StringUtils.isBlank(areaCode)) {
            return 0L;
        }

        Long total = redisHashMapAdapter.get(USER_TOTAL_CACHE, areaCode, Long.class);
        if (total == null) {
            total = 0L;
        }

        return total;
    }

    @Override
    public Long getVipTotal(String areaCode) {
        if (StringUtils.isBlank(areaCode)) {
            return 0L;
        }

        Long total = redisHashMapAdapter.get(USER_TOTAL_VIP_CACHE, areaCode, Long.class);
        if (total == null) {
            total = 0L;
        }

        // 增加虚拟人数
        total = total * userProperties.getVirtualMultiple() + userProperties.getVirtualTotalBase();

        return total;
    }

    private KeyGenerator buildKey(String areaCode) {
        return AREA_REBATE_RANK_CACHE.copy().appendKey(areaCode);
    }

    @Override
    public Long getSaveCashRank(Long userId, String areaCode) {
        KeyGenerator key = buildKey(areaCode);
        Long rank = redisZSetAdapter.reverseRank(key, userId);

        if (rank == null) {
            UserAccountDTO userAccount = getUserAccountService().getUserAccount(userId);
            BigDecimal saveCash = userAccount.getSaveCash();

            if (saveCash == null) {
                saveCash = BigDecimal.ZERO;
            } else {
                redisZSetAdapter.add(key, userId, saveCash.doubleValue());
            }

            if (BigDecimal.ZERO.equals(saveCash)) {
                rank = 0L;
            }
        } else {
            // 默认排序从0开始，+1
            rank += 1;
        }

        // 如果用户的节省金额为0，则排行也为0
        if (!Objects.equals(rank, 0L)) {
            Double score = redisZSetAdapter.score(key, userId);

            if (score == null || Objects.equals(score, 0D)) {
                rank = 0L;
            }
        }

        return rank;
    }

    @Override
    public Long getInviteNumRank(Long userId, String areaCode) {
        KeyGenerator rankKey = buildInviteRankKey(areaCode);

        // 在排行榜中，则直接返回排行
        Long rank = redisZSetAdapter.reverseRank(rankKey, userId);

        if (rank == null) {
            // 不在排行榜中，并且收徒数量为0，则返回0
            UserInfoDTO userCache = userService.getUserCache(userId);

            if (userCache.getInviteNum() != null && userCache.getInviteNum() > 0) {
                redisZSetAdapter.add(rankKey, userId, Double.valueOf(userCache.getInviteNum()));
            } else {
                rank = 0L;
            }
        } else {
            rank += 1;
        }

        return rank;
    }

    @EventListener
    public void changeLocation(LocationActionEvent event) {
        // 监听用户切换位置的事件
        // com.bxm.localnews.user.event.subscribe.UserActionStreamSubscriber
        moveRank(event.getUserId(), event.getOldLocationCode(), event.getLocationCode());
    }

    @Override
    public void moveRank(Long userId, String sourceAreaCode, String targetAreaCode) {
        redisZSetAdapter.remove(buildKey(targetAreaCode), userId);
        redisZSetAdapter.remove(buildKey(sourceAreaCode), userId);

        // 获取用户的目标地区排行，达到重新加载的目的
        getSaveCashRank(userId, targetAreaCode);

        redisZSetAdapter.remove(buildInviteRankKey(targetAreaCode), userId);
        redisZSetAdapter.remove(buildInviteRankKey(sourceAreaCode), userId);

        getInviteNumRank(userId, targetAreaCode);
    }

    @Override
    public void reloadUserSaveCashRank(Long userId) {
        UserInfoDTO userCache = userService.getUserCache(userId);

        if (Objects.nonNull(userCache.getLocationCode())) {
            redisZSetAdapter.remove(buildKey(userCache.getLocationCode()), userId);
            getSaveCashRank(userId, userCache.getLocationCode());
        }
    }

    @Override
    public void execInitCache() {
        // 初始化或更新各个地区总人数的缓存
        List<Tuple> userRankList = userCountMapper.queryUserTotal();

        if (CollectionUtils.isNotEmpty(userRankList)) {
            Map<String, Integer> userRankMap = userRankList.stream().collect(Collectors.toMap(Tuple::getLabel, Tuple::getTotal));
            redisHashMapAdapter.putAll(USER_TOTAL_CACHE, userRankMap);
        }

        // 更新各个地区VIP总人数的缓存
        List<Tuple> userVipRankList = userCountMapper.queryVipTotal();

        if (CollectionUtils.isNotEmpty(userVipRankList)) {
            Map<String, Integer> userRankMap = userVipRankList.stream().collect(Collectors.toMap(Tuple::getLabel, Tuple::getTotal));
            redisHashMapAdapter.putAll(USER_TOTAL_VIP_CACHE, userRankMap);
        }
    }

    @Override
    public void refreshInviteRank() {
        Date date = DateUtils.addField(new Date(), Calendar.MINUTE, -60);

        loadInviteRank(0, date);
    }

    private void loadInviteRank(int startNum, Date modifyDate) {
        List<UserInviteRankDTO> inviteRanks = userCountMapper.queryInviteByPage(startNum, LIMIT, modifyDate);

        Map<String, Set<ZSetOperations.TypedTuple>> areaMap = Maps.newHashMap();

        for (UserInviteRankDTO userInviteRank : inviteRanks) {
            Set<ZSetOperations.TypedTuple> tuples = areaMap.getOrDefault(userInviteRank.getLocationCode(), Sets.newHashSet());

            if (userInviteRank.getInviteNum() != null) {
                tuples.add(new DefaultTypedTuple(userInviteRank.getUserId(), Double.valueOf(userInviteRank.getInviteNum())));
            }

            areaMap.put(userInviteRank.getLocationCode(), tuples);
        }

        if (areaMap.size() > 0) {
            areaMap.forEach((areaCode, tuples) -> redisZSetAdapter.add(buildInviteRankKey(areaCode), tuples));

            // 如果存在下一页数据
            if (inviteRanks.size() > 0 && inviteRanks.size() == LIMIT) {
                startNum = startNum + LIMIT;
                loadInviteRank(startNum, modifyDate);
            }
        }
    }

    private KeyGenerator buildInviteRankKey(String areaCode) {
        return AREA_INVITE_RANK_CACHE.copy().appendKey(areaCode);
    }

    @Override
    public void refreshSaveCashRank() {
        // 查询省钱金额大于0的用户
        for (int i = 0; i < 10; i++) {
            int startNum = 0;

            loadSaveCash(i, startNum);
        }
    }

    private void loadSaveCash(int tableIndex, int startNum) {
        List<UserSaveCashDTO> saveCashList = userCountMapper.querySaveCashByPage(tableIndex, startNum, LIMIT);

        Map<String, Set<ZSetOperations.TypedTuple>> areaMap = Maps.newHashMap();

        for (UserSaveCashDTO userSaveCashDTO : saveCashList) {
            Set<ZSetOperations.TypedTuple> tuples = areaMap.getOrDefault(userSaveCashDTO.getLocationCode(), Sets.newHashSet());

            if (userSaveCashDTO.getSaveCash() != null) {
                tuples.add(new DefaultTypedTuple(userSaveCashDTO.getUserId(), userSaveCashDTO.getSaveCash()));
            }

            areaMap.put(userSaveCashDTO.getLocationCode(), tuples);
        }

        if (areaMap.size() > 0) {
            areaMap.forEach((areaCode, tuples) -> redisZSetAdapter.add(buildKey(areaCode), tuples));

            // 如果存在下一页数据
            if (saveCashList.size() > 0 && saveCashList.size() == LIMIT) {
                startNum = startNum + LIMIT;
                loadSaveCash(tableIndex, startNum);
            }
        }
    }
}




















