package com.bxm.localnews.user.tbk;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.tbk.factory.TaobaoClientFactory;
import com.bxm.localnews.user.domain.UserBindHistoryMapper;
import com.bxm.localnews.user.dto.RelationInfoRspDTO;
import com.bxm.localnews.user.dto.UserBindHistory;
import com.bxm.localnews.user.dto.UserTbkInfo;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.TbOpenBindRes;
import com.bxm.localnews.user.properties.TbOpenOrderProperties;
import com.bxm.localnews.user.vo.User;
import com.bxm.newidea.component.enums.AppNameEnum;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.taobao.api.request.TbkScPublisherInfoSaveRequest;
import com.taobao.api.request.TopAuthTokenCreateRequest;
import com.taobao.api.response.TbkScPublisherInfoSaveResponse;
import com.taobao.api.response.TopAuthTokenCreateResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.parseObject;

@Slf4j
@Service
public class UserTbkBindServiceImpl implements UserTbkBindService {

    @Autowired
    private TbOpenOrderProperties tbOpenOrderProperties;

    @Autowired
    private TaobaoClientFactory taobaoClientFactory;

    @Autowired
    private UserService userService;

    @Autowired
    private UserBindHistoryMapper userBindHistoryMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Override
    public RelationInfoRspDTO relateUserInfo(Long userId, String code) {
        log.info("绑定淘宝请求参数: userId: {} code: {}", userId, code);

        RelationInfoRspDTO result = new RelationInfoRspDTO();

        if (Objects.isNull(userId) || StringUtils.isBlank(code)) {
            result.setOk(false);
            result.setDesc("请求参数有误");
            return result;
        }

        // code换token
        TbOpenBindRes res = getTbToken(code);
        if (StringUtils.isBlank(res.getAccessToken())) {
            result.setOk(false);
            result.setDesc(res.getErrorMsg());
            createBindHistory(userId, res.getSourceRes(), 0);
            return result;
        }

        String accessToken = res.getAccessToken();
        // relation id
        res = getBindIdByTokenAndType(accessToken, 1);
        if (StringUtils.isBlank(res.getRelationId())) {
            log.warn("淘宝绑定失败，淘宝API返回relation id 信息为空, userId:{}, code:{} errMsg: {}", userId, code,
                    res.getErrorMsg());

            result.setOk(false);
            result.setDesc(res.getErrorMsg());
            createBindHistory(userId, res.getSourceRes(), 0);
            return result;
        }
        String relationId = res.getRelationId();

        // 查看是否绑定过
        UserTbkInfo info = userService.selectUserFromCacheByRelationId(relationId);
        if (Objects.nonNull(info)) {
            if (!Objects.equals(info.getId(), userId)) {
                log.warn("淘宝绑定失败，重复进行绑定, userId:{}, code:{}", userId, code);

                result.setOk(false);
                result.setRelationId(relationId);
                result.setDesc("当前账号已绑定用户: " + info.getNickname());
                return result;
            }

            log.warn("淘宝绑定成功，但进行了重复绑定, userId: {}, code: {}", userId, code);
            result.setOk(true);
            result.setRelationId(relationId);
            result.setDesc("绑定成功");
            return result;
        }

        // special id
        res = getBindIdByTokenAndType(accessToken, 2);
        if (StringUtils.isBlank(res.getSpecialId())) {
            log.warn("淘宝绑定失败，淘宝API返回special id 信息为空, userId:{}, code:{} errMsg: {}", userId, code,
                    res.getErrorMsg());

            result.setOk(false);
            result.setDesc(res.getErrorMsg());

            createBindHistory(userId, res.getSourceRes(), 0);
            return result;
        }

        // 更新绑定信息
        userService.updateTbkInfo(userId, relationId, res.getSpecialId());
        // 绑定记录
        createBindHistory(userId, res.getSourceRes(), 1);


        result.setOk(true);
        result.setRelationId(relationId);
        result.setDesc("绑定成功");
        return result;
    }

    /**
     * 创建绑定记录
     *
     * @param userId     userId
     * @param result     原始返回数据
     * @param bindResult 绑定状态 0失败 1成功
     */
    private void createBindHistory(Long userId, String result, Integer bindResult) {
        UserBindHistory bindHistory = new UserBindHistory();
        bindHistory.setId(sequenceCreater.nextLongId());
        bindHistory.setUserId(userId);
        bindHistory.setPlatform("淘宝联盟");
        bindHistory.setNickname("");
        bindHistory.setHeadImg("");
        bindHistory.setBindResult(bindResult);
        bindHistory.setSource(result);
        bindHistory.setCreateTime(new Date());


        userBindHistoryMapper.insertSelective(bindHistory);
    }


    /**
     * 绑定渠道关系
     *
     * @return
     */
    private TbOpenBindRes getBindIdByTokenAndType(String accessToken, int type) {
        TbOpenBindRes res = TbOpenBindRes.builder().build();
        String sourceRes = "";
        try {

            TbkScPublisherInfoSaveRequest req = new TbkScPublisherInfoSaveRequest();
            req.setOfflineScene("4");
            req.setOnlineScene("3");
            if (type == 1) {
                req.setInviterCode(tbOpenOrderProperties.getRelationInviteCode());
            } else {
                req.setInviterCode(tbOpenOrderProperties.getSpecialInviteCode());
            }

            req.setInfoType(1L);
            req.setNote("其他");

            if (log.isDebugEnabled()) {
                log.debug("绑定渠道关系请求参数: {}", JSON.toJSONString(req));
            }

            // 使用万事通的配置信息获取淘宝 client
            TbkScPublisherInfoSaveResponse resp = taobaoClientFactory.getClient(AppNameEnum.LOCAL_NEWS.name())
                    .execute(req, accessToken);

            sourceRes = JSON.toJSONString(resp);
            if (log.isDebugEnabled()) {
                log.debug("绑定渠道关系返回参数: {}", sourceRes);
            }

            if (Objects.isNull(resp)) {
                return TbOpenBindRes.builder().errorMsg("绑定渠道关系失败").sourceRes(sourceRes).build();
            }

            if (Objects.isNull(resp.getData())) {
                return TbOpenBindRes.builder().errorMsg(resp.getSubMsg()).sourceRes(sourceRes).build();
            }

            res = TbOpenBindRes.builder()
                    .relationId(Objects.toString(resp.getData().getRelationId()))
                    .specialId(Objects.toString(resp.getData().getSpecialId()))
                    .sourceRes(sourceRes)
                    .build();

        } catch (Exception e) {
            log.error("绑定渠道关系失败，请求参数: sessionKey: {} type: {}", accessToken, type, e);
        }

        log.info("根据 accessToken: {} type: {} 获取到的绑定信息: {}", accessToken, type, JSON.toJSONString(res));
        return res;
    }

    /**
     * code换取淘宝token
     */
    private TbOpenBindRes getTbToken(String code) {
        String accessToken = "";
        String sourceRes = "";
        try {
            TopAuthTokenCreateRequest req = new TopAuthTokenCreateRequest();
            req.setCode(code);
            if (log.isDebugEnabled()) {
                log.debug("code换取 access token 请求参数: {}", JSON.toJSONString(req));
            }

            // 使用万事通的配置信息获取淘宝 client
            TopAuthTokenCreateResponse rsp = taobaoClientFactory.getClient(AppNameEnum.LOCAL_NEWS.name())
                    .execute(req);

            sourceRes = JSON.toJSONString(rsp);
            if (log.isDebugEnabled()) {
                log.debug("code换取 access token 返回参数: {}", sourceRes);
            }

            if (Objects.isNull(rsp)) {
                return TbOpenBindRes.builder().errorMsg("获取access token失败").sourceRes(sourceRes).build();
            }

            if (!rsp.isSuccess() || StringUtils.isBlank(rsp.getTokenResult())) {
                return TbOpenBindRes.builder().errorMsg(rsp.getSubMsg()).sourceRes(sourceRes).build();
            }

            JSONObject token = parseObject(rsp.getTokenResult());
            if (Objects.isNull(token) || !token.containsKey("access_token")) {
                return TbOpenBindRes.builder().errorMsg("获取access token失败").sourceRes(sourceRes).build();
            }

            accessToken = token.getString("access_token");
        } catch (Exception e) {
            log.error("code: {} 换取淘宝token，请求淘宝接口失败", code, e);
        }

        log.info("code: {} 换取淘宝token: {}", code, accessToken);
        return TbOpenBindRes.builder().accessToken(accessToken).sourceRes(sourceRes).build();
    }

    @Override
    public RelationInfoRspDTO checkRelation(Long userId) {
        RelationInfoRspDTO result = new RelationInfoRspDTO();

        User info = userService.selectByPrimaryKey(userId);
        
        if (Objects.isNull(userId) || Objects.isNull(info)) {
            result.setOk(false);
            result.setDesc("用户不存在");
            return result;
        }

        if (StringUtils.isNotBlank(info.getTbkRelationId())) {
            result.setOk(true);
            result.setRelationId(info.getTbkRelationId());
            result.setDesc("已绑定");
        } else {
            result.setOk(false);
            result.setDesc("未绑定");
        }

        return result;
    }
}
