package com.bxm.localnews.user.utils;

import com.bxm.newidea.component.tools.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.Objects;

/**
 * 数值转化工具类
 *
 * @author liujia
 * @date 2020/6/3 15:12
 **/
public class NumberConvertUtils {
    private NumberConvertUtils() {
    }

    private static char[] CHAR_ARRAY = "0123456789ABCDEFGHJKLMNPQRSTUVWXYZ".toCharArray();

    private static ThreadLocal<DecimalFormat> FORMAT_THREAD_LOCAL
            = ThreadLocal.withInitial(() -> {
        DecimalFormat decimalFormat = new DecimalFormat("##.##");
        decimalFormat.setRoundingMode(RoundingMode.DOWN);
        return decimalFormat;
    });

    public static String convertToShortCode(Long num, Integer minLength) {
        StringBuilder res = new StringBuilder();

        int length = CHAR_ARRAY.length;

        while (num > 0) {
            res.append(CHAR_ARRAY[(int) (num % length)]);
            num /= length;
        }

        if (res.length() < minLength) {
            for (int i = 0; i < minLength - res.length(); i++) {
                res.append("0");
            }
        }

        return StringUtils.reverse(res.toString());
    }

    /**
     * 对超过一万的数据进行格式化处理
     *
     * @param value 原始数据
     * @return 格式化结果
     */
    public static String formatNum(Long value) {
        if (value == null) {
            return "0";
        }

        if (value < 10000L) {
            return value.toString();
        }

        float num = (float) value / 10000;

        return FORMAT_THREAD_LOCAL.get().format(num) + "万";
    }

    private static Range[] SAVE_CACHE_RANGE_ARRAY = {
            new Range(0, 1, 0.07F),
            new Range(1, 3, 0.12F),
            new Range(3, 5, 0.19F),
            new Range(5, 10, 0.28F),
            new Range(10, 20, 0.39F),
            new Range(20, 50, 0.45F),
            new Range(50, 100, 0.56F),
            new Range(100, 200, 0.69F),
            new Range(200, 300, 0.77F),
            new Range(300, 500, 0.83F),
            new Range(500, Integer.MAX_VALUE, 0.97F),
    };

    private static Range[] INVITE_NUM_RANGE_ARRAY = {
            new Range(0, 1, 0.02F),
            new Range(1, 5, 0.05F),
            new Range(5, 10, 0.13F),
            new Range(10, 20, 0.29F),
            new Range(20, 50, 0.45F),
            new Range(50, 100, 0.56F),
            new Range(100, 200, 0.69F),
            new Range(200, 300, 0.77F),
            new Range(300, 500, 0.83F),
            new Range(500, Integer.MAX_VALUE, 0.97F),
    };

    /**
     * 格式化省钱金额
     * 根据省钱金额进行一定的数据修饰
     *
     * @param numerator   当前用户的省钱排名
     * @param denominator 总人数
     * @param saveCash    省钱金额
     * @return 格式化后的结果
     */
    public static String formatSaveCashPercent(Long numerator, Long denominator, BigDecimal saveCash) {
        if (saveCash == null) {
            saveCash = BigDecimal.ZERO;
        }

        // 判断省钱金额所在区间
        float offPercent = 1;
        for (Range range : SAVE_CACHE_RANGE_ARRAY) {
            if (saveCash.floatValue() <= range.right) {
                offPercent = range.percent;
                break;
            }
        }

        return formatPercent(numerator, denominator, offPercent);
    }

    /**
     * 格式化收徒数量的百分比
     * 收徒排名前20的用户比例不变，为了与收徒排行榜保持一致
     * 排名靠后的用户，根据产品提供的数据比例进行修饰
     *
     * @param numerator   收徒排名
     * @param denominator 地区总人数
     * @param inviteNums  收徒数量
     * @return 收徒所占比例
     */
    public static String formatInvitePercent(Long numerator, Long denominator, Long inviteNums) {
        if (inviteNums == null) {
            inviteNums = 0L;
        }

        // 判断收徒数量所在区间
        float offPercent = 1;

        if (numerator > 20) {
            for (Range range : INVITE_NUM_RANGE_ARRAY) {
                if (inviteNums <= range.right) {
                    offPercent = range.percent;
                    break;
                }
            }
        }

        return formatPercent(numerator, denominator, offPercent);
    }

    private static String formatPercent(Long numerator, Long denominator, float offPercent) {
        if (numerator == null
                || denominator == null
                || Objects.equals(denominator, 0L)
                || Objects.equals(numerator, 0L)) {
            return "0%";
        }

        float percent = (((float) denominator - numerator) / denominator * 100) * offPercent;

        return FORMAT_THREAD_LOCAL.get().format(percent) + "%";
    }

    private static class Range {
        int left;

        int right;

        float percent;

        Range(int left, int right, float percent) {
            this.left = left;
            this.right = right;
            this.percent = percent;
        }
    }
}
