/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.user.vip.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.constant.VipActiveCode;
import com.bxm.localnews.user.domain.UserVipHistoryMapper;
import com.bxm.localnews.user.domain.UserVipMapper;
import com.bxm.localnews.user.domain.vip.UserActiveCodeMapper;
import com.bxm.localnews.user.dto.LocationUserInfoDTO;
import com.bxm.localnews.user.dto.UserAccountDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.vip.ActiveCodeDTO;
import com.bxm.localnews.user.dto.vip.VipActivationResultDTO;
import com.bxm.localnews.user.enums.ActivationVipEnum;
import com.bxm.localnews.user.enums.ActivationVipStatusEnum;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.invite.InviteRecordService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.VipCardInfoDTO;
import com.bxm.localnews.user.model.VipInfoDetailDTO;
import com.bxm.localnews.user.model.param.VipQueryParam;
import com.bxm.localnews.user.param.ActivationUserVipParam;
import com.bxm.localnews.user.param.VipActivationCodeParam;
import com.bxm.localnews.user.param.VipPaymentCallbackParam;
import com.bxm.localnews.user.properties.UserVipProperties;
import com.bxm.localnews.user.support.UserRankService;
import com.bxm.localnews.user.utils.NumberConvertUtils;
import com.bxm.localnews.user.vip.ActiveCodeService;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vip.VipCardService;
import com.bxm.localnews.user.vip.activation.ActivationVipManager;
import com.bxm.localnews.user.vip.code.ActiveCodeProcessContext;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserVip;
import com.bxm.localnews.user.vo.UserVipHistory;
import com.bxm.localnews.user.vo.vip.UserActiveCodeBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.bxm.localnews.user.constant.RedisConfig.HAS_NEW_SAVE_CASH_EVENT_CACHE;
import static com.bxm.localnews.user.constant.RedisConfig.USER_VIP_INFO;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
public class UserVipServiceImpl extends BaseService implements UserVipService {

    private final UserVipProperties userVipProperties;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserVipMapper userVipMapper;

    private final UserVipHistoryMapper userVipHistoryMapper;

    private final ActivationVipManager activationVipManager;

    private final ActiveCodeService activeCodeService;

    private final VipCardService vipCardService;

    private UserService userService;

    private final UserActiveCodeMapper userActiveCodeMapper;

    private final InviteRecordService inviteRecordService;

    private UserAccountService userAccountService;

    private UserRankService userRankService;

    private final RedisSetAdapter redisSetAdapter;

    @Autowired
    public UserVipServiceImpl(UserVipProperties userVipProperties,
                              RedisHashMapAdapter redisHashMapAdapter,
                              UserVipMapper userVipMapper,
                              UserVipHistoryMapper userVipHistoryMapper,
                              ActivationVipManager activationVipManager,
                              ActiveCodeService activeCodeService,
                              VipCardService vipCardService,
                              UserActiveCodeMapper userActiveCodeMapper,
                              InviteRecordService inviteRecordService,
                              UserAccountService userAccountService,
                              UserRankService userRankService,
                              RedisSetAdapter redisSetAdapter) {
        this.userVipProperties = userVipProperties;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userVipMapper = userVipMapper;
        this.userVipHistoryMapper = userVipHistoryMapper;
        this.activationVipManager = activationVipManager;
        this.activeCodeService = activeCodeService;
        this.vipCardService = vipCardService;
        this.userActiveCodeMapper = userActiveCodeMapper;
        this.inviteRecordService = inviteRecordService;
        this.userAccountService = userAccountService;
        this.userRankService = userRankService;
        this.redisSetAdapter = redisSetAdapter;
    }

    public UserService getUserService() {
        if (userService == null) {
            userService = SpringContextHolder.getBean(UserService.class);
        }
        return userService;
    }

    @Override
    public List<VipCardInfoDTO> getVipCardList() {
        return userVipProperties.getCardConfig();
    }

    @Override
    public VipCardInfoDTO getVipCardInfo(String id) {
        List<VipCardInfoDTO> list = this.getVipCardList();
        for (VipCardInfoDTO entity : list) {
            if (entity.getId().equals(id)) {
                return entity;
            }
        }
        return null;
    }

    @Override
    public VipActivationResultDTO execActivationVipByCode(VipActivationCodeParam vipActivationCodeParam, Integer type) {
        checkParam(vipActivationCodeParam);

        ActivationUserVipParam param = new ActivationUserVipParam();

        BeanUtils.copyProperties(vipActivationCodeParam, param);

        param.setCode(vipActivationCodeParam.getCode());
        param.setUserId(vipActivationCodeParam.getUserId());
        param.setActivationVipEnum(ActivationVipEnum.ACTIVATION_CODE);
        param.setPlatformEnum(vipActivationCodeParam.getPlatform());
        if (StringUtils.isEmpty(param.getChannel())) {
            param.setChannel(vipActivationCodeParam.getChnl());
        }
        logger.debug("激活码激活参数：[{}]", JSONObject.toJSONString(vipActivationCodeParam));
        if (Objects.equals(type, 1)) {
            VipActivationResultDTO result = new VipActivationResultDTO();
            noChanceToActiveVip(param, result);
            if (Objects.equals(result.getResult(), VipActiveCode.SUCCESS) || Objects.equals(result.getResult(), VipActiveCode.NOT_FOUND)) {
                return result;
            }
        }
        //激活vip
        Message message = activationVipManager.activationVip(param);

        VipActivationResultDTO result = new VipActivationResultDTO();
        result.setMsg(message.getLastMessage());
        result.setResult(message.getParam(VipActiveCode.KEY));

        return result;
    }

    private void checkParam(VipActivationCodeParam param) {
        if (null == param.getUserId()) {
            throw new RuntimeException("用户id不能为空");
        }
        if (StringUtils.isEmpty(param.getCode())) {
            throw new RuntimeException("激活码不能为空");
        }

    }

    /**
     * 激活码没有激活次数
     *
     * @param param  ： 激活VIP信息
     * @param result : 返回激活结果：SUCCESS 和 NOT_FOUND 不走后续逻辑，否则走
     **/
    private void noChanceToActiveVip(ActivationUserVipParam param, VipActivationResultDTO result) {
        if (Objects.equals(param.getPlatformEnum(), 3)) {
            User user = getUserService().selectByPrimaryKey(param.getUserId());
            //站外的新用户会走这个逻辑
            if (!user.getState().equals(AppConst.USER_STATE.NOT_ACTIVE)) {
                result.setResult(VipActiveCode.NO_TIMES);
                return;
            }
            UserActiveCodeBean activeCodeBean = userActiveCodeMapper.selectByCode(param.getCode());
            if (null == activeCodeBean) {
                result.setResult(VipActiveCode.NOT_FOUND);
                result.setMsg("激活码错误");
                return;
            }
            if (activeCodeBean.getMaxTimes() < 1) {
                //校验师徒关系
                inviteRecordService.addInviteRecord(activeCodeBean.getUserId(), param.getUserId(), param.getChannel());
                User userInvite = getUserService().selectByPrimaryKey(activeCodeBean.getUserId());
                //增加弹窗
                String windowInfo = "哎呀你来晚了\n" +
                        "你的好友" + userInvite.getNickname() + "的5个VIP名额\n" +
                        "已经被其他朋友抢完了";
                redisHashMapAdapter.put(UserRedisKey.ACTIVE_VIP_HAS_NOT_CHANCE, String.valueOf(param.getUserId()), windowInfo);
                result.setResult(VipActiveCode.SUCCESS);
                result.setMsg("激活成功");
                return;
            }
        }
        result.setResult(VipActiveCode.NO_TIMES);
    }


    @Override
    public Boolean execActivationVipByPay(VipPaymentCallbackParam callbackParam) {
        ActivationUserVipParam param = new ActivationUserVipParam();
        param.setOrderId(callbackParam.getOrderId());
        param.setPlatformEnum(null == callbackParam.getPlatformEnum()
                ? PlatformEnum.ANDROID.getCode()
                : callbackParam.getPlatformEnum());
        param.setChannel(callbackParam.getChannel());
        param.setUserId(callbackParam.getUserId());
        param.setVipCardId(callbackParam.getVipCardId());
        param.setActivationVipEnum(ActivationVipEnum.PAY);

        param.setPlatform(param.getPlatformEnum());
        param.setChnl(param.getChannel());
        //激活vip
        return activationVipManager.activationVip(param).isSuccess();
    }

    @Override
    public Message createVip(ActivationUserVipParam param) {
        UserVip existsVip = userVipMapper.selectByUserId(param.getUserId());

        if (null != existsVip) {
            logger.warn("用户VIP资格重复激活，请求参数：{}", JSON.toJSONString(param));

            return Message.build(false, "用户已经是VIP,无需重复激活");
        }

        ActiveCodeProcessContext context = ActiveCodeProcessContext.builder()
                .userId(param.getUserId())
                .code(param.getCode())
                .activationVipType(param.getActivationVipEnum())
                .build();

        // 根据当前的激活类型处理激活码的后续逻辑，获取当前用户的VIP卡号与新的激活码
        ActiveCodeDTO activeCodeDTO = activeCodeService.process(context);


        Date now = new Date();
        // 新增VIP记录
        UserVip userVip = new UserVip();
        userVip.setId(nextId());
        userVip.setUserId(param.getUserId());
        //设置对应的卡号和激活码
        userVip.setCard(activeCodeDTO.getCardNo());
        userVip.setActiveCode(activeCodeDTO.getActiveCode());
        userVip.setCreateTime(now);
        userVip.setChannel(param.getChannel());
        userVip.setStatus(1);
        userVip.setExpiredDate(DateUtils.addField(now, Calendar.DAY_OF_YEAR, param.getDuration()));
        userVip.setType(param.getActivationVipEnum().getCode());

        int result = userVipMapper.insert(userVip);

        // 记录VIP激活历史
        return saveVipHistory(param, now, userVip, result, "开通VIP");
    }

    private Message saveVipHistory(ActivationUserVipParam param, Date now, UserVip userVip, int result, String operation) {
        UserVipHistory history = new UserVipHistory();
        history.setId(nextId());
        history.setVipId(userVip.getId());
        history.setOperation(operation);
        history.setUserId(param.getUserId());
        history.setOrderId(param.getOrderId());
        history.setCreateTime(now);

        result += userVipHistoryMapper.insert(history);

        removeCache(param.getUserId());

        return Message.build(result);
    }

    @Override
    public Message execRenew(ActivationUserVipParam param) {
        UserVip userVip = getUserVipByUid(param.getUserId());

        if (userVip == null || userVip.getId() == null) {
            logger.warn("续约时，用户未开通VIP，请求参数：{}", JSON.toJSONString(param));
            return createVip(param);
        }

        // 更新VIP状态与到期时间
        Date now = new Date();
        userVip.setStatus(ActivationVipStatusEnum.USEFUL.getStatus());
        userVip.setExpiredDate(DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR, param.getDuration()));
        int result = userVipMapper.vipRenew(userVip);

        // 记录续约历史
        return saveVipHistory(param, now, userVip, result, "重新开通VIP");
    }

    /**
     * 执行卡号修复的逻辑
     * 当用户在站外购买VIP或其他原因，导致当时获取不到地区信息，在进入APP获取用户VIP信息时补录卡号信息
     * 这里主要是修复一些历史的数据，正常情况不会进入，需要记录相应的日志来排查问题
     *
     * @param userVip 开通VIP的用户信息
     */
    private void execFixCardNo(UserVip userVip) {
        //VIP处于可用状态，并且没有卡号时，才需要重新创建新的卡号
        if (Objects.equals(userVip.getUsable(), Boolean.TRUE) && isBlank(userVip.getCard())) {
            LocationUserInfoDTO locationUserInfo = getUserService().getLocationUserInfo(userVip.getUserId());

            if (Objects.nonNull(locationUserInfo) && isNotBlank(locationUserInfo.getLocationCode())) {
                String newCardNo = vipCardService.nextCard(locationUserInfo.getLocationCode());

                userVipMapper.updateCardNo(userVip.getUserId(), newCardNo);

                userVip.setCard(newCardNo);

                removeCache(userVip.getUserId());

                logger.info("填充用户卡号信息，用户ID：{},所属地区：{},卡号：{}",
                        userVip.getId(),
                        locationUserInfo.getLocationCode(),
                        newCardNo);
            }
        }
    }

    @Override
    public UserVip getUserVipByUid(Long userId) {
        KeyGenerator key = buildVipInfoCacheKey(userId);
        UserVip userVipInfo = redisHashMapAdapter.get(key, String.valueOf(userId), UserVip.class);

        if (null == userVipInfo) {
            userVipInfo = userVipMapper.selectByUserId(userId);

            if (null == userVipInfo) {
                // 不存在对应的记录，默认显示为待激活
                userVipInfo = new UserVip(userId, ActivationVipStatusEnum.WAIT_USE.getStatus());
            } else {
                //获取激活码信息，包括激活码、激活次数等
                UserActiveCodeBean activeCodeInfo = activeCodeService.getActiveCode(userVipInfo.getActiveCode());
                userVipInfo.setLeftOpenNum(activeCodeInfo.getMaxTimes());
            }

            userVipInfo.setUsable(Objects.equals(ActivationVipStatusEnum.USEFUL.getStatus(), userVipInfo.getStatus()));

            redisHashMapAdapter.put(key, String.valueOf(userId), userVipInfo);
        }

        execFixCardNo(userVipInfo);

        return userVipInfo;
    }

    @Override
    public Boolean isVip(Long userId) {
        if (null == userId) {
            return false;
        }
        UserVip userVip = this.getUserVipByUid(userId);
        return Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.USEFUL.getStatus());
    }


    @Override
    public List<UserVip> getExpiredVipList(Integer status, Date expiredDate) {
        return userVipMapper.getUserVipList(status, expiredDate);
    }

    @Override
    public int updateBatchStatusByUid(Integer status, List<Long> userIds) {
        return userVipMapper.updateBatchStatusByUid(status, userIds);
    }

    private KeyGenerator buildVipInfoCacheKey(Long userId) {
        return USER_VIP_INFO.copy().appendKey(userId % 10);
    }

    @Override
    public void removeCache(Long userId) {
        //删除VIP缓存信息
        redisHashMapAdapter.remove(buildVipInfoCacheKey(userId), String.valueOf(userId));
    }

    @Override
    public VipInfoDetailDTO getVipDetail(VipQueryParam param) {
        UserVip userVip = this.getUserVipByUid(param.getUserId());

        VipInfoDetailDTO userVipDetail = new VipInfoDetailDTO();
        BeanUtils.copyProperties(userVip, userVipDetail);

        userVipDetail.setExpiredDate(null == userVip.getExpiredDate() ? "" : DateUtils.formatDate(userVip.getExpiredDate()));

        // 用户累计已节省金额
        UserAccountDTO userAccount = userAccountService.getUserAccount(param.getUserId());
        if (userAccount.getSaveCash() != null) {
            userVipDetail.setRebateAmount(userAccount.getSaveCash());
        } else {
            userVipDetail.setRebateAmount(BigDecimal.ZERO);
        }

        // 设置用户头像
        UserInfoDTO userCache = getUserService().getUserCache(param.getUserId());
        userVipDetail.setNickName(userCache.getNickname());
        userVipDetail.setHeadImg(userCache.getHeadImg());
        userVipDetail.setPhone(StringUtils.hideMobile(userCache.getPhone()));

        // 如果是VIP主入口，则额外添加一些参数
        if ("MAIN".equals(param.getScene())) {
            Long vipTotal;
            String saveCashPercent;
            if (null == param.getAreaCode()) {
                vipTotal = 0L;
                saveCashPercent = "0%";
            } else {
                vipTotal = userRankService.getVipTotal(param.getAreaCode());

                Long saveCashRank = userRankService.getSaveCashRank(param.getUserId(), param.getAreaCode());
                Long total = userRankService.getTotal(param.getAreaCode());

                saveCashPercent = NumberConvertUtils.formatSaveCashPercent(saveCashRank, total, userAccount.getSaveCash());
            }

            // 省钱占比
            userVipDetail.setRebatePercent(saveCashPercent);
            // 当前地区的VIP总数
            userVipDetail.setTotalVip(NumberConvertUtils.formatNum(vipTotal));

            // 是否有新增的节省金额
            consumeSaveCashEvent(param, userVipDetail);
        }

        return userVipDetail;
    }

    @Override
    public Boolean activeVipPay(ActivationUserVipParam param) {
        if (null == param.getUserId()) {
            return false;
        }
        param.setActivationVipEnum(ActivationVipEnum.PAY);
        param.setChannel("buy goods");
        param.setVipCardId("1");
        return activationVipManager.activationVip(param).isSuccess();
    }

    /**
     * 用户产生新增的省钱金额时记录，用户访问VIP首页时消费掉，前端用于显示对应的动效
     *
     * @param param         查询参数
     * @param userVipDetail VIP详情
     */
    private void consumeSaveCashEvent(VipQueryParam param, VipInfoDetailDTO userVipDetail) {
        if (redisSetAdapter.exists(HAS_NEW_SAVE_CASH_EVENT_CACHE, param.getUserId())) {
            userVipDetail.setAddRebate(Boolean.TRUE);

            redisSetAdapter.remove(HAS_NEW_SAVE_CASH_EVENT_CACHE, param.getUserId());
        } else {
            userVipDetail.setAddRebate(Boolean.FALSE);
        }
    }
}
