package com.bxm.openlog.extension.client.ws;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.web.socket.PingMessage;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.client.WebSocketConnectionManager;
import org.springframework.web.socket.client.standard.StandardWebSocketClient;
import org.springframework.web.socket.handler.ConcurrentWebSocketSessionDecorator;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.io.IOException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author zhangdong
 * @date 2025/2/13
 */
@Slf4j
public class WsOpenLogClientHandler extends TextWebSocketHandler {

    private WebSocketSession session;
    private final ScheduledExecutorService scheduler = new ScheduledThreadPoolExecutor(2);
    private final WsOpenLogClientProperties properties;
    private final WebSocketConnectionManager manager;

    public WsOpenLogClientHandler(WsOpenLogClientProperties properties) {
        this.properties = properties;
        String endpoint = properties.getEndpoint();
        Preconditions.checkArgument(StringUtils.isNotBlank(endpoint), "endpoint is null or empty");
        StandardWebSocketClient socketClient = new StandardWebSocketClient();
        this.manager = new WebSocketConnectionManager(socketClient, this, endpoint);
        startHeartbeat();
        startReconnection();
    }

    @Override
    public void afterConnectionEstablished(WebSocketSession session) throws Exception {
        this.session = new ConcurrentWebSocketSessionDecorator(session, Integer.MAX_VALUE, Integer.MAX_VALUE);
    }


    public void sendMessage(String data) throws Exception {
        if (isClosed()) {
            throw new IOException("connection error");
        }
        session.sendMessage(new TextMessage(data));
    }


    private boolean isClosed() {
        return session == null || !session.isOpen();
    }

    private void startHeartbeat() {
        scheduler.scheduleAtFixedRate(() -> {
            if (isClosed()) {
                return;
            }
            try {
                PingMessage pingMessage = new PingMessage();
                session.sendMessage(pingMessage);
            } catch (Exception e) {
                log.error("ping error", e);
            }
        }, 2000, properties.getHeartCycle(), TimeUnit.MILLISECONDS);
    }

    private void startReconnection() {
        scheduler.scheduleAtFixedRate(() -> {
            if (!isClosed()) {
                return;
            }
            try {
                manager.stop();
                manager.start();
            } catch (Exception e) {
                log.error("reconnection error", e);
            }
        }, 0, properties.getReconnectionCycle(), TimeUnit.MILLISECONDS);
    }
}
