package com.bxm.openlog.sdk.listener.eventbus;

import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.Topic;
import com.bxm.openlog.sdk.serial.Serialization;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Listener;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.mq.autoconfigure.rocketmq.RocketmqProperties;
import com.bxm.warcar.mq.rocketmq.RocketmqConsumer;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.util.ClassUtils;

import java.util.List;
import java.util.Map;

/**
 * @author allen
 * @date 2022-01-21
 * @since 1.0
 */
@Slf4j
public class OpenLogEventBusFactory implements BeanPostProcessor, DisposableBean, ApplicationListener<ApplicationStartedEvent> {

    private final Map<String, OpenLogEventConfig> configFactoryMap = Maps.newHashMap();
    private final Map<String, Consumer> consumerMap = Maps.newHashMap();
    private final RocketmqProperties rocketmqProperties;
    private final Serialization serialization;

    public OpenLogEventBusFactory(RocketmqProperties rocketmqProperties, Serialization serialization) {
        this.rocketmqProperties = rocketmqProperties;
        this.serialization = serialization;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (!(bean instanceof OpenLogEventBusBean)) {
            return bean;
        }
        OpenLogEventBusBean busBean = (OpenLogEventBusBean) bean;

        String id = busBean.getId();
        Production production = busBean.getProduction();
        EventPark eventPark = busBean.getEventPark();
        Preconditions.checkNotNull(production, "production");
        Preconditions.checkNotNull(eventPark, "eventPark");

        List<OpenLogEventConfig> registry = Lists.newArrayList();
        busBean.bindTo(registry);
        if (CollectionUtils.isEmpty(registry)) {
            return bean;
        }
        for (OpenLogEventConfig config : registry) {
            String mt = config.getMt();
            String key = getKey(production, mt);
            configFactoryMap.put(key, config);

            log.info("Registered OpenLogEvent {} - {}", key, ClassUtils.getShortName(config.getEventType()));
        }

        String topic = Topic.dispatchMtLog(production.getName());

        String consumerId = ("CID_EVENT_" + id + "_" + topic).toUpperCase();

        EventBusSubscriberMessageListener listener = new EventBusSubscriberMessageListener(topic, consumerId, serialization, eventPark, configFactoryMap);

        Consumer consumer = createConsumer(listener);

        consumerMap.put(consumerId, consumer);

        if (log.isInfoEnabled()) {
            log.info("Registered consumer {} for topic {}", consumerId, topic);
        }

        return bean;
    }

    @Override
    public void destroy() throws Exception {
        for (Consumer consumer : consumerMap.values()) {
            consumer.shutdown();
        }
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        for (Consumer consumer : consumerMap.values()) {
            consumer.start();

            if (log.isInfoEnabled()) {
                Listener listener = consumer.getMessageListener();
                log.info("Consumer topic:{} id:{} instance:{} has been started!", listener.getTopic(), listener.getConsumerId(), consumer);
            }
        }
    }

    static String getKey(Production production, String mt) {
        return production + "-" + mt;
    }

    private Consumer createConsumer(SingleMessageListener listener) {
        String nameSrvAddr = rocketmqProperties.getNameSrvAddr();
        return new RocketmqConsumer(nameSrvAddr, listener, listener.getConsumerId(), rocketmqProperties.getBrokerAddrAliasMap());
    }
}
