package com.bxm.openlog.sdk.listener.eventbus;

import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.serial.Serialization;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.mq.ConsumeStatus;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.SingleMessageListener;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;

/**
 * @author allen
 * @date 2022-01-20
 * @since 1.0
 */
@Slf4j
class EventBusSubscriberMessageListener implements SingleMessageListener, BeanPostProcessor {

    private final String topic;
    private final String consumerId;
    private final Serialization serialization;
    private final EventPark eventPark;
    private final Map<String, OpenLogEventConfig> configFactoryMap;

    EventBusSubscriberMessageListener(String topic, String consumerId, Serialization serialization, EventPark eventPark,
                                      Map<String, OpenLogEventConfig> configFactoryMap) {
        this.topic = topic;
        this.consumerId = consumerId;
        this.serialization = serialization;
        this.eventPark = eventPark;
        this.configFactoryMap = configFactoryMap;
    }

    @Override
    public ConsumeStatus consume(Message message, Object context) {
        try {
            byte[] body = message.getBody();
            KeyValueMap keyValueMap = serialization.deserialize(body);

            Production production = Production.of(keyValueMap.getProduction());
            String mt = keyValueMap.getMt();
            String key = OpenLogEventBusFactory.getKey(production, mt);

            OpenLogEventConfig config = configFactoryMap.get(key);
            if (Objects.isNull(config)) {
                return ConsumeStatus.CONSUME_SUCCESS;
            }
            boolean isSendEvent = true;
            Predicate<KeyValueMap> condition = config.getCondition();
            if (Objects.nonNull(condition)) {
                isSendEvent = condition.test(keyValueMap);
            }
            if (isSendEvent) {
                OpenLogEvent logEvent = newInstance(config.getEventType(), this, keyValueMap);
                EventPark thisEventPark = config.getEventPark();
                if (Objects.nonNull(thisEventPark)) {
                    thisEventPark.post(logEvent);
                } else {
                    this.eventPark.post(logEvent);
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Check condition failed!");
                }
            }
        } catch (Exception e) {
            log.error("consume: ", e);
        }
        return ConsumeStatus.CONSUME_SUCCESS;
    }

    private static <T extends OpenLogEvent> T newInstance(Class<T> clazz, Object source, KeyValueMap keyValueMap) {
        try {
            Constructor<T> constructor = clazz.getConstructor(Object.class, KeyValueMap.class);
            return constructor.newInstance(source, keyValueMap);
        } catch (NoSuchMethodException | InstantiationException | IllegalAccessException | InvocationTargetException e) {
            log.error("newInstance: ", e);
            return null;
        }
    }

    @Override
    public String getTopic() {
        return topic;
    }

    @Override
    public String getConsumerId() {
        return consumerId;
    }
}
