package com.bxm.openlog.sdk.listener;

import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.Topic;
import com.bxm.openlog.sdk.serial.Serialization;
import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Listener;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.mq.autoconfigure.rocketmq.RocketmqProperties;
import com.bxm.warcar.mq.rocketmq.RocketmqConsumer;
import com.bxm.warcar.utils.MapHelper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;

import java.util.Map;

/**
 * @author allen
 * @date 2021-09-30
 * @since 1.0
 */
@Slf4j
public class LogSubscriberFactory implements BeanPostProcessor, DisposableBean, ApplicationListener<ApplicationStartedEvent> {

    /**
     * Map<{production}_{subscriberId}, Map<{mt}, LogSubscriber>>
     */
    private final Map<String, Map<String, LogSubscriber>> subscribers = Maps.newHashMap();
    private final Map<String, Consumer> consumerMap = Maps.newHashMap();
    private final RocketmqProperties rocketmqProperties;
    private final Serialization serialization;

    public LogSubscriberFactory(RocketmqProperties rocketmqProperties, Serialization serialization) {
        this.rocketmqProperties = rocketmqProperties;
        this.serialization = serialization;
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        for (Consumer consumer : consumerMap.values()) {
            consumer.start();

            if (log.isInfoEnabled()) {
                Listener listener = consumer.getMessageListener();
                log.info("Consumer topic:{} id:{} instance:{} has been started!", listener.getTopic(), listener.getConsumerId(), consumer);
            }
        }
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (!(bean instanceof LogSubscriber)) {
            return bean;
        }

        LogSubscriber logSubscriber = (LogSubscriber) bean;
        Production production = logSubscriber.getProduction();
        String id = logSubscriber.getId();
        String[] mts = logSubscriber.getMts();

        Preconditions.checkNotNull(production, bean + " production cannot be null!");
        Preconditions.checkNotNull(mts, bean + " mts cannot be blank!");
        Preconditions.checkArgument(StringUtils.isNotBlank(id), bean + " id cannot be blank!");

        String subscriberMapKey = production + "_" + id;
        String topic = Topic.dispatchMtLog(production.getName());
        String consumerId = ("CID_" + id + "_" + topic).toUpperCase();

        for (String mt : mts) {
            MapHelper.get(subscribers, subscriberMapKey, Maps.newHashMap()).put(mt, logSubscriber);
            if (log.isInfoEnabled()) {
                log.info("Registered subscriber [{}] for production: {} mt: {} on id: {}", subscriberMapKey, production, mt, id);
            }
        }
        if (consumerMap.containsKey(consumerId)) {
            return bean;
        }

        SingleMessageListener listener = new SubscriberMessageListener(id, topic, consumerId, serialization, subscribers.get(subscriberMapKey));
        Consumer consumer = createConsumer(listener);
        consumerMap.put(consumerId, consumer);

        if (log.isInfoEnabled()) {
            log.info("Registered consumer {} for topic {}", consumerId, topic);
        }

        return bean;
    }

    @Override
    public void destroy() {
        for (Consumer consumer : consumerMap.values()) {
            consumer.shutdown();
        }
    }

    private Consumer createConsumer(SingleMessageListener listener) {
        String nameSrvAddr = rocketmqProperties.getNameSrvAddr();
        return new RocketmqConsumer(nameSrvAddr, listener, listener.getConsumerId(), rocketmqProperties.getBrokerAddrAliasMap());
    }
}
