package com.bxm.adsmanager.web.controller.media;

import com.bxm.adsmanager.model.dao.common.Dictionaries;
import com.bxm.adsmanager.model.dao.media.AdMedia;
import com.bxm.adsmanager.model.dao.media.AdMediaPosition;
import com.bxm.adsmanager.model.vo.media.AdMediaTagVo;
import com.bxm.adsmanager.service.common.DictionariesService;
import com.bxm.adsmanager.service.media.MediaService;
import com.bxm.adsmanager.web.annotation.before.LogBefore;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.StringUtil;
import com.bxm.util.dto.ResultModel;
import com.github.pagehelper.PageInfo;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 媒体标签控制层
 *
 * @author 阿导
 * @version 1.0
 * @fileName com.bxm.adsmanager.web.controller.media.MediaController.java
 * @CopyRright (c) 2018-万物皆导
 * @created 2018-04-08 15:04:00
 */
@RestController
public class MediaController extends BaseController {
    /**
     * 日志记录
     */
    private static final Logger logger = Logger.getLogger(MediaController.class);

    /**
     * 业务层自动注入
     */
    @Autowired
    private MediaService mediaService;

    /**
     * 字典数据访问层注入
     */
    @Autowired
    private DictionariesService dictionariesService;

    /**
     * 获取媒体信息
     *
     * @param keyword
     * @param mjcode
     * @param bdcode
     * @param pageNum
     * @param pageSize
     * @return com.bxm.util.dto.ResultModel<com.github.pagehelper.PageInfo       <       com.bxm.adsmanager.model.dao.media.AdMedia>>
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/media", produces = "application/json")
    public ResultModel<PageInfo<AdMedia>> queryMedia(@RequestParam(value = "keyword", required = false) String keyword,
                                                     @RequestParam(value = "mjcode", required = false) String mjcode,
                                                     @RequestParam(value = "bdcode", required = false) String bdcode,
                                                     @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                     @RequestParam(value = "pageSize", required = false, defaultValue = "20") Integer pageSize) {
        //声明结果
        ResultModel<PageInfo<AdMedia>> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(mediaService.queryMedia(keyword, mjcode, bdcode, pageNum, pageSize));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 获取广告位信息
     *
     * @param appkey
     * @param pageNum
     * @param pageSize
     * @return com.bxm.util.dto.ResultModel<com.github.pagehelper.PageInfo       <       com.bxm.adsmanager.model.dao.media.AdMediaPosition>>
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/media/position", produces = "application/json")
    public ResultModel<PageInfo<AdMediaPosition>> queryMediaPosition(@RequestParam(value = "appkey") String appkey,
                                                                     @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                     @RequestParam(value = "pageSize", required = false, defaultValue = "20") Integer pageSize) {
        //声明结果
        ResultModel<PageInfo<AdMediaPosition>> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(mediaService.queryMediaPosition(appkey, pageNum, pageSize));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }


    /**
     * 查询标签信息
     *
     * @param appkey
     * @param business
     * @return com.bxm.util.dto.ResultModel<java.util.List       <       com.bxm.adsmanager.model.dao.media.AdMediaTag>>
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/media/position/tag", produces = "application/json")
    public ResultModel<List<AdMediaTagVo>> queryTagByPositionId(@RequestParam(value = "appkey") String appkey,
                                                                @RequestParam(value = "business") String business) {
        //声明结果
        ResultModel<List<AdMediaTagVo>> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(mediaService.queryTagByPositionId(appkey, business));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 获取字典
     *
     * @param code
     * @return com.bxm.util.dto.ResultModel<java.util.List       <       com.bxm.adsmanager.model.dao.common.Dictionaries>>
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/media/code", produces = "application/json")
    public ResultModel<List<Dictionaries>> queryDictionaries(@RequestParam(value = "code") String code) {
        //声明结果
        ResultModel<List<Dictionaries>> rs = new ResultModel<>();

        try {
            //调用服务层
            //获取 标签 字典信息
            rs.setReturnValue(dictionariesService.findAll(code));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }


    /**
     * 删除标签
     *
     * @param appkey
     * @param businessArray
     * @param tagArray
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     */
    @LogBefore(operType="/media/tag/delete",keyName="删除标签")
    @PostMapping(value = "/media/tag/delete", produces = "application/json")
    public ResultModel<Boolean> deleteTag(HttpServletRequest request, HttpServletResponse response,@RequestParam(value = "appkey") String appkey,
                                          @RequestParam(value = "businessArray") String businessArray,
                                          @RequestParam(value = "tagArray") String tagArray) {
        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();
        if (StringUtil.isEmpty(businessArray) || StringUtil.isEmpty(tagArray)) {
            rs.setReturnValue(false);
            rs.setSuccessed(false);
            rs.setErrorDesc("参数不正确");
        } else {
            try {
                //调用服务层
                mediaService.deleteTag(appkey, businessArray, tagArray);
                rs.setReturnValue(true);
                rs.setSuccessed(true);
            } catch (Exception e) {
                //异常处理
                logger.error(e.getMessage(), e);
                rs.setReturnValue(false);
                rs.setSuccessed(false);
                rs.setErrorDesc("system error");
            }
        }
        //返回
        return rs;
    }


    /**
     * 批量新增标签地域时间段黑名单
     *
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     * @param appkey
     * @param businessArray
     * @param tagArray
     * @param regionArray
     * @param timeArray
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     */
    @LogBefore(operType="/media/tag/add",keyName="新增标签")
    @PostMapping(value = "/media/tag/add", produces = "application/json")
    public ResultModel<Boolean> batchAddTag(HttpServletRequest request, HttpServletResponse response,@RequestParam(value = "appkey") String appkey,
                                            @RequestParam(value = "businessArray") String businessArray,
                                            @RequestParam(value = "tagArray") String tagArray,
                                            @RequestParam(value = "regionArray",required = false) String regionArray,
                                            @RequestParam(value = "timeArray",required = false) String timeArray){

        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();

        try {
            //调用服务层
            mediaService.bathInsertTag(appkey, businessArray, tagArray,regionArray,timeArray);
            rs.setReturnValue(true);
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setReturnValue(false);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }

        //返回
        return rs;
    }

    /**
     * 更新标签
     *
     * @author 阿导
     * @time 2018/4/9
     * @CopyRight 杭州微财科技有限公司
     * @param appkey
     * @param business
     * @param oldTag
     * @param newTag
     * @param regionArray
     * @param timeArray
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     */
    @LogBefore(operType="/media/tag/update",keyName="更新标签")
    @PostMapping(value = "/media/tag/update", produces = "application/json")
    public ResultModel<Boolean> updateTag(HttpServletRequest request, HttpServletResponse response,@RequestParam(value = "appkey") String appkey,
                                          @RequestParam(value = "business") String business,
                                          @RequestParam(value = "oldTag") String oldTag,
                                          @RequestParam(value = "newTag") String newTag,
                                          @RequestParam(value = "regionArray",required = false) String regionArray,
                                          @RequestParam(value = "timeArray",required = false) String timeArray){

        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();

            try {
                //调用服务层
                mediaService.updateTag(appkey, business, oldTag,newTag,regionArray,timeArray);
                rs.setReturnValue(true);
                rs.setSuccessed(true);
            } catch (Exception e) {
                //异常处理
                logger.error(e.getMessage(), e);
                rs.setReturnValue(false);
                rs.setSuccessed(false);
                rs.setErrorDesc("system error");
            }

        //返回
        return rs;
    }

    /**
     * 老数据同步到新数据
     *
     * @author 阿导
     * @time 2018/4/11
     * @CopyRight 杭州微财科技有限公司
     * @param
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     */
    @LogBefore(operType="/media/tag/oldtonew",keyName="同步老系统标签限制")
    @GetMapping(value = "/media/tag/oldtonew", produces = "application/json")
    public ResultModel<Boolean> oldToNew(HttpServletRequest request, HttpServletResponse response){
        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();

        try {
            //调用服务层
            mediaService.synchronizationTags();
            rs.setReturnValue(true);
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            logger.error(e.getMessage(), e);
            rs.setReturnValue(false);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }

        //返回
        return rs;
    }


}
