package com.bxm.adsmanager.web.controller.adkeeper;

import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONArray;
import com.bxm.adsmanager.model.dao.adkeeper.AdAssets;
import com.bxm.adsmanager.model.dao.adkeeper.AdAssetsCoupons;
import com.bxm.adsmanager.service.adkeeper.AdTicketAssetsCouponsService;
import com.bxm.adsmanager.service.adkeeper.AdTicketService;
import com.bxm.adsmanager.utils.BeanValidator;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;

/**
 * 广告优惠券素材
 * Created by zhengwangeng on 2019/5/22 14:01.
 */
@RestController
@RequestMapping("/adAssetsCoupons")
public class AdAssetsCouponsController extends BaseController {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdAssetsCouponsController.class);

    @Autowired
    private AdTicketAssetsCouponsService adTicketAssetsCouponsService;
    @Autowired
    private AdTicketService adTicketService;

    /**
     * 添加广告素材,票券还没有保存的时候
     *
     * @param request
     * @param response
     * @param adAssetsCouponsListConf json数组对象
     * @return
     */
    @LogBefore(operType = "/adAssetsCoupons/add", keyName = "广告")
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ResultModel addAssets(HttpServletRequest request, HttpServletResponse response, String
            adAssetsCouponsListConf) {
        ResultModel rs = new ResultModel();
        try {
            List<AdAssetsCoupons> list = JSONArray.parseArray(adAssetsCouponsListConf, AdAssetsCoupons.class);
            if (list.isEmpty()) {
                return ResultModelFactory.FAIL400("参数不可以为空");
            }
            for (AdAssetsCoupons adAssetsCoupons : list) {
                rs = checkParam(rs, adAssetsCoupons);
                if (!rs.isSuccessed()) {
                    return rs;
                }
                Long ticketId = -1L;
                adAssetsCoupons.setTicketId(ticketId);
                adAssetsCoupons.setCreateUser(getUser(request, response).getUsername());
                adAssetsCoupons.setCreateTime(new Date());
            }
            adTicketAssetsCouponsService.add(list);
            Long[] ids = list.stream().map(u -> u.getId()).toArray(Long[]::new);
            rs.setReturnValue(ids);
        } catch (Exception e) {
            LOGGER.error("添加广告素材出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("添加广告素材出错");
        }
        return rs;
    }

    /**
     * 添加广告素材,票券已经保存
     *
     * @param request
     * @param response
     * @param adAssetsCouponsListConf json数组对象
     * @param ticketId
     * @return
     */
    @LogBefore(operType = "/adAssetsCoupons/addNew", keyName = "广告")
    @RequestMapping(value = "/addNew", method = RequestMethod.POST)
    public ResultModel addNew(HttpServletRequest request, HttpServletResponse response, String
			adAssetsCouponsListConf, Long ticketId) {
        ResultModel rs = new ResultModel();
        try {
            List<AdAssetsCoupons> list = JSONArray.parseArray(adAssetsCouponsListConf, AdAssetsCoupons.class);
            if (list.isEmpty()) {
                return ResultModelFactory.FAIL400("参数不可以为空");
            }
            for (AdAssetsCoupons adAssetsCoupons : list) {
                rs = checkParam(rs, adAssetsCoupons);
                if (!rs.isSuccessed()) {
                    return rs;
                }
                adAssetsCoupons.setTicketId(ticketId);
                adAssetsCoupons.setCreateUser(getUser(request, response).getUsername());
                adAssetsCoupons.setCreateTime(new Date());
            }
            adTicketAssetsCouponsService.addNew(list, ticketId);
            Long[] ids = list.stream().map(u -> u.getId()).toArray(Long[]::new);
            rs.setReturnValue(ids);
        } catch (Exception e) {
            LOGGER.error("添加广告素材出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("添加广告素材出错");
        }
        return rs;
    }

    /**
     * 根据素材ids查找所有广告素材,这里没有做分页处理
     *
     * @param ids
     * @return
     */
    @RequestMapping(value = "/findAll", method = RequestMethod.GET)
    public ResultModel findAllAssets(String ids) {
        ResultModel rs = new ResultModel();
        try {
            List<AdAssetsCoupons> list = adTicketAssetsCouponsService.findByIds(ids);
            if (null == list) {
                rs.setSuccessed(false);
                rs.setErrorCode("5031");
                rs.setErrorDesc("获取数据为空");
            }
            rs.setReturnValue(list);
        } catch (Exception e) {
            LOGGER.error("根据素材ids查找所有广告素材出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("根据素材ids查找所有广告素材出错");
        }
        return rs;
    }

    /**
     * 根据广告券id查找广告素材,这里没有做分页处理
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/findAllAssetsById", method = RequestMethod.GET)
    public ResultModel findAllAssetsById(Long id) {
        ResultModel rs = new ResultModel();
        if (id == null) {
            return ResultModelFactory.FAIL400("id不能为空");
        }

        try {
            List<AdAssetsCoupons> list = adTicketAssetsCouponsService.findByTicketId(id);
            if (list != null) {
                rs.setSuccessed(true);
            }
            rs.setReturnValue(list);
        } catch (Exception e) {
            LOGGER.error("根据广告券id查找广告素材出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("根据广告券id查找广告素材出错");
        }
        return rs;
    }

    /**
     * @return java.lang.Object
     * @description 更新广告素材
     * @author hcmony
     * @time 2017/12/8
     * @CopyRight 杭州微财网络科技有限公司
     */
    @LogBefore(operType = "/adAssetsCoupons/update", keyName = "广告")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResultModel updateAllAssets(HttpServletRequest request, HttpServletResponse response, AdAssetsCoupons
			adAssetsCoupons, Long ticketId) {
        ResultModel rs = new ResultModel();
        Object result = null;
        try {
            if (adAssetsCoupons == null || adAssetsCoupons.getId() == null) {
                return ResultModelFactory.FAIL400("id不能为空");
            }

            rs = checkParam(rs, adAssetsCoupons);
            if (!rs.isSuccessed()) {
                return rs;
            }

            adAssetsCoupons.setModifyUser(getUser(request, response).getUsername());
            //增加是否需要审核的判断
//            String checkFlag = adTicketService.getIsCheck();
//            if (StringUtils.isNotBlank(checkFlag) && checkFlag.length() != 9 && "1".equals(checkFlag)) {
            if (adTicketService.checkTagNeedAudit(ticketId.intValue())){
                //调用审核的修改券素材的接口
                result = adTicketAssetsCouponsService.updateAndAudit(adAssetsCoupons, getUser(request, response));
            } else {
                result = adTicketAssetsCouponsService.update(adAssetsCoupons);
            }
            if (result != null) {
                rs.setSuccessed(true);
            }
        } catch (Exception e) {
            LOGGER.error("更新广告素材出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("更新广告素材出错");
        }
        rs.setReturnValue(result);
        return rs;
    }

    /**
     * 素材参数校验
     *
     * @param rs
     * @param adAssetsCoupons
     * @param <T>
     * @return
     */
    public <T> ResultModel checkParam(ResultModel<T> rs, AdAssetsCoupons adAssetsCoupons) {
        //参数校验
        if (BeanValidator.validateGroup(adAssetsCoupons, AdAssets.Add.class) != null) {
            return ResultModelFactory.FAIL400(BeanValidator.validateGroup(adAssetsCoupons, AdAssetsCoupons.Add.class).getMessage());
        }
        if (adAssetsCoupons.getType() == 2) {
            if (StringUtils.isBlank(adAssetsCoupons.getImage2())) {
                return ResultModelFactory.FAIL400("广告位展示素材图不能为空");
            }
        }

        return rs;
    }

}
