package com.bxm.adsmanager.web.controller.adkeeper;

import com.alibaba.fastjson.JSON;
import com.bxm.acl.facade.AclAuthInterceptor;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.adsmanager.integration.exception.PushProdException;
import com.bxm.adsmanager.model.dao.adkeeper.AdTicket;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.dto.mobile.TicketQueryConditionDTO;
import com.bxm.adsmanager.model.dto.mobile.TicketUpdateDTO;
import com.bxm.adsmanager.model.enums.TicketStatusEnum;
import com.bxm.adsmanager.model.vo.mobile.TicketDetailVO;
import com.bxm.adsmanager.service.adkeeper.mobile.MobileAdTicketService;
import com.bxm.adsmanager.utils.BeanValidator;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.util.dto.ValidateException;
import com.bxm.warcar.aspect.before.LogBefore;
import com.github.pagehelper.PageInfo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.Objects;

/**
 * 文 件 名 :
 * CopyRright (c) 2008-xxxx:
 * 文件编号：
 * 创 建 人： hxpeng
 * 日 期： 2018/7/31 14:57
 * 修 改 人：
 * 日 期：
 * 描 述： 移动端 广告券管理
 * 版 本 号： v1
 */
@Slf4j
@RestController
@RequestMapping(value = "/mobile/ticket")
public class MobileAdTicketController extends BaseController {

    @Autowired
    private MobileAdTicketService mobileAdTicketService;

    /**
     * Description: 广告券列表  查询分页
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/8/2 9:52
     */
    @RequestMapping(value = "/findAll", method = RequestMethod.GET)
    public ResultModel<PageInfo> findAll(TicketQueryConditionDTO conditionDTO, HttpServletResponse response, HttpServletRequest request) {
        ResultModel<PageInfo> resultModel = new ResultModel<>();
        try {
            UserVo userVo = (UserVo) request.getAttribute(AclAuthInterceptor.USER_KEY);
            if (userVo == null || StringUtils.isBlank(userVo.getUsername())) {
                return ResultModelFactory.FAIL500("未登录");
            }
            resultModel.setReturnValue(mobileAdTicketService.findAll(conditionDTO, (UserVo) request.getAttribute(AclAuthInterceptor.USER_KEY)));
        } catch (Exception e) {
            log.error("查询所有券异常！", e);
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return resultModel;
    }

    /**
     * Description: 广告券详情
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/8/2 9:52
     */
    @RequestMapping(value = "/detail", method = RequestMethod.GET)
    public ResultModel<TicketDetailVO> detail(@RequestParam(name = "id") Long id) {
        ResultModel<TicketDetailVO> resultModel = new ResultModel<>();
        try {
            resultModel.setReturnValue(mobileAdTicketService.detail(id));
        } catch (Exception e) {
            log.error("查询券详情异常！", e);
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return resultModel;
    }

    /**
     * Description: 广告券 价格修改
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/8/2 9:52
     */
    @LogBefore(operType = "/mobile/ticket/update", keyName = "广告")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResultModel<Boolean> update(HttpServletRequest request, HttpServletResponse response, TicketUpdateDTO updateDTO) {
        response.setHeader("Access-Control-Allow-Origin", "*");
        ResultModel<Boolean> resultModel = new ResultModel<>();
        try {
            //参数校验
            if (BeanValidator.validateGroup(updateDTO) != null) {
                return ResultModelFactory.FAIL400(Objects.requireNonNull(BeanValidator.validateGroup(updateDTO)).getMessage());
            }
            UserVo userVo = (UserVo) request.getAttribute(AclAuthInterceptor.USER_KEY);
            if (userVo == null || StringUtils.isBlank(userVo.getUsername())) {
                return ResultModelFactory.FAIL500("未登录");
            }
            resultModel.setReturnValue(mobileAdTicketService.update(updateDTO, userVo));
            //日志记录
            addLogs(getUser(request, response), "[移动端]编辑：" + JSON.toJSONString(updateDTO), updateDTO.getId() + "");
            //新的日志记录-记录券价格变化
            addMobileTicketNewLogs(getUser(request, response), updateDTO);
            return resultModel;
        } catch (Exception e) {
            log.error("更新广告券出错", e);
            return ResultModelFactory.FAIL500("更新广告券出错：" + e.getMessage());
        }
    }

    //新的日志记录-记录券价格变化
    private void addMobileTicketNewLogs(User user, TicketUpdateDTO updateDTO) {
        String log = "";
        try {
            if (updateDTO == null) {
                return;
            }
            //出价判断
            BigDecimal oldPrice = updateDTO.getOldPrice();
            BigDecimal price = updateDTO.getPrice();
            if (oldPrice != null && price != null) {
                String oldPriceStr = String.valueOf(oldPrice);
                String priceStr = String.valueOf(price);
                if (!oldPriceStr.equals(priceStr)) {
                    log += "出价从" + oldPriceStr + "调整为" + priceStr + ";";
                }
            }
            //入MongoDB
            if (StringUtils.isNotBlank(log)) {
                addNewLogs(user, "[移动端]编辑：" + log, String.valueOf(updateDTO.getId()));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Description: 广告券关闭/开启
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/8/2 9:52
     */
    @LogBefore(operType = "/mobile/ticket/updateStatus", keyName = "广告")
    @RequestMapping(value = "/updateStatus", method = RequestMethod.POST)
    public ResultModel<Boolean> updateStatus(HttpServletRequest request, HttpServletResponse response,
                                             @RequestParam(name = "id") Long id,
                                             @RequestParam(name = "statusFlag") Integer statusFlag) {
        if (0 != statusFlag && 1 != statusFlag) {
            return ResultModelFactory.FAIL500("请传入正确的参数！！！");
        }
        response.setHeader("Access-Control-Allow-Origin", "*");
        ResultModel<Boolean> rs = new ResultModel<>();
        try {
            UserVo userVo = (UserVo) request.getAttribute(AclAuthInterceptor.USER_KEY);
            if (userVo == null || StringUtils.isBlank(userVo.getUsername())) {
                return ResultModelFactory.FAIL500("未登录");
            }
            Short status = 1 == statusFlag ? TicketStatusEnum.NORMAL.getStatus().shortValue() : TicketStatusEnum.STOP.getStatus().shortValue();
            rs.setReturnValue(mobileAdTicketService.updateStatus(id, status, userVo));
            //日志记录
            addLogs(getUser(request, response), JSON.toJSONString(AdTicket.AdTicketStatus.getName(status) + "广告券"), id + "");
            return rs;
        } catch (ValidateException v) {
            return ResultModelFactory.FAIL400(v.getMessage());
        } catch (PushProdException e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        } catch (Exception e) {
            log.error("更新广告券状态出错" + e.getMessage(), e);
            return ResultModelFactory.FAIL500("更新广告券状态出错：" + e.getMessage());
        }
    }

}
