package com.bxm.adsmanager.web.controller.adticketgroup;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.repository.query.Param;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSON;
import com.bxm.adsmanager.model.dao.common.Dictionaries;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.model.enums.TicketGroupFrequencyEnum;
import com.bxm.adsmanager.model.vo.adticketgroup.TicketGroupVo;
import com.bxm.adsmanager.model.vo.adticketgroup.TicketVo;
import com.bxm.adsmanager.service.adticketgroup.TicketGroupService;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.StringUtil;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;
import com.github.pagehelper.PageInfo;

/**
 * 广告组控制层
 *
 * @author 阿导
 * @version 1.0
 * @fileName com.bxm.adsmanager.web.controller.adticketgroup.TicketGroupController.java
 * @CopyRright (c) 2018-杭州微财科技有限公司
 * @created 2018-03-23 16:43:00
 */

@RestController
public class TicketGroupController extends BaseController {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketGroupController.class);

    /**
     * 服务层 DI
     */
    @Autowired
    TicketGroupService ticketGroupService;

    /**
     * 根据关键字查询广告组
     *
     * @param keyword  关键字
     * @param pageNum  当前页码
     * @param pageSize 当前页面大小
     * @param sortName 排序字段名称
     * @param sortType 排序方式
     * @return com.bxm.util.dto.ResultModel<com.github.pagehelper.PageInfo < com.bxm.adsmanager.model.vo.adticketgroup.TicketGroupVo>>
     * @author 阿导
     * @time 2018/3/22
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/ticketgroup", produces = "application/json")
    public ResultModel<PageInfo<TicketGroupVo>> findTicketGroup(@RequestParam(value = "keyword", required = false) String keyword,
                                                                @RequestParam(value = "status", required = false) Integer status,
                                                                @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                @RequestParam(value = "pageSize", required = false, defaultValue = "20") Integer pageSize,
                                                                @RequestParam(value = "sortName", required = false, defaultValue = "modifyTime") String sortName,
                                                                @RequestParam(value = "sortType", required = false, defaultValue = "desc") String sortType) {
        //声明结果
        ResultModel<PageInfo<TicketGroupVo>> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(ticketGroupService.findTicketGroup(keyword, status, pageNum, pageSize, sortName,
                    sortType));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;

    }


    /**
     * 根据 ID 查询广告组信息
     *
     * @param id 广告组 ID
     * @return com.bxm.util.dto.ResultModel<com.bxm.adsmanager.model.vo.adticketgroup.TicketGroupVo>
     * @author 阿导
     * @time 2018/3/22
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/ticketgroup/get", produces = "application/json")
    public ResultModel<TicketGroupVo> findTicketGroupById(@Param("id") Long id) {
        //声明结果
        ResultModel<TicketGroupVo> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(ticketGroupService.findTicketGroupById(id));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 查询广告信息
     *
     * @param groupId  广告组
     * @param keyword  关键字
     * @param pageNum  当前页码
     * @param pageSize 当前页面大小
     * @return com.bxm.util.dto.ResultModel<com.github.pagehelper.PageInfo < com.bxm.adsmanager.model.vo.adticketgroup.TicketVo>>
     * @author 阿导
     * @time 2018/3/23
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/ticketgroup/ticket", produces = "application/json")
    public ResultModel<PageInfo<TicketVo>> findTicket(@RequestParam(value = "groupId", required = false) Long groupId,
                                                      @RequestParam(value = "keyword", required = false) String keyword,
                                                      @RequestParam(value = "ticketIdArray", required = false) String ticketIdArray,
                                                      @RequestParam(value = "type", defaultValue = "link") String type,
                                                      @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                      @RequestParam(value = "pageSize", required = false, defaultValue = "20") Integer pageSize) {
        //声明结果
        ResultModel<PageInfo<TicketVo>> rs = new ResultModel<>();
        try {
            //声明广告组关联的广告信息
            List<Long> ticketIds = new ArrayList<>();
            if (StringUtil.isNotEmpty(ticketIdArray)) {
                //强转
                ticketIds = JSON.parseArray(ticketIdArray, Long.class);
            }
            //若是查询的时候没有传入广告ID或者广告组ID
            if ("link".equalsIgnoreCase(type) && ticketIds.isEmpty()) {
                PageInfo<TicketVo> ticketVoPageInfo = new PageInfo<>();
                ticketVoPageInfo.setPageSize(pageSize);
                ticketVoPageInfo.setPageNum(pageNum);
                rs.setReturnValue(ticketVoPageInfo);
            } else {
                //调用服务层
                rs.setReturnValue(ticketGroupService.findTicket(groupId, keyword, ticketIds, type, pageNum, pageSize));
            }
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 新增或者更新广告组
     *
     * @param ticketGroupvo 广告组信息
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     * @author 阿导
     * @time 2018/3/22
     * @CopyRight 杭州微财科技有限公司
     */
    @LogBefore(operType = "/ticketgroup", keyName = "新增或者更新广告组")
    @PostMapping(value = "/ticketgroup", produces = "application/json")
    public ResultModel<Boolean> saveOrUpdateTicketGroup(HttpServletRequest request, HttpServletResponse response,
                                                        TicketGroupVo ticketGroupvo, @RequestParam(value = "ticketIdArray", required = false) String ticketIdArray) {
        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();
        try {
            //声明广告组关联的广告信息
            List<Long> ticketIds = new ArrayList<>();
            if (StringUtil.isNotEmpty(ticketIdArray)) {
                //强转
                ticketIds = JSON.parseArray(ticketIdArray, Long.class);
            }
            //调用服务层
            ticketGroupService.saveOrUpdateTicketGroup(ticketGroupvo, ticketIds, getUserName(request, response));
            rs.setReturnValue(true);
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 逻辑删除广告组
     *
     * @param id 广告组 ID
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     * @author 阿导
     * @time 2018/3/22
     * @CopyRight 杭州微财科技有限公司
     */
    @LogBefore(operType = "/ticketgroup/delete", keyName = "逻辑删除广告组")
    @PostMapping(value = "/ticketgroup/delete", produces = "application/json")
    public ResultModel<Boolean> deleteTicketGroup(HttpServletRequest request, HttpServletResponse response,
                                                  @RequestParam(value = "id") Long id) {
        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();
        try {
            //调用服务层
            boolean b = ticketGroupService.deleteTicketGroup(id, getUserName(request, response));
            rs.setReturnValue(b);
            if (!b) {
                rs.setErrorDesc("开启状态的广告组不能删除");
            }
            rs.setSuccessed(b);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 修改广告组的状态
     *
     * @param id     广告组 ID
     * @param status 修改后的状态
     * @return com.bxm.util.dto.ResultModel<java.lang.Boolean>
     * @author 阿导
     * @time 2018/3/23
     * @CopyRight 杭州微财科技有限公司
     */
    @LogBefore(operType = "/ticketgroup/changestatus", keyName = "修改广告组的状态")
    @PostMapping(value = "/ticketgroup/changestatus", produces = "application/json")
    public ResultModel<Boolean> changeTicketStatus(HttpServletRequest request, HttpServletResponse response,
                                                   @RequestParam(value = "id") Long id, @RequestParam(value = "status"
    ) Integer status) {
        //声明结果
        ResultModel<Boolean> rs = new ResultModel<>();
        try {
            //调用服务层
            ticketGroupService.changeTicketStatus(id, status, getUserName(request, response));
            rs.setReturnValue(true);
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 广告组频次限制字典
     *
     * @param
     * @return com.bxm.util.dto.ResultModel<java.util.Map < java.lang.Integer, java.lang.String>>
     * @author 阿导
     * @time 2018/3/24
     * @CopyRight 杭州微财科技有限公司
     */
    @GetMapping(value = "/ticketgroup/frequencycode", produces = "application/json")
    public ResultModel<List<Dictionaries>> getTicketGroupFrequencyCode() {
        //声明结果
        ResultModel<List<Dictionaries>> rs = new ResultModel<>();
        try {
            //获取字典并相关处理
            Map<Integer, String> allState = TicketGroupFrequencyEnum.getAllState();
            List<Dictionaries> list = new ArrayList<>();
            if (!allState.isEmpty()) {
                allState.forEach((k, v) -> {
                    Dictionaries dictionaries = new Dictionaries();
                    dictionaries.setTypecode(k.toString());
                    dictionaries.setTypename(v);
                    list.add(dictionaries);
                });
            }
            rs.setReturnValue(list);
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

    /**
     * 获取操作者的用户名
     *
     * @param request
     * @param response
     * @return java.lang.String
     * @author 阿导
     * @time 2018/3/23
     * @CopyRight 杭州微财科技有限公司
     */
    private String getUserName(HttpServletRequest request, HttpServletResponse response) {
        User user = getUser(request, response);
        if (user == null) {
            return "用户不明";
        }
        return user.getUsername();
    }

    /**
     * 分页
     *
     * @param groupIds
     * @param pageNum
     * @param pageSize
     * @return
     */
    @GetMapping(value = "/ticketgroup/findByIds", produces = "application/json")
    public ResultModel<PageInfo<TicketGroupVo>> findTicketGroup(@RequestParam(value = "groupIds", required = false) String groupIds,
                                                                @RequestParam(value = "keywords", required = false) String keywords,
                                                                @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                @RequestParam(value = "pageSize", required = false, defaultValue = "20") Integer pageSize) {
        //声明结果
        ResultModel<PageInfo<TicketGroupVo>> rs = new ResultModel<>();
        try {
            //调用服务层
            rs.setReturnValue(ticketGroupService.findByIds(groupIds, pageNum, pageSize,keywords));
            rs.setSuccessed(true);
        } catch (Exception e) {
            //异常处理
            LOGGER.error(e.getMessage(), e);
            rs.setSuccessed(false);
            rs.setErrorDesc("system error");
        }
        //返回
        return rs;
    }

}
