package com.bxm.adsmanager.web.controller.advertiser;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.bxm.adsmanager.dal.mapper.advertiser.ext.TblAdQualifyMapperExt;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserVo;
import com.bxm.adsmanager.model.dao.advertiser.TblAdQualify;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.adsmanager.dal.mapper.advertiser.AdvertiserBehaviorConfigMapper;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserDto;
import com.bxm.adsmanager.integration.advertiser.model.AdvertiserNewDto;
import com.bxm.adsmanager.integration.advertiser.model.Pagination;
import com.bxm.adsmanager.integration.advertiser.service.AdShopIntegration;
import com.bxm.adsmanager.model.dao.advertiser.AdvertiserBehaviorConfig;
import com.bxm.adsmanager.model.dao.user.User;
import com.bxm.adsmanager.service.diysite.SiteInstanceService;
import com.bxm.adsmanager.utils.ResultModelFactory;
import com.bxm.adsmanager.utils.UserRoleCodeUtil;
import com.bxm.adsmanager.web.controller.base.BaseController;
import com.bxm.util.dto.ResultModel;
import com.bxm.warcar.aspect.before.LogBefore;

import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 广告主管理
 */
@RestController
@RequestMapping("/advertiserManage")
@Slf4j
public class AdvertiserController extends BaseController {

    @Autowired
    private AdShopIntegration adShopIntegration;
    @Autowired
    private SiteInstanceService siteInstanceService;

    @Autowired
    private AdvertiserBehaviorConfigMapper advertiserBehaviorConfigMapper;
    @Autowired
    private TblAdQualifyMapperExt adQualifyMapperExt;

    /**
     * 获取广告主列表
     * @param request
     * @param response
     * @param advertiserNewDto
     * @return
     */
    @RequestMapping(value = "/queryAdvertiser", method = RequestMethod.GET)
    public ResultModel<Pagination> queryAdvertiser(HttpServletRequest request, HttpServletResponse response,
                                                   AdvertiserNewDto advertiserNewDto) {
        ResultModel<Pagination> rs = new ResultModel<>();
        try {
            User user = getUser(request, response);
            //登录用户是ae和传参ae不一致 不查
            if (StringUtils.isNotEmpty(UserRoleCodeUtil.isAe(user))) {
                if (StringUtils.isNotEmpty(advertiserNewDto.getAe())
                        && !UserRoleCodeUtil.isAe(user).equals(advertiserNewDto.getAe())) {
                    rs.setReturnValue(new Pagination());
                    return rs;
                }
                advertiserNewDto.setAe(user.getUsername());
            }
            //登录用户是sale和传参sale不一致时不查
            if (StringUtils.isNotEmpty(UserRoleCodeUtil.isSale(user))) {
                if (StringUtils.isNotEmpty(advertiserNewDto.getSale())
                        && !UserRoleCodeUtil.isSale(user).equals(advertiserNewDto.getSale())) {
                    rs.setReturnValue(new Pagination());
                    return rs;
                }
                advertiserNewDto.setSale(user.getUsername());
            }
            Pagination advertiserList = adShopIntegration.getAdvertiserList(advertiserNewDto);
            fillAudit(advertiserList);
            rs.setReturnValue(advertiserList);
        } catch (Exception e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return rs;
    }

    private void fillAudit(Pagination advertiserList) {
        List<AdvertiserVo> list = (List<AdvertiserVo>) advertiserList.getList();
        if (CollectionUtils.isEmpty(list)) {
            return;
        }

        // 填充审核信息 主体和资质
        List<Long> ids = list.stream().map(a -> a.getId() == null ? -99 : a.getId().longValue()).collect(Collectors.toList());
        List<TblAdQualify> qualifys = adQualifyMapperExt.getQualifyByAdvertiserIds(ids);
        for (AdvertiserVo advertiserVo : list) {
            StringBuffer qualifyAuditDescBuffer = new StringBuffer();

            // 处理资质的审核个数统计 按状态分组
            Long id = advertiserVo.getId().longValue();
            Map<Integer, Long> qualifyGroupByStatus = qualifys.stream().filter(q -> q.getAdvertiserId().equals(id)).collect(Collectors.groupingBy(TblAdQualify::getAuditStatus, Collectors.counting()));
            qualifyGroupByStatus.forEach((status, count) -> {
                switch (status) {
                    case 1:
                        qualifyAuditDescBuffer.append("审核通过：").append(count).append(" ；");
                        break;
                    case 2:
                        qualifyAuditDescBuffer.append("待审核：").append(count).append(" ；");
                        break;
                    case 3:
                        qualifyAuditDescBuffer.append("审核拒绝：").append(count).append(" ；");
                        break;
                }
            });
            advertiserVo.setQualifyStatus(qualifyAuditDescBuffer.toString());
        }
    }

    /**
     * @description 根据ID更新广告主信息
     * @CopyRight 杭州微财网络科技有限公司
     */
    @LogBefore(operType = "/advertiserManage/updateAdvertiser", keyName = "广告主")
    @RequestMapping(value = "/updateAdvertiser", method = RequestMethod.PUT)
    @Transactional
    public ResultModel<Integer> updateAdvertiser(HttpServletRequest request, HttpServletResponse response,
                                                 @RequestParam(required = true, name = "id") Integer id,
                                                 AdvertiserNewDto advertiserNewDto) {
        ResultModel<Integer> rs = new ResultModel<Integer>();
        try {
            advertiserNewDto.setId(id);
            boolean flag = adShopIntegration.updateAdShopMsg(advertiserNewDto);

            // 先删除后新增
            advertiserBehaviorConfigMapper.deleteByAdvertiserId(advertiserNewDto.getId());
            boolean needUpdateBehaviorConfig = StringUtils.isNotBlank(advertiserNewDto.getBehaviorConfig())
                    || advertiserNewDto.getDisplayValidClick() != null
                    || Objects.nonNull(advertiserNewDto.getCustomize());
            if (needUpdateBehaviorConfig) {
                AdvertiserBehaviorConfig insert = new AdvertiserBehaviorConfig();
                insert.setAdvertiserId(advertiserNewDto.getId());
                insert.setBehaviorNameConfig(advertiserNewDto.getBehaviorConfig());
                insert.setDisplayValidClick(advertiserNewDto.getDisplayValidClick());
                insert.setCustomize(advertiserNewDto.getCustomize());
                advertiserBehaviorConfigMapper.insertSelective(insert);
            }

            if (flag) {
                rs.setReturnValue(1);
                rs.setSuccessed(true);
            } else {
                rs.setReturnValue(0);
                rs.setSuccessed(false);
            }
        } catch (Exception e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return rs;
    }

    @LogBefore(operType = "/advertiserManage/updatePassword", keyName = "广告主")
    @RequestMapping(value = "/updatePassword", method = RequestMethod.PUT)
    public ResultModel updatePassword(AdvertiserNewDto advertiserNewDto, HttpServletRequest request,
                                      HttpServletResponse response) {
        ResultModel<Boolean> rs = new ResultModel<>();
        User user = getUser(request, response);
        Boolean isLeader = UserRoleCodeUtil.isLeader(user);
        if (!isLeader) {
            return ResultModelFactory.FAIL500("无修改权限");
        }
        AdvertiserNewDto advertiserNewDtoTemp = new AdvertiserNewDto();
        advertiserNewDtoTemp.setId(advertiserNewDto.getId());
        advertiserNewDtoTemp.setPassword(advertiserNewDto.getPassword());
        if (StringUtils.isEmpty(advertiserNewDto.getPassword())) {
            return ResultModelFactory.FAIL500("新密码不可为空");
        }
        try {
            AdvertiserDto advertiserDto = adShopIntegration.findAdShopMsg(advertiserNewDto.getId().longValue());
            advertiserNewDtoTemp.setCompany(advertiserDto.getCompany());
            boolean flag = adShopIntegration.updateAdShopMsg(advertiserNewDtoTemp);
            if (true == flag) {
                rs.setReturnValue(true);
                rs.setSuccessed(true);
            } else {
                rs.setReturnValue(false);
                rs.setSuccessed(false);
            }
        } catch (Exception e) {
            return ResultModelFactory.FAIL500(e.getMessage());
        }
        return rs;
    }

    /**
     * 广告主自建站配置
     */
    @RequestMapping(value = "/setOwnSite", method = RequestMethod.POST)
    public ResultModel setOwnSite(@RequestParam(name = "advertiserId") Integer advertiserId,
                                  @RequestParam(name = "ownSite") String ownSite) {
        ResultModel resultModel = new ResultModel();
        Boolean result = siteInstanceService.setOwnSite(advertiserId, ownSite);
        resultModel.setReturnValue(result);
        return resultModel;
    }

    @RequestMapping(value = "/getAdvertiserBehaviourConfig", method = RequestMethod.GET)
    public ResultModel getAdvertiserBehaviourConfig(@RequestParam("advertiserId") Integer advertiserId) {
        ResultModel<AdvertiserBehaviorConfig> rs = new ResultModel<>();
        AdvertiserBehaviorConfig advertiserBehaviorConfig = advertiserBehaviorConfigMapper.selectByAdvertiserId(advertiserId);
        if (null != advertiserBehaviorConfig) {
            rs.setReturnValue(advertiserBehaviorConfig);
        }
        return rs;
    }

}
